#ifndef __ATO_AKM_CR_H__
#define __ATO_AKM_CR_H__

/** @addtogroup atoakm_api_cr

@{
*/

/**
The credential object.

An credential has a limited lifetime defined by ato_cr_notbefore() and ato_cr_notafter().
*/
typedef struct _ato_Credential ato_Credential;

/**
Defines the type of credential.
*/
typedef enum {
    ATO_NONE, /**< Credential type is undefined */
    ATO_CREDUSER, /**< User Credential */
    ATO_CREDDEVICE /**< Device Credential */
} ato_eCredtype;

/**
Used to filter lists of ato_Credential in a keystore - see ato_ks_credentials().
*/
typedef enum {
    ATO_FILTER_ALL, /**< Include all credential records */
    ATO_FILTER_VALID, /**< Include credentials which are currently valid. See ato_cr_isvalid() */
    ATO_FILTER_USER, /**< Include user credentials only */
    ATO_FILTER_VALIDUSER, /**< Include user credentials which are valid */
    ATO_FILTER_DEVICE, /**< Include device credentials only */
    ATO_FILTER_VALIDDEVICE /**< Include device credentials which are valid */
} ato_eCredfilter;

#ifdef __cplusplus
extern "C" {
#endif

/**
Return the credential id. This corresponds to the credential "id" attributed in the store.
@param cr the credential object.
@return id.
*/
ATO_AKM_EXPORT const char *ato_cr_id(ato_Credential *cr);

/**
Same as ato_cr_id().
@param cr the credential object.
@return id.
*/
ATO_AKM_EXPORT const char *ato_cr_alias(ato_Credential *cr);

/**
Return the credential type.
@param cr the credential object.
@return credential type.
*/
ATO_AKM_EXPORT ato_eCredtype ato_cr_type(ato_Credential *cr);

/**
Return the credential ABN.
@param cr the credential object.
@return credential ABN.
*/
ATO_AKM_EXPORT const char *ato_cr_abn(ato_Credential *cr);

/**
Return the credential legal name.
@param cr the credential object.
@return credential legal name.
*/
ATO_AKM_EXPORT const char *ato_cr_legalname(ato_Credential *cr);

/**
Return the credential serial number.
@param cr the credential object.
@return credential serial number.
*/
ATO_AKM_EXPORT const char *ato_cr_serialnr(ato_Credential *cr);

/**
Return the credential creation time.
@param cr the credential object.
@return credential creation timestamp.
*/
ATO_AKM_EXPORT const char *ato_cr_creationdate(ato_Credential *cr);

/**
Returns the start of the credential validity period.
@param cr the credential object.
@return timestamp when credential is first valid.
*/
ATO_AKM_EXPORT const char *ato_cr_notbefore(ato_Credential *cr);

/**
Returns the end of the credential validity period.
@param cr the credential object.
@return timestamp when credential is last valid.
*/
ATO_AKM_EXPORT const char *ato_cr_notafter(ato_Credential *cr);

/**
Return the credential SHA1 fingerprint.
@param cr the credential object.
@return SHA1 fingerprint as base64.
*/
ATO_AKM_EXPORT const char *ato_cr_sha1fingerprint(ato_Credential *cr);

/**
Return the credential SHA1 fingerprint as a hex string.
@param ctx the current thread local Context
@param cr the credential object.
@return SHA1 fingerprint as hex.
*/
ATO_AKM_EXPORT const char *ato_cr_sha1fingerprint_hex(ato_Ctx *ctx, ato_Credential *cr);

/**
Return the credential person id or NULL if not a USER credential.
@param cr the credential object.
@return Family name.
*/
ATO_AKM_EXPORT const char *ato_cr_personid(ato_Credential *cr);

/**
Return the credential device name or NULL if not a DEVICE credential.
@param cr the credential object.
@return Family name.
*/
ATO_AKM_EXPORT const char *ato_cr_devicename(ato_Credential *cr);

/**
Return the credential given name(s) or NULL if not a USER credential.
@param cr the credential object.
@return Family name.
*/
ATO_AKM_EXPORT const char *ato_cr_givenames(ato_Credential *cr);

/**
Return the credential family name or NULL if not a USER credential.
@param cr the credential object.
@return Family name.
*/
ATO_AKM_EXPORT const char *ato_cr_familyname(ato_Credential *cr);

/**
Determines if the credential matches the given filter criteria.
@param cr the credential object.
@param filter the filter to check.
@return TRUE or FALSE.
*/
ATO_AKM_EXPORT bool ato_cr_matchesfilter(ato_Credential *cr, ato_eCredfilter filter);

/**
Determines if the credential is current (between NotBefore and NotAfter dates).

@param cr the credential object.
@return TRUE or FALSE.
*/
ATO_AKM_EXPORT bool ato_cr_iscurrent(ato_Credential *cr);

/*
@deprecated
Use ato_cr_iscurrent().

@param cr the credential object.
@return TRUE or FALSE.
*/
ATO_AKM_EXPORT bool ato_cr_isvalid(ato_Credential *cr);

/**
Check if the credential has been changed since the last load.
@param cr the credential object.
@return TRUE or FALSE.
*/
ATO_AKM_EXPORT bool ato_cr_ismodified(ato_Credential *cr);

/**
Determines if the credential is renewed, this relates only to the current session.

@param cr the credential object.
@return TRUE or FALSE.
*/
ATO_AKM_EXPORT bool ato_cr_isrenewed(ato_Credential *cr);

/**
Determines if the credential is ready to be renewed - see ato_cr_renew().

Prior to using a method that access a credential private key, ato_cr_renew() or ato_cr_isrenewable() MUST be called
at least once per session, for the given credential, otherwise the method in questions will return the error ATO_AKM_ERR_RENEWNOTCALLED.

To effectively disable renewal, simply call this method and not ato_cr_renew().

If the credential is due to expire within 14 months it is within the expiry period.
This expiry period can be set in the config file, although the minimum is 14 months.
A 25 month period for a 2 year credential will result in a new credential always being
renewed - useful for testing.

@param ctx the current thread local Context
@param cr the credential object.
@return TRUE if it is not currently renewed, is valid, and is within the expiry period.
*/
ATO_AKM_EXPORT bool ato_cr_isrenewable(ato_Ctx *ctx, ato_Credential *cr);

/**
Determines if the credential matches its expected integrity value.

This will always be true, unless the credential has changed or, possibly if the keystore was
loaded without integrity validation and there is a mismatch. Used for testing scenarious.

@param ctx the current thread local Context
@param cr the credential object.
@return TRUE if valid; FALSE otherwise or if an internal exception has occurred.
*/
ATO_AKM_EXPORT bool ato_is_integrity_valid(ato_Ctx *ctx, ato_Credential *cr);

/**
Test if the password is correct for the credential.

@param ctx the current thread local Context.
@param cr the credential object.
@param pwd the password

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_PWD if the password is wrong
For any other error, also check for the presence of a nested inner error.
*/
ATO_AKM_EXPORT int ato_cr_checkcorrectpwd(ato_Ctx *ctx, ato_Credential *cr, const char *pwd);

/**
@deprecated { Use ato_cr_checkcorrectpwd(). }

Determines if the credential password is valid.

Given that all credentials in a keystore must have the same password, this is really just
a convenience method on the credential.

@param ctx the current thread local Context
@param cr the credential object.
@param correct address of the bool/int to set as TRUE or FALSE
@param pwd the password to test
@return ATO_ERR_OK, else an error code if an internal exception has occured
*/
ATO_AKM_EXPORT int ato_cr_iscorrectpwd(ato_Ctx *ctx, ato_Credential *cr, bool *correct, const char *pwd);

/**
Determines if the credential P8 uses PBES1 or PBES2 scheme.

@param ctx the current thread local Context
@param cr the credential object.
@return TRUE if PBES2
*/
ATO_AKM_EXPORT bool ato_cr_ispbe2(ato_Ctx *ctx, ato_Credential *cr);

/**
Construct a X509 certificate object from the credential and return as a DER encoded array.
This includes the principle as well as the issuer certificates.

@param ctx the current thread local Context
@param cr the credential object.
@param certificate the address of an ato_String object to allocate. The caller must free this using ato_str_free().

@return ATO_ERR_OK(0) if successful, otherwise
- ?
*/
ATO_AKM_EXPORT int ato_cr_certificate(ato_Ctx *ctx, ato_Credential *cr, ato_String **certificate);

/**
Construct a P7 certificate chain object from the credential and return as a DER encoded array.
This includes the principle as well as the issuer certificates.

@param ctx the current thread local Context.
@param cr the credential object.
@param certificates the address of an ato_String object to allocate. The caller must free this using ato_str_free().

@return ATO_ERR_OK(0) if successful, otherwise
- ?
*/
ATO_AKM_EXPORT int ato_cr_chain(ato_Ctx *ctx, ato_Credential *cr, ato_String **certificates);

/**
Construct a decrypted private key object from the credential and return as a DER encoded array
- see also ato_cr_isrenewable() and ato_cr_renew().

@param ctx the current thread local Context.
@param cr the credential object.
@param privatekey the address of an ato_String object to allocate. The caller must free this using ato_str_free().
@param pwd the password used to access the private key.

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_BADPWD if the  password is wrong
- ATO_AKM_ERR_RENEWNOTCALLED if ato_cr_renew() or ato_cr_isrenewable() has not been called
*/
ATO_AKM_EXPORT int ato_cr_privatekey(ato_Ctx *ctx, ato_Credential *cr, ato_String **privatekey, const char *pwd);

/**
Construct a P12 object from the credential and return as a DER encoded array
- see also ato_cr_isrenewable() and ato_cr_renew().

@param ctx the current thread local Context.
@param cr the credential object.
@param p12 the address of an ato_String object to allocate. The caller must free this using ato_str_free().
@param pwd the password used to access the private key.

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_BADPWD if the  password is wrong
- ATO_AKM_ERR_RENEWNOTCALLED if ato_cr_renew() or ato_cr_isrenewable() has not been called
*/
ATO_AKM_EXPORT int ato_cr_p12(ato_Ctx *ctx, ato_Credential *cr, ato_String **p12, const char *pwd);

/*
Renew the credential - see ato_cr_isrenewable().

The renewal service url is defined in the configuration file.

Checks if the credential is renewable and if it is attempts renewal.
This is done synchronously - no separate thread is created.
This can be done on a seperate thread if desired.
The renewed credential is made available after a keystore save() operation.

@param ctx the current thread local Context.
@param cr the credential object.
@param pwd the password used to access the private key.

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_BADPWD if the  password is wrong
*/
//ATO_AKM_EXPORT int ato_cr_renew(ato_Ctx *ctx, ato_Credential *cr, const char *pwd);

/**
Return the integrityvalue stored in the keystore for the credential.

@param cr the credential object.

@return An ato_String object containing the data, which is null terminated.
*/
ATO_AKM_EXPORT ato_String *ato_cr_integrityvalue(ato_Credential *cr);

/**
Return the salt stored in the keystore for the credential.

@param cr the credential object.

@return An ato_String object containing the data, which is null terminated.
*/
ATO_AKM_EXPORT ato_String *ato_cr_b64salt(ato_Credential *cr);

/**
Return the base64 encoded P7c stored in the keystore for the credential.

@param cr the credential object.

@return An ato_String object containing the data, which is null terminated.
*/
ATO_AKM_EXPORT ato_String *ato_cr_b64p7(ato_Credential *cr);

/**
Return the base64 encoded P8 stored in the keystore for the credential.

@param cr the credential object.

@return An ato_String object containing the data, which is null terminated.
*/
ATO_AKM_EXPORT ato_String *ato_cr_b64p8(ato_Credential *cr);

/**
Sign some data using the private key
- see also ato_cr_isrenewable() and ato_cr_renew().

@param ctx the current thread local Context.
@param cr the credential object.
@param signed_data the address of an ato_String object to allocate. The caller must free this using ato_str_free().
@param pwd the password used to access the private key.
@param data the data to sign.
@param len the length of data.

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_BADPWD if the  password is wrong
*/
ATO_AKM_EXPORT int ato_cr_sign(ato_Ctx *ctx, ato_Credential *cr, ato_String **signed_data, const char *pwd, const char *data, size_t len);

/**
Credential request.

@param ctx the current thread local Context.
@param cr the credential object.
@param pwd the password used to access the private key.
@param id used to match request/response - should be unique.

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_NOTRENEWABLE if credential cannot be "renewed"
- ATO_AKM_ERR_BADPWD if the  password is wrong
- other
*/
ATO_AKM_EXPORT int ato_cr_request_generate(ato_Ctx *ctx, ato_Credential *cr, const char *pwd, const char *id);

/**
Returns the p10 request

@param cr the credential object.

@return request data.
*/
ATO_AKM_EXPORT ato_String *ato_cr_b64p10(ato_Credential *cr);

/**
Return the data generated from ato_cr_request_generate().

@param cr the credential object.

@return request data.
*/
ATO_AKM_EXPORT ato_String *ato_cr_request_data(ato_Credential *cr);

/**
Credential request.

@param ctx the current thread local Context.
@param cr the credential object.
@param responseData the response from the service.

@return ATO_ERR_OK(0) if successful
*/
ATO_AKM_EXPORT int ato_cr_request_process(ato_Ctx *ctx, ato_Credential *cr, const char *responseData);

#ifdef __cplusplus
}
#endif

/*! @} */

#endif /*__ATO_AKM_CR_H__*/
