#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"
#include "atoksprop.h"
#include "atorenew.h"
#include "atocredential.h"

static const char *_library = ATO_AKM_LIBRARY;
static const char *_module = ATO_AKM_MODULE_CR;
static unsigned long _moduleid = ATO_AKM_MODULEID_CR;
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

struct _ato_Credential {
    ato_ksProperties *ksprop;
        ato_Xml *xml;
        ato_Crypto *crypto;
        void *xnode;
        char *id;
        char *credentialType;
        ato_String *integrityValue;
        char *name1;
        char *name2;
        char *abn;
        char *personId;
        ato_String *sha1fingerprint;
    char *sha1fingerprint_hex;
        char *legalName;
        char *creationDate;
        char *serialNumber;
        char *notBefore;
        char *notAfter;
        bool isdirty;
        bool isrenewalcheckdone;
        ato_Crr *crr;
};

/*********************************************************************************/
static const char *_xpath_creds_ctr = "/ns:credentialStore/ns:credentials";
//static const char *_xpath_cred = "/ns:credentialStore/ns:credentials/ns:credential[@id=\"$\"]";

static const char *_xpath_cr_id = "@id";
static const char *_xpath_cr_credentialType = "@credentialType";
static const char *_xpath_cr_credentialSalt = "@credentialSalt";
static const char *_xpath_cr_integrityValue = "@integrityValue";
static const char *_xpath_cr_name1 = "ns:name1";
static const char *_xpath_cr_name2 = "ns:name2";
static const char *_xpath_cr_abn = "ns:abn";
static const char *_xpath_cr_legalName = "ns:legalName";
static const char *_xpath_cr_personId = "ns:personId";
static const char *_xpath_cr_serialNumber = "ns:serialNumber";
static const char *_xpath_cr_creationDate = "ns:creationDate";
static const char *_xpath_cr_notBefore = "ns:notBefore";
static const char *_xpath_cr_notAfter = "ns:notAfter";
static const char *_xpath_cr_sha1fingerprint = "ns:sha1fingerprint";
static const char *_xpath_cr_publicCertificate = "ns:publicCertificate";
static const char *_xpath_cr_protectedPrivateKey = "ns:protectedPrivateKey";

static bool test_date_set = FALSE;
static time_t test_date;

void set_now(time_t to_set) {
        test_date_set = TRUE;
        test_date = to_set;
}

time_t get_now(void) {
        if (test_date_set) {
                return test_date;
        } else {
                time_t now;
                now = time(&now);
                return now;
        }
}

/*********************************************************************************/
/*
//http://stackoverflow.com/a/12839870/716662
void tohex(const char *in, size_t insz, char *out)
{
        char *pin = (char *)in;
        const char *hex = "0123456789abcdef";
        char *pout = out;
        for (; pin < in + insz; pout += 2, pin++) {
                pout[0] = hex[(*pin>>4) & 0xf];
                pout[1] = hex[*pin & 0xf];
        }
}*/

static char *_cr_sha1fingerprint_hex(ato_Ctx *ctx, ato_Credential *cr) {
    if (cr->sha1fingerprint_hex == NULL) {
            ato_String *decoded = NULL;
        int errcode = ato_base64decode(ctx, &decoded, cr->sha1fingerprint);
            if (errcode == ATO_ERR_OK) {
            unsigned int i = 0;
            size_t len = ato_str_len(decoded);
            const char *ch_decoded = ato_str_value(decoded);
            cr->sha1fingerprint_hex = ato_alloc((len+5)*2);
            for (i = 0; i < len; i++) {
                sprintf(cr->sha1fingerprint_hex + i*2, "%02X", (unsigned char) (ch_decoded[i]));
            }
        }
        ato_str_free(decoded);
    }
        return cr->sha1fingerprint_hex;
}

/*********************************************************************************/
static void _deletenode(ato_Credential *cr)
{
    if (cr->xml == NULL) { return; }
        ato_xml_deletenode(cr->xml, cr->xnode);
}

// Node doesn't exist, we need to create it
static void _addcrsalt(ato_Ctx *ctx, ato_Credential *cr, char **credentialSalt)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_addcrsalt";
        char *s = NULL;
        void *xnode = NULL;

    if (cr->xml == NULL) { return; }
        if (!ato_xml_addattr(cr->xml, cr->xnode, _xpath_cr_credentialSalt, ato_ksprop_salt(cr->ksprop)))
                Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_SAVE, "Error in adding credential salt into credential.");

        xnode = ato_xml_findnode(cr->xml, cr->xnode, _xpath_cr_credentialSalt, NULL);
        s = *credentialSalt = ato_xml_value(cr->xml, xnode, credentialSalt);
        if (s == NULL)
                Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_LOAD, "Cannot find credential salt after adding into credential.");
}

// Node exists but there is no value, so set it
static void _setcrsalt(ato_Ctx *ctx, ato_Credential *cr, char **credentialSalt)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_setcrsalt";
        char *s = NULL;

    if (cr->xml == NULL) { return; }
        s = *credentialSalt = ato_strdup(ato_ksprop_salt(cr->ksprop), 0);
        if (s == NULL) {
                Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_LOAD, "Cannot find keystore salt.");
        } else if (!ato_xml_setnodevalue(cr->xml, cr->xnode, _xpath_cr_credentialSalt, s)) {
                Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_SAVE, "Error in saving credential salt into credential.");
        }
}

static void _checkcrsalt(ato_Ctx *ctx, ato_Credential *cr, char **credentialSalt)
{
        void *xnode = NULL;

    if (cr->xml == NULL) { return; }
        xnode = ato_xml_findnode(cr->xml, cr->xnode, _xpath_cr_credentialSalt, NULL);
        if (xnode == NULL) {
                _addcrsalt(ctx, cr, credentialSalt);
        } else {
                char *s = ato_xml_value(cr->xml, xnode, credentialSalt);
                if (s == NULL || strlen(s) == 0) {
                        if (s != NULL) free(s);
                        _setcrsalt(ctx, cr, credentialSalt);
                }
        }
}
static void _load(ato_Ctx *ctx, ato_Credential *cr, char **publicCertificate, char **protectedPrivateKey, char **credentialSalt)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_load";
    if (cr->xml == NULL) { return; }
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_id, &(cr->id), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_id), "Failed to find credential value %s", _xpath_cr_id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_credentialType, &(cr->credentialType), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_credentialType) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_credentialType, cr->id);
        {
                char *data = NULL;
                if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_integrityValue, &data, ATO_XML_RAW_FALSE) == NULL)
                        Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_integrityValue) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_integrityValue, cr->id);
                ato_str_create(&(cr->integrityValue), data, strlen(data), TRUE);
        }
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_name1, &(cr->name1), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_name1) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_name1, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_name2, &(cr->name2), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_name2) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_name2, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_abn, &(cr->abn), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_abn) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_abn, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_legalName, &(cr->legalName), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_legalName) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_legalName, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_personId, &(cr->personId), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_personId) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_personId, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_serialNumber, &(cr->serialNumber), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_serialNumber) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_serialNumber, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_creationDate, &(cr->creationDate), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_creationDate) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_creationDate, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_notBefore, &(cr->notBefore), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_notBefore) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_notBefore, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_notAfter, &(cr->notAfter), ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_notAfter) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_notAfter, cr->id);
        {
                char *sha1fingerprint = NULL;
                if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_sha1fingerprint, &sha1fingerprint, ATO_XML_RAW_FALSE) == NULL)
                        Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_sha1fingerprint) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_sha1fingerprint, cr->id);
                ato_str_create(&(cr->sha1fingerprint), sha1fingerprint, strlen(sha1fingerprint), TRUE);
        }
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_publicCertificate, publicCertificate, ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_publicCertificate) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_publicCertificate, cr->id);
        if (ato_xml_nodevalue(cr->xml, cr->xnode, _xpath_cr_protectedPrivateKey, protectedPrivateKey, ATO_XML_RAW_FALSE) == NULL)
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_LOAD, strlen(_xpath_cr_protectedPrivateKey) + strlen(cr->id), "Failed to find value %s for credential %s", _xpath_cr_protectedPrivateKey, cr->id);

        _checkcrsalt(ctx, cr, credentialSalt);
}

static ato_eCredtype _type(ato_Credential *cr)
{
        char type = '\0';
        assert(cr != NULL);
        type = cr->credentialType[0];
        if (type == 'U')
                return ATO_CREDUSER;
        return ATO_CREDDEVICE;
}
static bool _isuser(ato_Credential *cr)
{
        return _type(cr) == ATO_CREDUSER;
}

static bool _isdevice(ato_Credential *cr)
{
        return _type(cr) == ATO_CREDDEVICE;
}

static void _calculate_integrityvalue(ato_Ctx *ctx, ato_String **integrityvalue, ato_Credential *cr)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_calculate_integrityvalue";
        int errcode = ATO_ERR_OK;
        char *iv = NULL;
        size_t size = 0, len = 0, delta = 0;
        ato_String *p7str = NULL, *p8str = NULL, *fpstr = NULL, *saltstr = NULL, *ivstr = NULL, *hmacstr = NULL;

        ATO_ASSERT_ISNOTALLOCATED(integrityvalue);

        size = strlen(cr->credentialType);
        assert(size); //Incorporating validity check - ensure each element has a value.
        delta = strlen(cr->id);
        assert(delta);
        size += delta;
        delta =  strlen(cr->name1);
        assert(delta);
        size += delta;
        delta =  strlen(cr->name2);
        // for name2 delta may be 0, as no name2 required for device credentials.
        size += delta;
        delta = strlen(cr->abn);
        assert(delta) ;
        size += delta;
        delta = strlen(cr->legalName);
        assert(delta) ;
        size += delta;
        delta = strlen(cr->personId);
        // personId delta my be 0 for device credentials.

        size += delta;
        delta =  strlen(cr->serialNumber);
        assert(delta);
        size += delta;
        ato_base64decode(ctx, &fpstr, cr->sha1fingerprint);
        delta =  ato_str_len(fpstr);
        assert(delta);
        size += delta;
        ato_base64decode(ctx, &p7str, ato_crypto_b64p7(cr->crypto));
        delta = ato_str_len(p7str);
        assert(delta);
        size += delta;
        ato_base64decode(ctx, &p8str, ato_crypto_b64p8(cr->crypto));
        delta = ato_str_len(p8str);
        assert(delta);
        size += delta;

        iv = calloc(size + 1, sizeof(char));

        strcat(iv, cr->credentialType);
        strcat(iv, cr->id);
        strcat(iv, cr->name1);
        strcat(iv, cr->name2);
        strcat(iv, cr->abn);
        strcat(iv, cr->legalName);
        strcat(iv, cr->personId);
        strcat(iv, cr->serialNumber);
        len = strlen(iv);
        assert(len <= size);
        memcpy(iv+len, ato_str_value(fpstr), ato_str_len(fpstr));
        len += ato_str_len(fpstr);
        assert(len <= size);
        memcpy(iv+len, ato_str_value(p7str), ato_str_len(p7str));
        len += ato_str_len(p7str);
        assert(len <= size);
        memcpy(iv+len, ato_str_value(p8str), ato_str_len(p8str));
        len += ato_str_len(p8str);
        assert(len == size);

        ato_str_create(&ivstr, iv, size, FALSE);

        ato_base64decode(ctx, &saltstr, ato_crypto_b64salt(cr->crypto));

        errcode = ato_generatehmac(ctx, &hmacstr, ivstr, saltstr);
        if (errcode == ATO_ERR_OK) {
                ato_String *b64 = NULL;
                errcode = ato_base64encode(ctx, &b64, hmacstr);
                ato_str_dup2(integrityvalue, b64, TRUE); // We need this as null terminated string
                ato_str_free(b64);
        }

        ato_str_free(p7str);
        ato_str_free(p8str);
        ato_str_free(saltstr);
        ato_str_free(ivstr);
        ato_str_free(hmacstr);
        ato_str_free(fpstr);

        if (errcode != ATO_ERR_OK)
                Throw ATO_CTX_VNEWERR(ctx, errcode, strlen(cr->id), "Failed to calculate integrity value for credential %s", cr->id);
}

static int _expirytime_months(ato_Ctx *ctx, ato_Credential *cr)
{
    int minvalue_months = 14;
    int value = ato_ksprop_var_int(cr->ksprop, "renew.expiry.months", minvalue_months);
    if (value < minvalue_months)
        value = minvalue_months;
    ATO_IGNORE(ctx);
    return value;
}

static void _set_value(ato_Ctx *ctx, ato_Credential *cr, const char *descr, const char *name, const char *value)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_set_value";

    if (cr->xml == NULL) { return; }
        assert(value != NULL);
    if (!ato_xml_setnodevaluelen(cr->xml, cr->xnode, name, value, strlen(value))) {
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_SAVE, strlen(descr), "Error in saving %s into memory store.", descr);
    }
}
static void _set_value_enc(ato_Ctx *ctx, ato_Credential *cr, const char *descr, const char *name, const char *value)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_set_value";

    if (cr->xml == NULL) { return; }
        assert(value != NULL);
    if (!ato_xml_setnodevaluelen(cr->xml, cr->xnode, name, value, 0)) {
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_SAVE, strlen(descr), "Error in saving %s into memory store.", descr);
    }
}

static void _save_reset_iv(ato_Ctx *ctx, ato_Credential *cr) {
    cr->integrityValue = ato_str_free(cr->integrityValue);
        _calculate_integrityvalue(ctx, &(cr->integrityValue), cr);
        _set_value(ctx, cr, "integrityValue", _xpath_cr_integrityValue, ato_str_value(cr->integrityValue));
}

static void _save_set_p8(ato_Ctx *ctx, ato_Credential *cr, const char *protectedPrivateKey) {
        _set_value(ctx, cr, "protectedPrivateKey", _xpath_cr_protectedPrivateKey, protectedPrivateKey);
}

static void _save_set_p7(ato_Ctx *ctx, ato_Credential *cr, const char *publicCertificate) {
        _set_value(ctx, cr, "publicCertificate", _xpath_cr_publicCertificate, publicCertificate);
}

static void _save_set_salt(ato_Ctx *ctx, ato_Credential *cr, const char *salt) {
        _set_value(ctx, cr, "credentialSalt", _xpath_cr_credentialSalt, salt);
}

static void _save_set_crypto(ato_Ctx *ctx, ato_Credential *cr, ato_Crypto *crypto) {
        cr->crypto = ato_crypto_free(cr->crypto);
    cr->crypto = crypto;
    _save_set_salt(ctx, cr, ato_str_value(ato_crypto_b64salt(crypto)));
    _save_set_p7(ctx, cr, ato_str_value(ato_crypto_b64p7(crypto)));
    _save_set_p8(ctx, cr, ato_str_value(ato_crypto_b64p8(crypto)));
}

static void _save_set_sha1fingerprint(ato_Ctx *ctx, ato_Credential *cr, char *sha1fingerprint) {
        cr->sha1fingerprint = ato_str_free(cr->sha1fingerprint);
        ato_str_create(&(cr->sha1fingerprint), sha1fingerprint, strlen(sha1fingerprint), TRUE);
        _set_value(ctx, cr, "sha1fingerprint", _xpath_cr_sha1fingerprint, sha1fingerprint);
}
static void _save_set_legalName(ato_Ctx *ctx, ato_Credential *cr, char *legalName) {
        cr->legalName = ato_free(cr->legalName);
        cr->legalName = legalName;
        _set_value_enc(ctx, cr, "legalName", _xpath_cr_legalName, legalName);
}
static void _save_set_creationDate(ato_Ctx *ctx, ato_Credential *cr, char *creationDate) {
        cr->creationDate = ato_free(cr->creationDate);
        cr->creationDate = creationDate;
        _set_value(ctx, cr, "creationDate", _xpath_cr_creationDate, creationDate);
}

static void _save_set_serialNumber(ato_Ctx *ctx, ato_Credential *cr, char *serialNumber) {
        cr->serialNumber = ato_free(cr->serialNumber);
        cr->serialNumber = serialNumber;
        _set_value(ctx, cr, "serialNumber", _xpath_cr_serialNumber, serialNumber);
}

static void _save_set_notBefore(ato_Ctx *ctx, ato_Credential *cr, char *notBefore) {
        cr->notBefore = ato_free(cr->notBefore);
        cr->notBefore = notBefore;
        _set_value(ctx, cr, "notBefore", _xpath_cr_notBefore, notBefore);
}

static void _save_set_notAfter(ato_Ctx *ctx, ato_Credential *cr, char *notAfter) {
        cr->notAfter = ato_free(cr->notAfter);
        cr->notAfter = notAfter;
        _set_value(ctx, cr, "notBefore", _xpath_cr_notAfter, notAfter);
}

static void _save_crr_new(ato_Ctx *ctx, ato_Credential *cr) {
        void *pnode = NULL;
        ato_Xml *xml = NULL;
        void *node = NULL;

        cr->isdirty = TRUE;
        cr->xml = xml = (ato_Xml *)ato__ksprop_xml(cr->ksprop);
        pnode = ato_xml_findnode(xml, NULL, _xpath_creds_ctr, NULL);
        cr->xnode = node = ato_xml_createelt(xml, pnode, "credential");

        ato_xml_createattr(xml, node, _xpath_cr_integrityValue, NULL, NULL);
        ato_xml_createattr(xml, node, _xpath_cr_credentialType, NULL, NULL);
        ato_xml_createattr(xml, node, _xpath_cr_id, NULL, NULL);
        ato_xml_createattr(xml, node, _xpath_cr_credentialSalt, NULL, NULL);

        ato_xml_createelt(xml, node, _xpath_cr_name1);
        ato_xml_createelt(xml, node, _xpath_cr_name2);
        ato_xml_createelt(xml, node, _xpath_cr_abn);
        ato_xml_createelt(xml, node, _xpath_cr_legalName);
        ato_xml_createelt(xml, node, _xpath_cr_personId);
        ato_xml_createelt(xml, node, _xpath_cr_serialNumber);
        ato_xml_createelt(xml, node, _xpath_cr_creationDate);
        ato_xml_createelt(xml, node, _xpath_cr_notBefore);
        ato_xml_createelt(xml, node, _xpath_cr_notAfter);
        ato_xml_createelt(xml, node, _xpath_cr_sha1fingerprint);
        ato_xml_createelt(xml, node, _xpath_cr_publicCertificate);
        ato_xml_createelt(xml, node, _xpath_cr_protectedPrivateKey);

        _set_value(ctx, cr, "credentialType", _xpath_cr_credentialType, cr->credentialType);
        _set_value_enc(ctx, cr, "id", _xpath_cr_id, cr->id);
        _set_value(ctx, cr, "abn", _xpath_cr_abn, cr->abn);
        _set_value_enc(ctx, cr, "name1", _xpath_cr_name1, cr->name1);
        _set_value_enc(ctx, cr, "name2", _xpath_cr_name2, cr->name2);
        _set_value(ctx, cr, "personId", _xpath_cr_personId, cr->personId);
}

static void _save_crr(ato_Ctx *ctx, ato_Credential *cr) {
    ato_String *str = NULL;
    ato_Crypto *crypto = NULL;
    if (cr->crr == NULL) { return; }
    crypto = ato__crr_crypto_asowner(cr->crr);

        _save_set_crypto(ctx, cr, crypto);
        _save_set_sha1fingerprint(ctx, cr, ato_strdup(ato__crr_fingerprint(cr->crr), 0));
        //_save_set_legalName(ctx, cr, ato_strdup(ato__crr_legalName(cr->crr), 0));
        _save_set_legalName(ctx, cr, ato_strdup(cr->legalName, 0));
        _save_set_creationDate(ctx, cr, ato_strdup(ato__crr_creationDate(cr->crr), 0));

        _save_set_serialNumber(ctx, cr, ato_str_valueasowner(ato_crypto_serialnr(ctx, crypto, &str)));
    str = ato_str_free(str);
        _save_set_notBefore(ctx, cr, ato_str_valueasowner(ato_crypto_notbefore(ctx, crypto, &str)));
    str = ato_str_free(str);
        _save_set_notAfter(ctx, cr, ato_str_valueasowner(ato_crypto_notafter(ctx, crypto, &str)));
    str = ato_str_free(str);

    ato_ksprop_PbeMode_Set(ctx, cr->ksprop, ato__crr_pbemode(cr->crr));
}

static bool _is_integrity_valid(ato_Ctx *ctx, ato_Credential *cr) {
    ato_String *newiv = NULL;
    bool ok = FALSE;
    _calculate_integrityvalue(ctx, &newiv, cr);
    ok = (newiv != NULL && ato_str_eq(newiv, cr->integrityValue));
    ato_str_free(newiv);
    return ok;
}

static int _cr_request_create(ato_Ctx *ctx, ato_Credential *cr, const char *id)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_cr_create_request";
        int errcode = ATO_ERR_OK;

    if (ato_cr_isrenewed(cr)) { return ATO_ERR_OK; }

    if (!ato_cr_isrenewable(ctx, cr)) {
                Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_NOTRENEWABLE, cr->id);
    }



    if (cr->crypto != NULL) {
        ato__crr_create(ctx, &(cr->crr), cr->ksprop, id, cr->credentialType, cr->crypto);
        if (cr->sha1fingerprint != NULL) {
            // test renew
        }
    } else {
        ato__crr_createnew(ctx, &(cr->crr), cr->ksprop, id, cr->credentialType, cr->abn, cr->name1, cr->name2, cr->personId);
    }

    return errcode;
}
static bool _cr_iscorrectpwd(ato_Ctx *ctx, ato_Credential *cr, ato_PwdKdf *pk)
{
    bool correct = TRUE;
    if (cr->crypto != NULL) {
        ato_crypto_iscorrectpwd(ctx, cr->crypto, &correct, pk);
    }
    return correct;
}

static ato_PwdKdf *_getpwd(ato_Ctx *ctx, ato_Credential *cr, const char *pwd) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "_getpwd";
    ato_PwdKdf *pk = ato__ksprop_pk(ctx, cr->ksprop, pwd);
    if (!_cr_iscorrectpwd(ctx, cr, pk)) {
        ato__ksprop_pk_clr(ctx, cr->ksprop);
                Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_BADPWD, cr->id);
    }
    return pk;
}

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
        _loglevel = level;
}
/*********************************************************************************/
int ato__cr_init(void)
{
        static bool invoked = FALSE;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        return ATO_ERR_OK;
}

void ato__cr_deinit(void)
{
}

/*********************************************************************************/
void ato__cr_create(ato_Ctx *ctx, ato_Credential **obj, ato_ksProperties *ksprop, void *xnode)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato__cr_create";
        int errcode = ATO_ERR_OK;
        char *publicCertificate = NULL, *protectedPrivateKey = NULL, *salt = NULL;
        ato_Credential *cr = NULL;
        ato_Log *log = NULL;

        log = ato_ctx_log(ctx);
        ATO_CTX_FN_START(ctx);
        ATO_ASSERT_ISNOTALLOCATED(obj);
        assert(ksprop != NULL); assert(xnode != NULL);

        *obj = cr = calloc(1, sizeof(ato_Credential));
        assert(cr != NULL);

    cr->ksprop = ksprop;
        cr->xml = (ato_Xml *)ato__ksprop_xml(ksprop);
        cr->xnode = xnode;

        Try {
                _load(ctx, cr, &publicCertificate, &protectedPrivateKey, &salt);
                ato_crypto_create(ctx, &(cr->crypto), publicCertificate, protectedPrivateKey, salt);

                if (ato_value_bool(ato_ksprop_var(cr->ksprop, ato_ksprop_varn_checkIntegrityValue()), TRUE)) {
                    if (!_is_integrity_valid(ctx, cr))
                        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_INTEGRITYVALUE, "Integrity value failed to validate.");
                }
                ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Created cr %s", cr->id);

        } Catch(errcode) {
           //printf("caught error %d.\n", errcode);
           ATO_LOG_MSGV(log, ATO_LOG_ERR, "Error loading credential cr %s", cr->id);
        }

        if (publicCertificate != NULL) free(publicCertificate);
        if (protectedPrivateKey != NULL) free(protectedPrivateKey);
        if (salt != NULL) free(salt);
    ATO_RETHROW_ONERR(errcode);
        ATO_CTX_FN_END(ctx, errcode);
}

void ato__cr_createnew(ato_Ctx *ctx, ato_Credential **obj, ato_ksProperties *ksprop, const char *id, const char *abn, const char *name1, const char *name2, const char *pid) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato__cr_createnew";
        int errcode = ATO_ERR_OK;
        ato_Credential *cr = NULL;
        ato_Log *log = NULL;

        log = ato_ctx_log(ctx);
        ATO_CTX_FN_START(ctx);
        ATO_ASSERT_ISNOTALLOCATED(obj);
        assert(ksprop != NULL);

        *obj = cr = calloc(1, sizeof(ato_Credential));
        assert(cr != NULL);

    cr->ksprop = ksprop;
        //cr->xml = (ato_Xml *)ato__ksprop_xml(ksprop);
        //cr->xnode = ato__cr_clonenode(ctx, cr->xml, cr);
        //cr->isdirty = TRUE;

        Try {
        cr->id = ato_strdup(id, 0);
        cr->abn = ato_strdup(abn, 0);
        cr->name1 = ato_strdup(name1, 0);
        cr->name2 = ato_strdup(name2, 0);
        cr->personId = ato_strdup(pid, 0);
        cr->credentialType = ato_strdup(((cr->personId == NULL || strlen(cr->personId) == 0) ? "D" : "U"), 0);

                ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Created empty cr %s", cr->id);

        } Catch(errcode) {
           //printf("caught error %d.\n", errcode);
           ATO_LOG_MSGV(log, ATO_LOG_ERR, "Error creating empty cr %s", cr->id);
        }

    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

void *ato__cr_clonenode(ato_Ctx *ctx, ato_Xml *dstksxml, ato_Credential *srccr)
{
        void *node = ato_xml_addnodecopy(dstksxml, _xpath_creds_ctr, srccr->xnode, TRUE);
        ATO_IGNORE(ctx);
        return node;
}

void *ato__cr_free(ato_Credential *cr, bool deletenode)
{
        if (cr == NULL) return NULL;
        if (deletenode) _deletenode(cr);
        cr->crr = ato__crr_free(cr->crr);

        cr->id = ato_free(cr->id);
        cr->credentialType = ato_free(cr->credentialType);
        cr->name1 = ato_free(cr->name1);
        cr->name2 = ato_free(cr->name2);
        cr->abn = ato_free(cr->abn);
        cr->personId = ato_free(cr->personId);
        cr->legalName = ato_free(cr->legalName);

        cr->integrityValue = ato_str_free(cr->integrityValue);
        cr->serialNumber = ato_free(cr->serialNumber);
        cr->creationDate = ato_free(cr->creationDate);
        cr->notBefore = ato_free(cr->notBefore);
        cr->notAfter = ato_free(cr->notAfter);
        cr->sha1fingerprint = ato_str_free(cr->sha1fingerprint);
    cr->sha1fingerprint_hex = ato_free(cr->sha1fingerprint_hex);
        cr->crypto = ato_crypto_free(cr->crypto);

        free(cr);
    return NULL;
}

bool ato__cr_iscorrectpwd(ato_Ctx *ctx, ato_Credential *cr, ato_PwdKdf *pk)
{
    return _cr_iscorrectpwd(ctx, cr, pk);
}

//TODO: what about renewed/new crs? Possible disable changepwd if there are any such crs in the keystore?
void ato__cr_changepwd(ato_Ctx *ctx, ato_Credential *cr, ato_PwdKdf *pk, ato_PwdKdf *newpk)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato__cr_changepwd";

    if (cr->crypto == NULL) { return; }
    if (!ato_crypto_changepwd(ctx, cr->crypto, pk, newpk)) {
                Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_BADPWD, strlen(cr->id), "Credential %s", cr->id);
    }
    cr->isdirty = TRUE;
    _save_set_p8(ctx, cr, ato_str_value(ato_crypto_b64p8(cr->crypto)));
    cr->integrityValue = ato_str_free(cr->integrityValue);
}

ato_ksProperties *ato__cr_properties(ato_Credential *cr)
{
    assert(cr != NULL);
    return cr->ksprop;
}

void ato__cr_save(ato_Ctx *ctx, ato_Credential *cr) {
    assert(cr != NULL);
        if (cr->xml == NULL) {
                _save_crr_new(ctx, cr);
        }
    if (cr->isdirty) {
        _save_crr(ctx, cr);
        _save_reset_iv(ctx, cr);
    }

}

ato_eCredtype ato_cr_type(ato_Credential *cr) { assert(cr != NULL); return _type(cr); }
const char *ato_cr_abn(ato_Credential *cr) { assert(cr != NULL); return cr->abn; }
const char *ato_cr_legalname(ato_Credential *cr) { assert(cr != NULL); return cr->legalName; }
const char *ato_cr_serialnr(ato_Credential *cr) { assert(cr != NULL); return cr->serialNumber; }
const char *ato_cr_creationdate(ato_Credential *cr) { assert(cr != NULL); return cr->creationDate; }
const char *ato_cr_notbefore(ato_Credential *cr) { assert(cr != NULL); return cr->notBefore; }
const char *ato_cr_notafter(ato_Credential *cr) { assert(cr != NULL); return cr->notAfter; }
const char *ato_cr_sha1fingerprint(ato_Credential *cr) { assert(cr != NULL); return ato_str_value(cr->sha1fingerprint); }
const char *ato_cr_sha1fingerprint_hex(ato_Ctx *ctx, ato_Credential *cr) { assert(cr != NULL); return _cr_sha1fingerprint_hex(ctx, cr); }
const char *ato_cr_personid(ato_Credential *cr) { assert(cr != NULL); return _isuser(cr) ? cr->personId : NULL; }
const char *ato_cr_devicename(ato_Credential *cr) { assert(cr != NULL); return _isdevice(cr) ? cr->name1 : NULL; }
const char *ato_cr_givenames(ato_Credential *cr) { assert(cr != NULL); return _isuser(cr) ? cr->name1 : NULL; }
const char *ato_cr_familyname(ato_Credential *cr) { assert(cr != NULL); return _isuser(cr) ? cr->name2 : NULL; }

ato_String *ato_cr_integrityvalue(ato_Credential *cr) { assert(cr != NULL); return cr->integrityValue; }
ato_String *ato_cr_b64salt(ato_Credential *cr) { assert(cr != NULL); return ato_crypto_b64salt(cr->crypto); }
ato_String *ato_cr_b64p7(ato_Credential *cr) { assert(cr != NULL); return ato_crypto_b64p7(cr->crypto); }
ato_String *ato_cr_b64p8(ato_Credential *cr) { assert(cr != NULL); return ato_crypto_b64p8(cr->crypto); }

ato_String *ato_cr_request_data(ato_Credential *cr) { assert(cr != NULL); return ato__crr_requestdata(cr->crr); }
ato_String *ato_cr_b64p10(ato_Credential *cr) { assert(cr != NULL); return ato__crr_b64p10(cr->crr); }

bool ato_cr_ismodified(ato_Credential *cr)
{
        assert(cr != NULL);
        return cr->isdirty;
}

void atoe_cr_certificate(ato_Ctx *ctx, ato_Credential *cr, ato_String **certificate)
{
    ato_crypto_certificate(ctx, cr->crypto, certificate);
}


int ato_cr_certificate(ato_Ctx *ctx, ato_Credential *cr, ato_String **certificate)
{
        static const char *function = "ato_cr_certificate";
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_certificate(ctx, cr, certificate);
        } Catch (errcode) {
                errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}

void atoe_cr_chain(ato_Ctx *ctx, ato_Credential *cr, ato_String **certificates)
{
    ato_crypto_chain(ctx, cr->crypto, certificates);
}


int ato_cr_chain(ato_Ctx *ctx, ato_Credential *cr, ato_String **certificates)
{
        static const char *function = "ato_cr_chain";
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_chain(ctx, cr, certificates);
        } Catch (errcode) {
                errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}

void atoe_cr_privatekey(ato_Ctx *ctx, ato_Credential *cr, ato_String **privatekey, const char *pwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "atoe_cr_privatekey";
    int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL;
    ATO_CTX_FN_START(ctx);

    assert(pwd != NULL);
    pk = _getpwd(ctx, cr, pwd);
    Try {
        ////////////// skip until we reimplement renewal
        //if (cr->isrenewalcheckdone != TRUE)
        //    Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEWNOTCALLED, "When accessing the private key first call the renewal method");
        if (!ato_crypto_privatekey(ctx, cr->crypto, privatekey, pk))
            Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_BADPWD, strlen(cr->id), "Credential %s", cr->id);
    } Catch (errcode) {
    }
    ato__ksprop_pk_clr(ctx, cr->ksprop);
    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

int ato_cr_privatekey(ato_Ctx *ctx, ato_Credential *cr, ato_String **privatekey, const char *pwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_cr_privatekey";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        atoe_cr_privatekey(ctx, cr, privatekey, pwd);
    } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

//static void _checkNotBefore(ato_Ctx *ctx, ato_Credential *cr) {
//    ato_String *notBefore = NULL;
//    ato_crypto_notbefore(ctx, cr->crypto, &notBefore);
//    printf("\n");
//    printf("%s\n", "*************************************************");
//    printf("notBefore=%s\n", ato_str_value(notBefore));
//    printf("%s\n", "*************************************************");
//    ato_str_free(notBefore);
//}

void atoe_cr_p12(ato_Ctx *ctx, ato_Credential *cr, ato_String **p12, const char *pwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "atoe_cr_p12";
    int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL;
    ATO_CTX_FN_START(ctx);

    assert(pwd != NULL);
    //pk = ato__ksprop_pk(ctx, cr->ksprop, pwd);
    pk = _getpwd(ctx, cr, pwd);
    Try {
        //_checkNotBefore(ctx, cr);
        ////////////// skip until we reimplement renewal
        //if (cr->isrenewalcheckdone != TRUE)
        //        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEWNOTCALLED, "When accessing the private key first call the renewal method");
        if (!ato_crypto_p12(ctx, cr->crypto, p12, pk))
            Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_BADPWD, strlen(cr->id), "Credential %s", cr->id);
    } Catch (errcode) {
    }
    ato__ksprop_pk_clr(ctx, cr->ksprop);
    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

int ato_cr_p12(ato_Ctx *ctx, ato_Credential *cr, ato_String **p12, const char *pwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_cr_p12";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        atoe_cr_p12(ctx, cr, p12, pwd);
    } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

bool ato_cr_iscurrent(ato_Credential *cr)
{
        time_t start, end, now;
        start = ato_str2time(ato_cr_notbefore(cr));
        end = ato_str2time(ato_cr_notafter(cr));
        now = get_now();
        //now = time(&now);
        return start <= now && now <= end;
}
bool ato_cr_isvalid(ato_Credential *cr)
{
        return ato_cr_iscurrent(cr);
}

bool ato_is_integrity_valid(ato_Ctx *ctx, ato_Credential *cr) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_is_integrity_valid";
        int errcode = ATO_ERR_OK;
        //ato_Log *log = NULL;
    bool ok = FALSE;

        //log = ato_ctx_log(ctx);
        ATO_CTX_FN_START(ctx);

        Try {
        ok = _is_integrity_valid(ctx, cr);
        } Catch (errcode) {
        ok = FALSE;
                errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }

        ATO_CTX_FN_END(ctx, errcode);
        return ok;
}


int ato_cr_checkcorrectpwd(ato_Ctx *ctx, ato_Credential *cr, const char *pwd)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_cr_checkcorrectpwd";
        int errcode = ATO_ERR_OK;

        ATO_CTX_FN_START(ctx);
    assert(pwd != NULL);

        Try {
                _getpwd(ctx, cr, pwd);
        } Catch (errcode) {
                errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }

    ato__ksprop_pk_clr(ctx, cr->ksprop);
    ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}

int ato_cr_iscorrectpwd(ato_Ctx *ctx, ato_Credential *cr, bool *correct, const char *pwd)
{
        int errcode = ato_cr_checkcorrectpwd(ctx, cr, pwd);
        *correct = errcode == ATO_ERR_OK;
        return errcode;
}

const char *ato_cr_id(ato_Credential *cr)
{
        assert(cr != NULL);
        return cr->id;
}
const char *ato_cr_alias(ato_Credential *cr)
{
        return ato_cr_id(cr);
}

bool ato_cr_matchesfilter(ato_Credential *cr, ato_eCredfilter filter)
{
        bool match = FALSE;
        assert(cr != NULL);

    if (cr->xml == NULL) { return FALSE; } // Always ignore new credentials

        switch (filter) {
                case ATO_FILTER_ALL:
                        match = 1;
                        break;
                case ATO_FILTER_VALID:
                        match = ato_cr_iscurrent(cr);
                        break;
                case ATO_FILTER_USER:
                        match = (ato_cr_type(cr) == ATO_CREDUSER);
                        break;
                case ATO_FILTER_VALIDUSER:
                        match = (ato_cr_iscurrent(cr) && ato_cr_type(cr) == ATO_CREDUSER);
                        break;
                case ATO_FILTER_DEVICE:
                        match = (ato_cr_type(cr) == ATO_CREDDEVICE);
                        break;
                case ATO_FILTER_VALIDDEVICE:
                        match = (ato_cr_iscurrent(cr) && ato_cr_type(cr) == ATO_CREDDEVICE);
                        break;
        }
        return match;
}

//char *ato__cr_str(ato_Credential *cr, char **str) {
//    return ato_xml_nodevaluevar(cr->xml, NULL, _xpath_cred, cr->id, str, ATO_XML_RAW_OUTER);
//}

bool ato_cr_ispbe2(ato_Ctx *ctx, ato_Credential *cr) {
    ATO_IGNORE(ctx);
    return ato_crypto_ispbe2(cr->crypto);
}

bool ato_cr_isrenewed(ato_Credential *cr)
{
        assert(cr != NULL);
        return cr->crr != NULL;
}

static bool _disable_renewal = FALSE;
bool ato_cr_isrenewable(ato_Ctx *ctx, ato_Credential *cr)
{
        assert(ctx != NULL);
        assert(cr != NULL);
    if (ato_cr_isrenewed(cr)) { return FALSE; }
    if (cr->xml == NULL) { return TRUE; }
        cr->isrenewalcheckdone = TRUE;
    if (_disable_renewal) { return FALSE; }
        {
                time_t start, end, now, expiry;
                struct tm *t;

                start = ato_str2time(ato_cr_notbefore(cr));
                end = ato_str2time(ato_cr_notafter(cr));
                now = get_now();
                //now = time(&now);

                t = localtime(&now);
                t->tm_mon += _expirytime_months(ctx, cr);
                expiry = mktime(t);
                return start <= now && now <= end && end <= expiry;
        }
}

static bool _doDataMatch(ato_Credential *cr) {
    const char *abn = NULL;
    size_t len = 0;
    if (cr == NULL || cr->crr == NULL) { return FALSE; }
    abn = ato__crr_abn(cr->crr);
    if (abn == NULL) { return FALSE; }
    len = strlen(abn);
    return abn[len] == 'u';
}

void atoe_cr_check_data(ato_Ctx *ctx, ato_Credential *cr) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "atoe_cr_check_data";
    char name_buf[240];
    char abn_buf[240];
    ato_Crypto *crypto = ato__crr_crypto_asowner(cr->crr);
    const char *certname = ato_crypto_usercert_name(ctx, crypto, name_buf, sizeof(name_buf) - 5);
    const char *certabn = ato_crypto_usercert_abn(ctx, crypto, abn_buf, sizeof(abn_buf) - 5);
    bool doMatch = _doDataMatch(cr);

    if (certname == NULL || certabn == NULL) {
        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_INVALID_P7, cr->id);
    }
    if (doMatch && strcmp(ato__crr_abn(cr->crr), certabn) != 0) {
        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_MISMATCHED_P7, cr->id);
    }
    if (strcmp(cr->credentialType, "U") == 0) {
        const char *name1 = ato__crr_name1(cr->crr);
        const char *name2 = ato__crr_name2(cr->crr);
        char *name = (char *)calloc((strlen(name1) + strlen(name2) + 2), sizeof(char));

        strcpy(name, name1);
        strcat(name, " ");
        strcat(name, name2);
        if (doMatch && strcmp(name, certname) != 0) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_MISMATCHED_P7, cr->id);
        }
        free(name);

    } else if (strcmp(cr->credentialType, "D") == 0) {
        if (doMatch && strcmp(ato__crr_name1(cr->crr), certname) != 0) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_MISMATCHED_P7, cr->id);
        }
    }
}

int ato_cr_check_data(ato_Ctx *ctx, ato_Credential *cr) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_cr_check_data";
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_check_data(ctx, cr);
        } Catch (errcode) {
            errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}

void atoe_cr_request_generate(ato_Ctx *ctx, ato_Credential *cr, const char *pwd, const char *id)
{
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "atoe_cr_request_generate";
        int errcode = ATO_ERR_OK;
        //ato_Log *log = ato_ctx_log(ctx);
    ato_PwdKdf *pk = NULL;

        ATO_CTX_FN_START(ctx);

        assert(pwd != NULL);

        Try {
        _cr_request_create(ctx, cr, id);
                //pk = ato__ksprop_pk(ctx, cr->ksprop, pwd);
                pk = _getpwd(ctx, cr, pwd);
        ato__crr_renew_generate(ctx, cr->crr, pk);
        } Catch (errcode) {
        }

    ato__ksprop_pk_clr(ctx, cr->ksprop);
    ATO_RETHROW_ONERR(errcode);
        ATO_CTX_FN_END(ctx, errcode);
}

int ato_cr_request_generate(ato_Ctx *ctx, ato_Credential *cr, const char *pwd, const char *id) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_cr_request_generate";
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_request_generate(ctx, cr, pwd, id);
        } Catch (errcode) {
            errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}

void atoe_cr_request_process(ato_Ctx *ctx, ato_Credential *cr, const char *responseData) {
        static const char *function = "atoe_cr_request_process";
        int errcode = ATO_ERR_OK;

        ATO_CTX_FN_START(ctx);

    ato__crr_renew_process(ctx, cr->crr, responseData);
    cr->isdirty = TRUE;
        cr->integrityValue = ato_str_free(cr->integrityValue);

        ATO_CTX_FN_END(ctx, errcode);
}

int ato_cr_request_process(ato_Ctx *ctx, ato_Credential *cr, const char *responseData) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_cr_request_process";
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_request_process(ctx, cr, responseData);
        } Catch (errcode) {
            errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}

ato_String *atoe_cr_request_salt(ato_Credential *cr) {  return ato__crr_salt(cr->crr); }
ato_String *atoe_cr_request_p8(ato_Credential *cr) {  return ato__crr_p8(cr->crr); }
const char *atoe_cr_request_id(ato_Credential *cr) {  return ato__crr_requestid(cr->crr); }


/*
void atoe_cr_request_send(ato_Ctx *ctx, ato_Credential *cr)
{
        static const char *function = "ato_cr_request_send";
        int errcode = ATO_ERR_OK;

        ATO_CTX_FN_START(ctx);
    ato__crr_renew_send(ctx, cr->crr);
        ATO_CTX_FN_END(ctx, errcode);
}

int ato_cr_request_send(ato_Ctx *ctx, ato_Credential *cr) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_cr_request_send";
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_request_send(ctx, cr);
        } Catch (errcode) {
            errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}

void atoe_cr_renew(ato_Ctx *ctx, ato_Credential *cr, const char *pwd) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "atoe_cr_renew";
        int errcode = ATO_ERR_OK;
        ato_Log *log = ato_ctx_log(ctx);
    ato_PwdKdf *pk = NULL;

        ATO_CTX_FN_START(ctx);

        assert(pwd != NULL);

    if (ato_cr_isrenewed(cr)) { return; }

        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Renewing cr %s", cr->id);

    pk = _getpwd(ctx, cr, pwd);
        Try {
        _cr_request_create(ctx, cr, cr->id);
                ato__crr_renew(ctx, cr->crr, pk);
        cr->isdirty = TRUE;
        cr->integrityValue = ato_str_free(cr->integrityValue);
                ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Renewed cr %s", cr->id);
        } Catch (errcode) {
                cr->crr = ato__crr_free(cr->crr);
        if (errcode != ATO_AKM_ERR_BADPWD && errcode != ATO_AKM_ERR_NOTRENEWABLE) {
            // wrap any other error
            errcode = ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, cr->id);
        }
        }

    ato__ksprop_pk_clr(ctx, cr->ksprop);
    ATO_RETHROW_ONERR(errcode);
        ATO_CTX_FN_END(ctx, errcode);
}

int ato_cr_renew(ato_Ctx *ctx, ato_Credential *cr, const char *pwd) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_cr_renew";
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_renew(ctx, cr, pwd);
        } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}
*/

void atoe_cr_sign(ato_Ctx *ctx, ato_Credential *cr, ato_String **signed_data, const char *pwd, const char *data, size_t len) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "atoe_cr_sign";
        int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL;

        ATO_CTX_FN_START(ctx);
        ATO_ASSERT_ISNOTALLOCATED(signed_data);

        assert(pwd != NULL);
        assert(data != NULL);
        assert(len > 0);

        pk = _getpwd(ctx, cr, pwd);
    Try {
            *signed_data = ato_crypto_sign(ctx, cr->crypto, pk, data, len);
    } Catch (errcode) {
    }

    ato__ksprop_pk_clr(ctx, cr->ksprop);
    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

int ato_cr_sign(ato_Ctx *ctx, ato_Credential *cr, ato_String **signed_data, const char *pwd, const char *data, size_t len) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
        static const char *function = "ato_cr_sign";
        int errcode = ATO_ERR_OK;
        ATO_CTX_FN_START(ctx);
        Try {
                atoe_cr_sign(ctx, cr, signed_data, pwd, data, len);
        } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
        }
        ATO_CTX_FN_END(ctx, errcode);
        return errcode;
}
