#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"
#include "atoksprop.h"
#include "atocredential.h"
#include "atokeystore.h"

static const char *_library = ATO_AKM_LIBRARY;
static const char *_module = ATO_AKM_MODULE_KS;
static unsigned long _moduleid = ATO_AKM_MODULEID_KS;
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
struct _ato_Keystore {
    ato_ksProperties *ksprop;
    size_t count;
    ato_List *crlist;
    bool isvalid;
};

/*********************************************************************************/
static const char *DEFAULT_emptyks = "\
<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\
<credentialStore xmlns=\"http://auth.abr.gov.au/credential/xsd/SBRCredentialStore\">\
  <salt>$</salt>\
  <credentials>\
  </credentials>\
</credentialStore>";

static const char *_ks_ns_uri_auth = "http://auth.abr.gov.au/credential/xsd/SBRCredentialStore";
static const char *_ks_ns_uri_ausk = "http://ausk.ato.gov.au/credential/xsd/SBRCredentialStore";

static const char *_ks_ns_prefix = "ns";
static const char *_xpath_store = "/ns:credentialStore";
static const char *_xpath_store_salt = "/ns:credentialStore/ns:salt";
static const char *_xpath_store_pbe = "/ns:credentialStore/ns:pbe";
static const char *_xpath_creds = "/ns:credentialStore/ns:credentials/ns:credential";
static const char *_xpath_cr_id = "@id";
static const char *_xpath_store_pbe_name = "ns:pbe";

#define ATO_MAX_ALIAS_LEN 1000

/*********************************************************************************/
static void _clearlist(ato_Keystore *ks)
{
    ato_ListNode *lnode = NULL;
    if (ks->crlist != NULL) {
        while ((lnode = ato_lst_next(ks->crlist, lnode)) != NULL)
            ato__cr_free(ato_lst_value(lnode), FALSE);
    }
    ato_lst_clear(ks->crlist);
    ks->isvalid = TRUE;
}

static void _setnode(ato_Ctx *ctx, void *obj, const char *id, void *node)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_setnode";
    int errcode = ATO_ERR_OK;
    ato_Credential *cr = NULL;
    ato_Keystore *ks = obj;
    assert(ks != NULL);

    Try {
        ato__cr_create(ctx, &cr, ks->ksprop, node);
        assert(strcmp(ato_cr_alias(cr),id) == 0);
        if (ato_lst_add(ks->crlist, id, cr) == NULL)
            Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_DUPLICATECR, strlen(id), "Failed to load credential %s", id);
    } Catch (errcode) {
        ato__cr_free(cr, FALSE);
        Throw errcode;
    }
}

static ato_Credential *_setnode_new(ato_Ctx *ctx, void *obj, const char *id, const char *abn, const char *name1, const char *name2, const char *pid)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_setnode";
    int errcode = ATO_ERR_OK;
    ato_Credential *cr = NULL;
    ato_Keystore *ks = obj;
    assert(ks != NULL);

    Try {
        ato__cr_createnew(ctx, &cr, ks->ksprop, id, abn, name1, name2, pid);
        assert(strcmp(ato_cr_alias(cr),id) == 0);
        if (ato_lst_add(ks->crlist, id, cr) == NULL)
            Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_DUPLICATECR, strlen(id), "Failed to load credential %s", id);
    } Catch (errcode) {
        ato__cr_free(cr, FALSE);
        Throw errcode;
    }

    return cr;
}

static size_t _countall(ato_Keystore *ks)
{
    return ato_lst_count(ks->crlist);
}

static ato_Xml *_loadxml(ato_Ctx *ctx, ato_String *buffer)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    //const char *function = "_loadxml";
    int errcode = ATO_ERR_OK;
    ato_String *xmlbuffer = NULL;
    ato_Xml *xml = NULL;

    Try {
        if (ato_str_isnullorempty(buffer)) {
            char salt[50];
            char *s = NULL;
            ato_substitute(&s, DEFAULT_emptyks, ato_new_salt(ctx, salt));
            ato_str_create(&xmlbuffer, s, strlen(s), TRUE);
        } else {
            xmlbuffer = buffer;
        }

        ato_xml_create(ctx, &xml, xmlbuffer);

    } Catch (errcode) {
    }

    if (xmlbuffer != buffer) {
        ato_str_free(xmlbuffer);
    }
    ATO_RETHROW_ONERR(errcode);
    return xml;
}

static bool _patch_buffer(ato_String **buf, const char *buffer) {
    bool isLegacy = FALSE;
    assert(buffer != NULL);
    isLegacy = (strstr(buffer, _ks_ns_uri_ausk) == NULL);
    if (isLegacy) {
        ato_str_createconst(buf, buffer, ato_strlen(buffer), TRUE);
    } else {
        char *bufcpy = calloc((strlen(buffer) + 1), sizeof(char));
        char *offset = NULL, *href = NULL;
        strcpy(bufcpy, buffer);
        href = strstr(bufcpy, _ks_ns_uri_ausk);
        assert(href != NULL);
        offset = strstr(href, "ausk");
        assert(offset != NULL);
        memcpy(offset, "auth", 4);
        ato_str_create(buf, bufcpy, ato_strlen(bufcpy), TRUE);
    }
    return isLegacy;
}

static bool _deletecr(ato_ListNode *node)
{
    if (node != NULL) {
        ato_Credential *cr = ato_lst_value(node);
        if (cr != NULL) {
            ato__cr_free(cr, TRUE);
            ato_lst_delete(node);
            return TRUE;
        }
    }
    return FALSE;
}

// Should really use http://www.pcre.org/ library
static bool _pwdpolicy_isvalid(ato_Keystore *ks, const char *pwd)
{
    bool includesLower = FALSE, includesUpper = FALSE, includesNumberOrPunctuation = FALSE;
    size_t i = 0, len = ato_strlen(pwd);

    ATO_IGNORE(ks);
    if (len < 10) return FALSE;
    for (i = 0; i < len; i++) {
        int c = pwd[i];
        if (c < 0x21 || c > 0x7E) { return FALSE; }
        if (islower(c)) {
            includesLower = TRUE;
        } else if (isupper(c)) {
            includesUpper = TRUE;
        } else if (isdigit(c)) {
            includesNumberOrPunctuation = TRUE;
        } else {
            includesNumberOrPunctuation = TRUE;
        }
    }
    return includesLower && includesUpper && includesNumberOrPunctuation;
}

#if defined(_MSC_VER)
__pragma(warning(push))
__pragma(warning(disable:4127))
#endif
static void _changepwd(ato_Ctx *ctx, ato_Keystore *ks, ato_PwdKdf *pk, ato_PwdKdf *newpk)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_changepwd";

    assert(ks != NULL); assert(pk != NULL); assert(newpk != NULL);

    if (!_pwdpolicy_isvalid(ks, ato_pk_rpwd(newpk))) {
        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_PWDPOLICY, "New password does not meet password policy strength requirements");
    }

    if (_countall(ks) == 0) { return; }

    if (!ato__cr_iscorrectpwd(ctx, ato_lst_value(ato_lst_next(ks->crlist, NULL)), pk)) {
        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_BADPWD, "Error while changing password");
    }

    Try {
        ato_ListNode *node = NULL;
        for (node = NULL; (node = ato_lst_next(ks->crlist, node)) != NULL; ) {
            ato_Credential *cr = ato_lst_value(node);
            ato__cr_changepwd(ctx, cr, pk, newpk);
        }
    } Catch_anonymous {
        ks->isvalid = FALSE;
        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_KEYSTORE_CORRUPTED, "Error while changing password - do not save keystore");
    }
}
#if defined(_MSC_VER)
__pragma(warning(pop))
#endif

static size_t _nrsatisfiesfilter(ato_Keystore *ks, ato_eCredfilter filter)
{
    size_t nr = 0;
    ato_ListNode *node = NULL;
    if (filter == ATO_FILTER_ALL)
        return _countall(ks);

    while ((node = ato_lst_next(ks->crlist, node)) != NULL) {
        if (ato_cr_matchesfilter(ato_lst_value(node), filter))
            nr++;
    }
    return nr;
}

/*********************************************************************************/
void atoe_ks_create(ato_Ctx *ctx, ato_Keystore **obj, const char *buffer, const ato_ksProperties *properties) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "atoe_ks_create";
    int errcode = ATO_ERR_OK;
    ato_Keystore *ks = NULL;
    ato_Log *log = NULL;
    ato_String *str = NULL;
    char *salt = NULL, *pbe = NULL;

    log = ato_ctx_log(ctx);
    ATO_CTX_FN_START(ctx);
    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = ks = calloc(1, sizeof(ato_Keystore));
    assert(ks != NULL);
    ks->isvalid = TRUE;

    Try {
        bool isLegacy = TRUE;
        ato_Xml *xml = NULL;

        ato_lst_create(&(ks->crlist));

        if (buffer != NULL) {
            isLegacy = _patch_buffer(&str, buffer);
        }
        xml = _loadxml(ctx, str);

        ato_xml_namespace_register(ctx, xml, _ks_ns_prefix, _ks_ns_uri_auth);

        ato_xml_nodevalue(xml, NULL, _xpath_store_salt, &salt, FALSE);
        ato_xml_nodevalue(xml, NULL, _xpath_store_pbe, &pbe, FALSE);
        ato__ksprop_createX(ctx, &(ks->ksprop), properties, xml, isLegacy, salt, pbe);

        ato_xml_setnodes(ctx, xml, _xpath_creds, _xpath_cr_id, _setnode, ks);
        ATO_LOG_MSG(log, ATO_LOG_INFO, "Keystore loaded");
    } Catch (errcode) {
        ATO_LOG_MSG(log, ATO_LOG_ERR, "Keystore load failed");
    }

    ato_free(salt); ato_free(pbe);
    ato_str_free(str);
    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

#if defined(_MSC_VER)
__pragma(warning(push))
__pragma(warning(disable:4127))
#endif
static void _ks_save_crs(ato_Ctx *ctx, ato_Keystore *ks) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_ks_save_crs";
    Try {
        ato_ListNode *node = NULL;
        for (node = NULL; (node = ato_lst_next(ks->crlist, node)) != NULL; ) {
            ato_Credential *cr = ato_lst_value(node);
            ato__cr_save(ctx, cr);
        }
    } Catch_anonymous {
        ks->isvalid = FALSE;
        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_KEYSTORE_CORRUPTED, "Error while changing password - do not save keystore");
    }
}
#if defined(_MSC_VER)
__pragma(warning(pop))
#endif

void atoe_ks_save(ato_Ctx *ctx, ato_Keystore *ks, char **buffer)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "atoe_ks_save";
    ato_String *str = NULL;
    ato_Xml *xml = ato__ksprop_xml(ks->ksprop);
    ato_Pbe *pbe = ato__ksprop_pbe(ks->ksprop);
    ATO_ASSERT_ISNOTALLOCATED(buffer);
    assert(ks != NULL);

    if (!ks->isvalid) {
        Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_KEYSTORE_CORRUPTED, "Cannot save a corrupt keystore, please reload");
    }

    _ks_save_crs(ctx, ks);

    if (ato__ksprop_isPbe2(ks->ksprop)) {
        ato_xml_createnodevalue(xml, NULL, _xpath_store, _xpath_store_pbe_name, ato_valuec(ato_pbe_str(pbe), NULL));
    }
    ato_xml_savetrim(ctx, xml, &str);
    *buffer = ato_str_valueasowner(str);
    ato_str_free(str);
    if (ato__ksprop_isPbe2(ks->ksprop)) {
        char *href = strstr(*buffer, _ks_ns_uri_auth);
        if (href != NULL) {
            char *auth_offset = strstr(href, "auth");
            assert(auth_offset != NULL);
            memcpy(auth_offset, "ausk", 4);
        }
    }
}

void atoe_ks_changepwd(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd, const char *newpwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL, *newpk = NULL;
    pk = ato__ksprop_pk(ctx, ks->ksprop, pwd);
    Try {
        newpk = ato_pk_create(ctx, &newpk, newpwd, ato_ksprop_salt(ks->ksprop), ato__ksprop_pbe(ks->ksprop));
    } Catch (errcode) {
    }
    if (errcode == ATO_ERR_OK) {
        Try {
            _changepwd(ctx, ks, pk, newpk);
        } Catch (errcode) {
        }
    }
    ato__ksprop_pk(ctx, ks->ksprop, NULL);
    ato_pk_free(newpk);
    ATO_RETHROW_ONERR(errcode);
}

static int _atoe_ks_iscorrectpwd(ato_Ctx *ctx, ato_Keystore *ks, ato_PwdKdf *pk, bool throwOnFail)
{
    static const char *function = "_atoe_ks_iscorrectpwd";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    assert(ks != NULL);
    if (_countall(ks) == 0) {
        if (!_pwdpolicy_isvalid(ks, ato_pk_rpwd(pk))) {
            errcode = ATO_AKM_ERR_PWDPOLICY;
        }
    } else {
        if (!ato__cr_iscorrectpwd(ctx, ato_lst_value(ato_lst_next(ks->crlist, NULL)), pk)) {
            errcode = ATO_AKM_ERR_BADPWD;
        }
    }
    if (errcode != ATO_ERR_OK && throwOnFail) {
        Throw ATO_CTX_NEWERR(ctx, errcode, "");
        //Throw errcode;
    }
    return errcode;
}

void atoe_ks_checkcorrectpwd(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL;

    pk = ato__ksprop_pk(ctx, ks->ksprop, pwd);
    Try {
        _atoe_ks_iscorrectpwd(ctx, ks, pk, TRUE);
    } Catch (errcode) {
    }

    ato__ksprop_pk_clr(ctx, ks->ksprop);
    ATO_RETHROW_ONERR(errcode);
}

static void _atoe_ks_addcredential(ato_Ctx *ctx, ato_Keystore *ks, ato_PwdKdf *pk, bool isreplacable, ato_Credential *cr, ato_PwdKdf *crpk)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_atoe_ks_addcredential";
    ato_Log *log = NULL;
    int errcode = ATO_ERR_OK;
    ato_Credential *dstcr = NULL;
    bool foundalias = FALSE;
    void *node = NULL;
    const char *msg = NULL;
    int errcode_dstks = ATO_ERR_OK;
    const char *alias = ato_cr_alias(cr);
    ATO_CTX_FN_START(ctx);

    log = ato_ctx_log(ctx);
    assert(alias != NULL); assert(pk != NULL); assert(ks != NULL); assert(pk != NULL);


    Try {
        msg = "alias=%s (credential exists in the ks and replaceflag is false)";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, msg, alias);
        foundalias = (ato_ks_credential(ks, alias) != NULL);
        if (foundalias && !isreplacable)
            Throw ATO_AKM_ERR_ALIASEXISTS;

        msg = "alias=%s (Check the credential password is valid)";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, msg, alias);
        if (!ato__cr_iscorrectpwd(ctx, cr, crpk ? crpk : pk)) {
            Throw ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_BADPWD, strlen(alias), "Credential %s", alias);
        }

        msg = "alias=%s (Check the ks password is valid)";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, msg, alias);
        errcode_dstks = _atoe_ks_iscorrectpwd(ctx, ks, pk, FALSE);
        if (errcode_dstks == ATO_AKM_ERR_BADPWD) {
            Throw ATO_AKM_ERR_PWDDEST;
        } else if (errcode_dstks != ATO_ERR_OK) {
            Throw errcode_dstks;
        }

        msg = "alias=%s (Clone the credential node for the ks)";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, msg, alias);
        if ((node = ato__cr_clonenode(ctx, ato__ksprop_xml(ks->ksprop), cr)) == NULL)
            Throw ATO_AKM_ERR_COPYCREDENTIAL;

        msg = "alias=%s (First delete an existing credential, since we are replacing it)";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, msg, alias);
        if (foundalias) {
            if (ato_ks_delete(ks, alias) == 0)
                Throw ATO_AKM_ERR_COPYCREDENTIAL;
        }

        msg = "alias=%s (Create the credential for the ks)";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, msg, alias);
        _setnode(ctx, ks, alias, node);
        msg = "alias=%s (Find the credential for the ks)";
        if ((dstcr = ato_ks_credential(ks, alias)) == NULL)
            Throw ATO_AKM_ERR_COPYCREDENTIAL;

        msg = "alias=%s (Change the cloned credential password)";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, msg, alias);
        ato__cr_changepwd(ctx, dstcr, crpk ? crpk : pk, pk);

    } Catch (errcode) {
        ato_ks_delete(ks, alias);
        ATO_CTX_VNEWERR(ctx, errcode, strlen(alias), "alias=%s", alias);
    }

    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

static void _construct_alias(char *alias, const char *abn, const char *name1, const char *pid) {
    //TODO: Check for bad alias formats such as alphanum for name1 if device
    const char *n1p = NULL;
    if (pid == NULL || strlen(pid) == 0) {
        strcpy(alias, "ABRD:");
        n1p = name1;
    } else {
        strcpy(alias, "ABRP:");
        n1p = pid;
    }
    strcat(alias, abn);
    strcat(alias, "_");
    strcat(alias, n1p);
}

static void _atoe_ks_newcredential(ato_Ctx *ctx, ato_Keystore *ks, ato_Credential **obj, const char *abn, const char *name1, const char *name2, const char *pid)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    //static const char *function = "_atoe_ks_newcredential";
    char alias[ATO_MAX_ALIAS_LEN];
    //void *node;
    ATO_ASSERT_ISNOTALLOCATED(obj);
    assert(alias != NULL); assert(ks != NULL);

    // password policy check if empty keystore
    _construct_alias(alias, abn, name1, pid);
    if (ato_ks_credential(ks, alias) != NULL) {
        Throw ATO_AKM_ERR_ALIASEXISTS;
    }
    //node = ato__cr_clonenode(ctx, ato__ksprop_xml(ks->ksprop), cr);
    *obj = _setnode_new(ctx, ks, alias, abn, name1, name2, pid);
}
/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
int ato__ks_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__ks_deinit(void)
{
}

/*********************************************************************************/
int ato_ks_create(ato_Ctx *ctx, ato_Keystore **ks, const char *buffer, const ato_ksProperties *properties)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "ato_ks_create";
    int errcode = ATO_ERR_OK;
    Try {
        atoe_ks_create(ctx, ks, buffer, properties);
    } Catch (errcode) {
        ato_ks_free(*ks);
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    return errcode;
}

void *ato_ks_free(ato_Keystore *ks)
{
    if (ks == NULL) return NULL;

    _clearlist(ks);
    ato_lst_free(ks->crlist);
    ks->crlist = NULL;
    ks->ksprop = ato_ksprop_free(ks->ksprop);
    free(ks);
    return NULL;
}

int ato_ks_save(ato_Ctx *ctx, ato_Keystore *ks, char **buffer)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "ato_ks_save";
    int errcode = ATO_ERR_OK;
    Try {
        atoe_ks_save(ctx, ks, buffer);
    } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    return errcode;
}

const char *ato_ks_schemaversion(ato_Keystore *ks) { assert(ks != NULL); return ato_ksprop_ver(ks->ksprop); }
const char *ato_ks_salt(ato_Keystore *ks) { assert(ks != NULL); return ato_ksprop_salt(ks->ksprop); }

size_t ato_ks_count(ato_Keystore *ks, ato_eCredfilter filter)
{
    assert(ks != NULL);
    return _nrsatisfiesfilter(ks, filter);
}

ato_Credential *ato_ks_credential(ato_Keystore *ks, const char *alias)
{
    assert(ks != NULL);
    if (ato_isnullorempty(alias))
        return NULL;
    return ato_lst_value(ato_lst_find(ks->crlist, alias));
}

size_t ato_ks_credentials(ato_Keystore *ks, ato_Credential ***creds, ato_eCredfilter filter)
{
    size_t nr = 0, len = 0;

    ATO_ASSERT_ISNOTALLOCATED(creds);
    assert(creds != NULL);

    nr = _nrsatisfiesfilter(ks, filter);
    if (nr > 0) {
        ato_ListNode *node = NULL;
        *creds = calloc(nr, sizeof(void *));
        assert(*creds != NULL);
        while ((node = ato_lst_next(ks->crlist, node)) != NULL) {
            ato_Credential *cr = ato_lst_value(node);
            if (ato_cr_matchesfilter(cr, filter))
                (*creds)[len++] = cr;
        }
        assert(len == nr);
    }
    return len;
}

bool ato_ks_ismodified(ato_Keystore *ks)
{
    return ato_xml_isdirty(ato__ksprop_xml(ks->ksprop));
}

bool ato_ks_isdirty(ato_Keystore *ks)
{
    return ato_ks_ismodified(ks);
}

bool ato_ks_isvalid(ato_Keystore *ks)
{
    return ks->isvalid;
}

size_t ato_ks_delete(ato_Keystore *ks, const char *alias)
{
    size_t count = 0;
    if (alias != NULL) {
        if (_deletecr(ato_lst_find(ks->crlist, alias)))
            count = 1;
    } else {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(ks->crlist, node)) != NULL) {
            if (_deletecr(node))
                count++;
        }
    }

    return count;
}

int ato_ks_checkcorrectpwd(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "ato_ks_checkcorrectpwd";
    int errcode = ATO_ERR_OK;

    Try {
        atoe_ks_checkcorrectpwd(ctx, ks, pwd);
    } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }

    return errcode;
}

int ato_ks_iscorrectpwd(ato_Ctx *ctx, ato_Keystore *ks, bool *correct, const char *pwd)
{
    int errcode = ato_ks_checkcorrectpwd(ctx, ks, pwd);
    *correct = (errcode == ATO_ERR_OK);
    return errcode;
}

bool ato_ks_pwdpolicy_isvalid(ato_Keystore *ks, const char *pwd)
{
    return _pwdpolicy_isvalid(ks, pwd);
}

const char *ato_ks_pwdpolicy_desc(ato_Keystore *ks)
{
    ATO_IGNORE(ks);
    return "A password must contain:\n\
- no whitespace\n\
- printable characters only\n\
- at least 1 uppercase character\n\
- at least 1 lowercase character\n\
- at least 1 digit OR 1 punctuation character\n\
- at least 10 characters";
}

int ato_ks_changepwd(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd, const char *newpwd)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "ato_ks_changepwd";
    int errcode = ATO_ERR_OK;
    Try {
        atoe_ks_changepwd(ctx, ks, pwd, newpwd);
    } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    return errcode;
}

static void atoe_ks_addcredential(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd, ato_Credential *cr, const char *crpwd, bool isreplacable)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL, *crpk = NULL;

    pk = ato__ksprop_pk(ctx, ks->ksprop, pwd);
    Try {
        if (crpwd != NULL) { crpk = ato__ksprop_pk(ctx, ato__cr_properties(cr), crpwd); }
    } Catch (errcode) {
    }
    if (errcode == ATO_ERR_OK) {
        Try {
            _atoe_ks_addcredential(ctx, ks, pk, isreplacable, cr, crpk);
        } Catch (errcode) {
        }
    }
    ato__ksprop_pk_clr(ctx, ks->ksprop);
    ato__ksprop_pk_clr(ctx, ato__cr_properties(cr));
    ATO_RETHROW_ONERR(errcode);
}

int ato_ks_addcredential(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd, ato_Credential *cr, const char *crpwd, bool isreplacable)
{
    const char *function = "ato_ks_addcredential";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    Try {
        atoe_ks_addcredential(ctx, ks, pwd, cr, crpwd, isreplacable);
    } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    return errcode;
}

ato_ksProperties *ato_ks_properties(ato_Keystore *ks) {
    assert(ks != NULL);
    return ks->ksprop;
}

void atoe_ks_newcredential(ato_Ctx *ctx, ato_Keystore *ks, ato_Credential **obj, const char *pwd, const char *abn, const char *name1, const char *name2, const char *pid) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL;
    ATO_ASSERT_ISNOTALLOCATED(obj);
    pk = ato__ksprop_pk(ctx, ks->ksprop, pwd);
    Try {
        _atoe_ks_iscorrectpwd(ctx, ks, pk, TRUE);
        _atoe_ks_newcredential(ctx, ks, obj, abn, name1, name2, pid);
    } Catch (errcode) {
    }
    ato__ksprop_pk_clr(ctx, ks->ksprop);
    ATO_RETHROW_ONERR(errcode);
}


int ato_ks_newcredential(ato_Ctx *ctx, ato_Keystore *ks, ato_Credential **obj, const char *pwd, const char *abn, const char *name1, const char *name2, const char *pid) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "ato_ks_newcredential";
    int errcode = ATO_ERR_OK;
    Try {
        atoe_ks_newcredential(ctx, ks, obj, pwd, abn, name1, name2, pid);
    } Catch (errcode) {
        errcode = ato__akmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    return errcode;
}
