#ifndef __ATO_TYPES_H__
#define __ATO_TYPES_H__

#if defined(_WIN32) || defined(_WIN64) || defined(WIN32) || defined(WIN64)
#define ATO_WINXX
#endif

#if defined(_WIN64)
        #define PRINTF_SIZET "I"
#elif defined(_WIN32)
        #define PRINTF_SIZET ""
#else
        #define PRINTF_SIZET "z"
#endif
// e.g          [size_t size = 0; printf("%" PRINTF_SIZET "u", size);]
// instead of [size_t size = 0; printf("%zu", size);] which is not portable (i.e. to MSVC)


#if defined(_MSC_VER)
//#define DEBUG_VISUAL_LEAK_DETECTOR
#ifdef DEBUG_VISUAL_LEAK_DETECTOR
//Used to detect memory leaks in Visual Studio environments: http://vld.codeplex.com/
//To work with the existing project files, install Visual Leak Detector to the default location
//(C:\Program Files\Visual Leak Detector).
//Note: this is set to only detect leaks within DLLs using this. If a dependent DLL allocates memory
//on behalf of a client DLL/application but the client DLL/app does not release it, this will
//not be detected. Also memory corruptions may destabalise VLD.
//For a more robust analysis a comprehensive tool such Valgrind on Linux/Ubuntu is strongly recommended.
//Note2: on Windows do not directly release memory in a client DLL/app that has been allocated within
//a dependent DLL. Instead call a relevant free method provided by the dependent DLL (if one is not
//provided, don't use the method and/or the DLL that cause the allocation).
//For more information see http://msdn.microsoft.com/en-us/library/ms235460.aspx.
#include <vld.h>
#endif
#endif

#if defined(_MSC_VER)
// We need to cast to non-const due to BUG in MSVC.
// See http://connect.microsoft.com/VisualStudio/feedback/details/101661/warning-c4090-is-incorrectly-issued-when-using-visual-c-net-2003
#define BADCAST2NC_MSVSBUG_C4090 (void *)
#else
#define BADCAST2NC_MSVSBUG_C4090
#endif
#define ATO_BADCAST2NC BADCAST2NC_MSVSBUG_C4090

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE (!FALSE)
#endif
#ifndef __cplusplus
#ifndef bool
typedef int bool;
#endif
#endif

/** @addtogroup atobase_api_public_types

@{
*/

/**
Parameter semantics when passing information to some API methods
*/
typedef enum {
        ATO_PARAMACTION_DEFAULT = 0,  /**< Action is up to individual method */
        ATO_PARAMACTION_COPY, /**< Take a deep copy of the parameter */
        ATO_PARAMACTION_CONST,  /**< Reference the parameter but treat as const so don't free */
        ATO_PARAMACTION_FREE  /**< Reference the parameter but take on ownership so free when finished with it */
} ato_eParamAction;

/**
Loglevels
*/
typedef enum {
        ATO_LOG_OFF = 1,  /**< Do not log */
        ATO_LOG_FATAL, /**< Typically this will be an assertion for invalid arguments or out of memory */
        ATO_LOG_ERR,  /**< An infrastructure type <i>error</i> has occured, such as a network error or no configuration */
        ATO_LOG_WARN, /**< Usually a business level <i>error</i> such as wrong password */
        ATO_LOG_INFO, /**< Information usually indicating a method succeeded */
        ATO_LOG_DEBUG, /**< Additional information, useful where <i>errors</i> have occurred */
        ATO_LOG_TRACE, /**< Trace the entry and exit calls of methods (exits do not occur on <i>errors</i>) */
        ATO_LOG_ALL /**< Log everything */
} ato_eLoglevel;


/**
The thread local context object
*/
typedef struct _ato_Ctx ato_Ctx;

/** @} */

/** @defgroup atobase_api_public_types_modules Module defines
@ingroup atobase_api_public_types
Each module within the library is individually defined by a name and id.
These can be used to target logging for particular modules and to identify the source of errors.
When setting logging, the MODULEID values can be bitwised or-ed (see ato_setloglevel()).

Note that not all modules are neccessarily exposed via a public API, even though they appear here.
@{
*/
#define ATO_BASE_LIBRARY "atobase" /**< The name of the library. */

#define ATO_BASE_MODULE_LIB "lib" /**< The lib module name. */
#define ATO_BASE_MODULE_ERRDEF "errdef" /**< The errdef module name. */
#define ATO_BASE_MODULE_ERR "err" /**< The error module name. */
//#define ATO_BASE_MODULE_NET "net" /**< The network module name. */
#define ATO_BASE_MODULE_XML "xml" /**< The xml module name. */
#define ATO_BASE_MODULE_CTX "ctx" /**< The ctx module name. */
#define ATO_BASE_MODULE_LOG "log" /**< The log module name. */
#define ATO_BASE_MODULE_UTIL "util" /**< The util module name. */
#define ATO_BASE_MODULE_LST "lst" /**< The lst (container) module name. */
#define ATO_BASE_MODULE_STR "str" /**< The str module name. */
#define ATO_BASE_MODULE_CRYPTO "crypto" /**< The crypto module name. */
#define ATO_BASE_MODULE_ERRFN "errfn" /**< The errfn module name. */
#define ATO_BASE_MODULE_CFG "cfg" /**< The cfg module name. */
#define ATO_BASE_MODULE_ITER "iter" /**< The iter module name. */
#define ATO_BASE_MODULE_ERRSOAP "errsoap" /**< The errsoap module name. */
//#define ATO_BASE_MODULE_NETD "netdata" /**< The netdata module name. */
#define ATO_BASE_MODULE_SI "si" /**< The softwareinfo module name. */
#define ATO_BASE_MODULE_PBE "pbe" /**< The PasswordBasedEncryption support module. */
#define ATO_BASE_MODULE_PK "pk" /**< The pwd/kdf support module. */

#define ATO_BASE_MODULEID_LIB     0x00000001UL /**< The lib module id. */
#define ATO_BASE_MODULEID_ERRDEF  0x00000002UL /**< The errdef module id. */
#define ATO_BASE_MODULEID_ERR     0x00000004UL /**< The error module id. */
//#define ATO_BASE_MODULEID_NET     0x00000008UL /**< The network module id. */
#define ATO_BASE_MODULEID_XML     0x00000010UL /**< The xml module id. */
#define ATO_BASE_MODULEID_CTX     0x00000020UL /**< The ctx module id. */
#define ATO_BASE_MODULEID_LOG     0x00000040UL /**< The log module id. */
#define ATO_BASE_MODULEID_UTIL    0x00000080UL /**< The util module id. */
#define ATO_BASE_MODULEID_LST     0x00000100UL /**< The lst (container) module id. */
#define ATO_BASE_MODULEID_STR     0x00000200UL /**< The str module id. */
#define ATO_BASE_MODULEID_CRYPTO  0x00000400UL /**< The crypto module id. */
#define ATO_BASE_MODULEID_ERRFN   0x00000800UL /**< The errfn module id. */
#define ATO_BASE_MODULEID_CFG     0x00001000UL /**< The cfg module id. */
#define ATO_BASE_MODULEID_ITER    0x00002000UL /**< The iter module id. */
#define ATO_BASE_MODULEID_ERRSOAP 0x00004000UL /**< The errsoap module id. */
//#define ATO_BASE_MODULEID_NETD    0x00008000UL /**< The netdata module id. */
#define ATO_BASE_MODULEID_SI      0x00010000UL /**< The softwareinfo module id. */
#define ATO_BASE_MODULEID_PBE     0x00020000UL /**< The PasswordBasedEncryption support module id. */
#define ATO_BASE_MODULEID_PK      0x00040000UL /**< The pwd/kdf support module id. */

/** @} */

/** @defgroup atobase_api_public_types_otherdefines Other defines
@ingroup atobase_api_public_types
@{
*/

/**
The ATO_BASE_INIT_... defines are used to control the initialisation that takes place.

Currently only ATO_BASE_INIT_ALL is defined.
*/
#define ATO_BASE_INIT_ALL 0xffff

/**
Ignore unused arguments in a function.

Useful during development or for placeholder arguments.
*/
#define ATO_IGNORE(v) ((void) &(v))

#define ATO_BASE_CFGN_LOG_LEVEL  ATO_BASE_LIBRARY"."ATO_BASE_MODULE_LOG".""level"
#define ATO_BASE_CFGN_LOG_INDENT ATO_BASE_LIBRARY"."ATO_BASE_MODULE_LOG".""indent"
#define ATO_BASE_CFGN_LOG_TYPE ATO_BASE_LIBRARY"."ATO_BASE_MODULE_LOG".""type" //path,stderr,stdout,none
#define ATO_BASE_CFGN_LOG_PATH ATO_BASE_LIBRARY"."ATO_BASE_MODULE_LOG".""path"

/** @} */

#ifdef __cplusplus
extern "C" {
#endif

#ifdef __cplusplus
}
#endif

//@ingroup atobase_api_public_types
/** @defgroup atobase_api_public_types_errcodes Error codes
@ingroup atobase
Use ato_err_inner() to retrieve nested errors and their details.
For SOAP related errors see @ref ato_ErrSoap.

@{
*/

#define ATO_ERR_OK 0 /**< No error. */
#define ATO_ERR_MIN 11000 /**< The mimimum error number. */
#define ATO_ERR_UNKNOWN ATO_ERR_MIN /**< Unknown error - catch-all or unanticipated error not covered by other error codes. */
#define ATO_ERR_CFGNOMGR (ATO_ERR_MIN+1) /**< Deprecated.   */
#define ATO_ERR_MEMORY (ATO_ERR_MIN+2) /**< Memory allocaton error - currently this results in an assert. */
#define ATO_ERR_CFGNOTFOUND (ATO_ERR_MIN+3) /**< Configuration setting not found. */
#define ATO_ERR_INIT (ATO_ERR_MIN+4) /**< Failing to initialise library. */
#define ATO_ERR_VERSION (ATO_ERR_MIN+5) /**< Version mismatch between client and library. */
#define ATO_ERR_VERSION_DEP (ATO_ERR_MIN+6) /**< Version mismatch with dependant library. */
#define ATO_ERR_B64 (ATO_ERR_MIN+7) /**< Failed to decode from or encode to base64. */
#define ATO_ERR_NET (ATO_ERR_MIN+8) /**< Network communication error. */
#define ATO_ERR_FILEOPEN (ATO_ERR_MIN+9) /**< Cannot open file. */
#define ATO_ERR_FILESTAT (ATO_ERR_MIN+10) /**< Cannot determine file size. */
#define ATO_ERR_FILEREAD (ATO_ERR_MIN+11) /**< Cannot read file */
#define ATO_ERR_FILEWRITE (ATO_ERR_MIN+12) /**< Cannot write file */
#define ATO_ERR_P7 (ATO_ERR_MIN+13) /**< Failed to access P7 (certificate). */
#define ATO_ERR_P8 (ATO_ERR_MIN+14) /**< Failed to access P8 (private key). */
#define ATO_ERR_DER (ATO_ERR_MIN+15) /**< Failed to DER encode value. */
#define ATO_ERR_DUPLICATE_ERRDEF (ATO_ERR_MIN+16) /**< Internal/SDK error - duplicate error definition entry detected. */
#define ATO_ERR_DUPLICATE_ERRHANDLER (ATO_ERR_MIN+17) /**< Failed to add error handler for SDK - possible duplicate. */
#define ATO_ERR_XMLPARSE (ATO_ERR_MIN+18) /**< Invalid XML. */
#define ATO_ERR_XMLROOT (ATO_ERR_MIN+19) /**< Failed to find or load root node. */
#define ATO_ERR_CFGDUPLICATE (ATO_ERR_MIN+20) /**< Possible duplicate confuration entry. */
#define ATO_ERR_XMLNODENOTFOUND (ATO_ERR_MIN+21) /**< Failed to find or load expected XML node. */
#define ATO_ERR_NET_TIMEOUT (ATO_ERR_MIN+22) /**< Network timeout connected to remote services. */
#define ATO_ERR_XML (ATO_ERR_MIN+23) /**< Error processing XML. */
#define ATO_ERR_NET_SOAP (ATO_ERR_MIN+24) /**< Network SOAP error returned by remote client. */
#define ATO_ERR_HASH (ATO_ERR_MIN+25) /**< Failed to generate crytographic hash. */
#define ATO_ERR_XMLXPCONTEXT (ATO_ERR_MIN+26) /**< Failed to set XML Xpath context. */
#define ATO_ERR_INVALIDSOFTWAREINFO (ATO_ERR_MIN+27) /**< Invalid or missing Software Info. */
#define ATO_ERR_CRYPTO (ATO_ERR_MIN+28) /**< General error from crypto routines - see message. */
#define ATO_ERR_SDK_EXPIRED (ATO_ERR_MIN+29) /**< SDK expiry date has been reached; please get an updated SDK. */
#define ATO_ERR_INVALID_ARGS (ATO_ERR_MIN+30) /**< Usually because of NULL arguments */
#define ATO_ERR_MAX (ATO_ERR_MIN+300) /**< The maximum error number. */

/*! @} */

#endif /*__ATO_TYPES_H__*/
