#ifndef __ATO_UTIL_H__
#define __ATO_UTIL_H__

#include <stddef.h>

/** @defgroup atobase_api_util Util
@ingroup atobase_api_public

General utility methods used in the SDK codebase. They are available to use by application developers
should they so wish.

Note: in API comments elsewhere, the terms "string" generally refers to a '\0' terminated char array (C string),
whereas "ato string" refers the @ref ato_String object.

@{
*/

#ifdef __cplusplus
extern "C" {
#endif

/**
Accepts a Base64 array and returns the decoded (possibly binary) data and data length in an ato_String.

To access content, use ato_str_value() and ato_str_len().

@param ctx the current thread local Context
@param data the address of ato_String buffer for the decoded content - caller must free using ato_str_free().
@param b64 the encoded base64 array to decode.
@param len the length of the b64 array

@return ATO_ERR_OK or an error.
*/
ATO_EXPORT int ato_base64decode_array(ato_Ctx *ctx, ato_String **data, const char *b64, size_t len);

/**
Accepts data and returns the Base64 encoded data as an ato_String.

To access content, use ato_str_value() and ato_str_len().

@param ctx the current thread local Context
@param b64 the address of ato_String buffer for the encoded content - caller must free using ato_str_free().
@param data the content - possibly binary
@param len the length of the content

@return ATO_ERR_OK or an error.
*/
ATO_EXPORT int ato_base64encode_array(ato_Ctx *ctx, ato_String **b64, const char *data, size_t len);

/**
Generate a null terminated (pseudo) guid.

Make sure the guidstr buff is big enough - 50 chars will do.

Not robust (guaranteed to be really GUID) but suitable for testing purposes.

@param ctx the current thread local Context
@param guidstr the address of the buffer to write the guid to.

@return guidstr or NULL if it failed.
*/
ATO_EXPORT char *ato_new_pseudo_guid(ato_Ctx *ctx, char *guidstr);

#ifdef __cplusplus
}
#endif

/** @} */

#endif /*__ATO_UTIL_H__*/
