/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atostr.h"
#include "atoerrdef.h"
#include "atoutil.h"

#include "atoerr.h"

extern void ato__errh_free(ato_Err *err);

/*********************************************************************************/
static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_ERR;
static unsigned long _moduleid = ATO_BASE_MODULEID_ERR;

static int _loglevel = ATO_LOG_WARN;

/*********************************************************************************/

struct _ato_Err {
    char *library;
    char *module;
    const char *function;
    int line;
    bool isprocessed;
    bool isdeleted;

    int code;
    const char *codename;
    const char *codemsg;
    ato_eErrSeverity severity;
    char *msg;
    void *customobj;
    ato_Err *inner; /* Nested errors. This is "owned" by a module, which is responsible for freeing it. */
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}
/*********************************************************************************/
int ato__err_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__err_deinit(void)
{
}

/*********************************************************************************/
static void _set(ato_Err *err, const char *function, int line, int code, ato_eErrSeverity severity, const char *msg, void *customobj, ato_Err *inner);

/*********************************************************************************/
void ato__err_create(ato_Err **obj, const char *library, const char *module, const char *function, int line, int code, ato_eErrSeverity severity, const char *msg, void *customobj, ato_Err *inner)
{
    ato_Err *err = NULL;
    ATO_ASSERT_ISNOTALLOCATED(obj);

    err = calloc(1, sizeof(ato_Err));
    assert(err != NULL);

    err->library = ato_strdup(library, 0);
    err->module = ato_strdup(module, 0);
     _set(err, function, line, code, severity, msg, customobj, inner);

    *obj = err;
}

void ato__err_free(ato_Err *err)
{
    if (err == NULL) return;

    ato__err_free(err->inner);
    err->inner = NULL;
    err->isdeleted = TRUE;
    if (err->customobj != NULL) {
        ato__errh_free(err);
        //free(err->customobj);
        err->customobj = NULL;
    }
    if (err->library != NULL) { free(err->library); err->library = NULL; }
    if (err->module != NULL) { free(err->module); err->module = NULL; }
    if (err->msg != NULL) { free(err->msg); err->msg = NULL; }
    free(err);
}

/*********************************************************************************/
bool ato_err_isdeleted(ato_Err *err) { assert(err != NULL); return err->isdeleted; }
bool ato_err_isprocessed(ato_Err *err) { assert(err != NULL); return err->isprocessed; }
void ato_err_setisprocessed(ato_Err *err, bool isprocessed) { if (err != NULL) err->isprocessed = isprocessed; }

bool ato_err_iserror(ato_Err *err) { return err == NULL ? FALSE : err->code != ATO_ERR_OK; }
int ato_err_code(ato_Err *err) { assert(err != NULL); return err->code; }
const char *ato_err_codename(ato_Err *err) { assert(err != NULL); return err->codename; }
const char *ato_err_codemsg(ato_Err *err) { assert(err != NULL); return err->codemsg; }
ato_eErrSeverity ato_err_severity(ato_Err *err) { assert(err != NULL); return err->severity; }
const char *ato_err_library(ato_Err *err) { assert(err != NULL); return err->library; }
const char *ato_err_module(ato_Err *err) { assert(err != NULL); return err->module; }
const char *ato_err_function(ato_Err *err) { assert(err != NULL); return err->function; }
int ato_err_line(ato_Err *err) { assert(err != NULL); return err->line; }
const char *ato_err_msg(ato_Err *err) { assert(err != NULL); return err->msg; }
void *ato_err_custom(ato_Err *err) { assert(err != NULL); return err->customobj; }
ato_Err *ato_err_inner(ato_Err *err) { assert(err != NULL); return err->inner; }

static void _set(ato_Err *err, const char *function, int line, int code, ato_eErrSeverity severity, const char *msg, void *customobj, ato_Err *inner)
{
    assert(msg != NULL);
    err->msg = ato_strdup(msg, 0);
    assert(err->msg != NULL);
    err->code = code;
    err->codename = ato_errdef_name(err->code);
    err->codemsg = ato_errdef_msg(err->code);
    err->severity = severity;
    err->customobj = customobj;
    err->inner = inner;
    err->function = function;
    err->line = line;
}

/*********************************************************************************/

