/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atostr.h"
#include "atoutil.h"
#include "atolst.h"
#include "atoerrdef.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_ERRDEF;
static unsigned long _moduleid = ATO_BASE_MODULEID_ERRDEF;

static int _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
typedef struct _ato_ErrDef ato_ErrDef;
struct _ato_ErrDef {
    int code;
    const char *name;
    const char *msg;
    const char *library;
};

static ato_List *_errdefs;

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
void ato__errdef_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return;
    invoked = TRUE;

    ato_lst_create(&_errdefs);
    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
}
void ato__errdef_deinit(void)
{
    ato_ListNode *node = NULL;
    if (!_errdefs) return;

    while ((node = ato_lst_next(_errdefs, node)) != NULL) {
        ato_free(ato_lst_value(node));
    }
    ato_lst_free(_errdefs);
}

/*********************************************************************************/
void ato_errdef_add(int errcode, const char *library, const char *name, const char *msg)
{
    char key[101];
    ato_ErrDef *errdef = NULL;
    assert(_errdefs != NULL);
    assert(name != NULL);

    errdef = calloc(1, sizeof(ato_ErrDef));
    errdef->code = errcode;
    errdef->name = name;
    errdef->msg = msg;
    errdef->library = library;

    sprintf(key, "%d", errcode);
    if (ato_lst_add(_errdefs, key, errdef) == NULL) {
        free(errdef);
        printf("%s\n", name);
        errcode = ATO_ERR_DUPLICATE_ERRDEF;
        assert(errcode != ATO_ERR_DUPLICATE_ERRDEF);
    }
}

static ato_ErrDef *_errdef_item(int errcode)
{
    char key[101];
    sprintf(key, "%d", errcode);
    return ato_lst_value(ato_lst_find(_errdefs, key));
}

const char *ato_errdef_name(int errcode)
{
    ato_ErrDef *errdef = _errdef_item(errcode);
    if (errdef == NULL)
        return "";
    return errdef->name;
}
const char *ato_errdef_msg(int errcode)
{
    ato_ErrDef *errdef = _errdef_item(errcode);
    if (errdef == NULL)
        return "";
    return errdef->msg;
}
const char *ato_errdef_library(int errcode)
{
    ato_ErrDef *errdef = _errdef_item(errcode);
    if (errdef == NULL)
        return "";
    return errdef->library;
}

/*********************************************************************************/
