#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atolst.h"
#include "atoiter.h"
#include "atostr.h"
#include "atoutil.h"
#include "atolog.h"
#include "atoerr.h"
#include "atoctx.h"
#include "atoerrh.h"

#include "atoerrsoap.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_ERRSOAP;
static unsigned long _moduleid = ATO_BASE_MODULEID_ERRSOAP;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
struct _ato_ErrSoapCode {
    char *value;
    char *ns;
};

struct _ato_ErrSoap {
    const char *iid;
    char *detail;
    char *node;
    char *reason;
    ato_ErrSoapCode **codes;
    size_t codecount;
    size_t index;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void _check(ato_ErrSoap *err)
{
    assert(err != NULL); ATO_IGNORE(err);
    assert(ato_streq(err->iid, ATO_ERRSOAP_IID));
}

/*********************************************************************************/
int ato__errsoap_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__errsoap_deinit(void)
{
}
/*********************************************************************************/

static void _ato_free(ato_ErrSoap *err)
{
    size_t i = 0;
    if (err == NULL) return;

    if (err->detail != NULL) free(err->detail);
    if (err->node != NULL) free(err->node);
    if (err->reason != NULL) free(err->reason);
    if (err->codes) {
        for (i = 0; i < err->codecount; i++) {
            if (err->codes[i]) {
                if (err->codes[i]->value != NULL) free(err->codes[i]->value);
                if (err->codes[i]->ns != NULL) free(err->codes[i]->ns);
                free(err->codes[i]);
            }
        }
        free(err->codes);
    }
}

void ato_errsoap_free(ato_ErrSoap *err)
{
    if (err == NULL) return;

    _check(err);
    _ato_free(err);
    free(err);
}

void ato_errsoap_createassign(ato_Ctx *ctx, ato_ErrSoap **obj,
    char *detail, char *node, char *reason, size_t codecount)
{
    //struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "ato_errsoap_createassign";
    int errcode = ATO_ERR_OK;
    ato_ErrSoap *err = NULL;
    assert(ctx != NULL);
    ato_ctx_err_clear(ctx);
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = err = calloc(1, sizeof(ato_ErrSoap));
    assert(err != NULL);

    err->iid = ATO_ERRSOAP_IID;
    err->detail = detail;
    err->node = node;
    err->reason = reason;
    err->codecount = codecount;
    err->codes = calloc(codecount+1, sizeof(void *));
    assert(err->codes);

    ATO_CTX_FN_END(ctx, errcode);
}

void ato_errsoap_code_addassign(ato_Ctx *ctx, ato_ErrSoap *err, char *value, char *ns, size_t index)
{
    ato_ErrSoapCode *code = NULL;
    ATO_IGNORE(ctx);
    _check(err);
    assert(index < err->codecount);
    code = calloc(1, sizeof(ato_ErrSoapCode));
    code->value = value;
    code->ns = ns;
    err->codes[index] = code;
}

const char *ato_errsoap_iid(ato_ErrSoap *err) { _check(err); return err->iid; }

/*********************************************************************************/
bool ato_errsoap_issoaperr(ato_ErrSoap *err)
{
    if (!err) return FALSE;
    return ato_streq(err->iid, ATO_ERRSOAP_IID);
}

const char *ato_errsoap_detail(ato_ErrSoap *err) { _check(err); return err->detail; }
const char *ato_errsoap_node(ato_ErrSoap *err) { _check(err); return err->node; }
const char *ato_errsoap_reason(ato_ErrSoap *err) { _check(err); return err->reason; }

size_t ato_errsoap_code_count(ato_ErrSoap *err) { _check(err); return err->codecount; }

void ato_errsoap_code_iterator(ato_ErrSoap *err, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    _check(err);
    *iter = ato_iter_create_arr((void **)err->codes, err->codecount);
}
ato_ErrSoapCode *ato_errsoap_code_firstv(ato_ErrSoap *err)
{
    _check(err);
    err->index = 0;
    if (err->index >= err->codecount)
        return NULL;
    return err->codes[err->index++];
}
ato_ErrSoapCode *ato_errsoap_code_nextv(ato_ErrSoap *err)
{
    _check(err);
    if (err->index >= err->codecount)
        return NULL;
    return err->codes[err->index++];
}

const char *ato_errsoap_code_value(ato_ErrSoapCode *code)
{
    assert(code != NULL);
    return code->value;
}

const char *ato_errsoap_code_ns(ato_ErrSoapCode *code)
{
    assert(code != NULL);
    return code->ns;
}
