/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <stdarg.h>

#ifdef __APPLE__
#include "TargetConditionals.h"
#if TARGET_OS_IPHONE && TARGET_IPHONE_SIMULATOR
#define TARGET_OS_IOS 1
#elif TARGET_OS_IPHONE
#define TARGET_OS_IOS 1
#else
#define TARGET_OS_OSX 1
#endif
#endif

#if TARGET_OS_IOS
#include "akversion_ios.h"
#else
#include "akversion.h"
#endif

#include "atointernal.h"
#include "atotypes.h"
#include "atosi.h"
#include "atostr.h"
#include "atopbe.h"
#include "atopk.h"
#include "atoerr.h"
#include "atolog.h"
#include "atoutil.h"
#include "atolst.h"
#include "atoiter.h"
#include "atoerrdef.h"
#include "atoctx.h"
#include "atobcrypto.h"
#include "atoxml.h"
#include "atoerrh.h"
#include "atoarg.h"

#include "atolib.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_LIB;
static unsigned long _moduleid = ATO_BASE_MODULEID_LIB;

static int _loglevel = ATO_LOG_WARN;

// Expires on or after the 1st day of this month. Set to 0 to disabled expiry.
static int _sdk_expiry_date = 0;//(2015 - 1900) * 12 + 7; // (year - 1900) * 12 + month;

static bool _ato_inited = FALSE;

static ato_List *_vars = NULL;
/*********************************************************************************/
struct _ato_LibModule {
        const char *library;
        const char *module;
        unsigned long moduleid;
        ato_eLoglevel level;
        ato_setloglevelfn *setloglevelfn;
        bool filedump;
        ato_setlogfiledumpfn *setlogfiledumpfn;
};

static ato_List *_libmodules = NULL;

/*********************************************************************************/
static void _load_errdefs(void);

/*********************************************************************************/
const char *ato_base_version(void) { return ATOBASE_VERSION; }

static void _setloglevel(ato_eLoglevel level)
{
        _loglevel = level;
}

//ATO_EXPORT void ato_arg_add(const char *name, bool required, const char *defvalue, const char *allowedvalues, const char *help);

static void s_init_lib_properties(void) {
    ato_arg_add(ATO_BASE_CFGN_LOG_LEVEL,  FALSE, "d", "off|fatal|err|warn|info|debug|trace|all", "The log level - only need first letter");
    ato_arg_add(ATO_BASE_CFGN_LOG_INDENT, FALSE, "2", NULL, "The log indent");
    ato_arg_add(ATO_BASE_CFGN_LOG_TYPE,   FALSE, "none", "none|stdout|stderr|path", "Set the log type");
    ato_arg_add(ATO_BASE_CFGN_LOG_PATH,   FALSE, "messages.log", NULL, "Set the log path for log type of path");
}

int ato__baselib_init(void)
{
        static bool invoked = FALSE;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

//        ato_arg_init(); // need to do this on windows/msys to force the file to be exported ????????

        ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);

    if (_sdk_expiry_date > 0) {
        time_t now = 0;
        struct tm tm_now;
        int now_date = 0;
            time(&now);
            ato_localtime(&now, &tm_now);
        now_date = tm_now.tm_year * 12 + (tm_now.tm_mon + 1);
        if (now_date >= _sdk_expiry_date) { return ATO_ERR_SDK_EXPIRED; }
    }
        return ATO_ERR_OK;
}
void ato__baselib_deinit(void)
{
        static bool invoked = FALSE;
        if (invoked) return;
        invoked = TRUE;
}
/*********************************************************************************/
bool ato_base_isinited(void)
{
        return _ato_inited;
}

int ato_base_init(ato_Ctx *ctx, unsigned short flag)
{
        const char *function = "ato_base_init";
        int errcode = ATO_ERR_OK;
        static bool invoked = FALSE;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        assert(flag != 0); ATO_IGNORE(flag);
        assert(ctx != NULL);

        ato_lst_create(&_libmodules);
        ato_lst_create(&_vars);

        ato__errdef_init();
        _load_errdefs();

        if (errcode == ATO_ERR_OK)
                errcode = ato__baselib_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__iter_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__lst_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__str_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__util_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__err_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__errh_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__log_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__xml_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__ctx_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__crypto_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__si_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__pbe_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__pk_init();

        s_init_lib_properties();

        if (errcode == ATO_ERR_OK)
                if (!ato__si_isvalid())
                        errcode = ATO_ERR_INVALIDSOFTWAREINFO;

        if (errcode != ATO_ERR_OK)
                ATO_CTX_NEWERR(ctx, errcode, "Failed to initialise BASE library");
    _ato_inited = (errcode == ATO_ERR_OK);
        return errcode;
}

void ato_base_deinit(void)
{
        static bool invoked = FALSE;
        ato_ListNode *node = NULL;
        if (invoked) return;
        invoked = TRUE;

        ato__pk_deinit();
        ato__pbe_deinit();
        ato__crypto_deinit();
        ato__ctx_deinit();
        ato__xml_deinit();
        ato__log_deinit();
        ato__errh_deinit();
        ato__err_deinit();
        ato__util_deinit();
        ato__str_deinit();
        ato__lst_deinit();
        ato__iter_deinit();
        ato__baselib_deinit();
        ato__errdef_deinit();

        if (_libmodules){
           while ((node = ato_lst_next(_libmodules, node)) != NULL) {
                   ato_free(ato_lst_value(node));
           }
        }
        ato_lst_free(_libmodules);

        if (_vars){
           while ((node = ato_lst_next(_vars, node)) != NULL) {
                   ato_free(ato_lst_value(node));
           }
        }
        ato_lst_free(_vars);

        ato__si_deinit();
}

/*********************************************************************************/
void ato_base_var_set(const char *name, const char *value)
{
        ato_ListNode *node = ato_lst_find(_vars, name);
        if (node != NULL) {
                char *v = ato_lst_value(node);
                ato_free(v);
                ato_lst_setvalue(node, ato_strdup(value, 0));
        } else {
                ato_lst_add(_vars, name, ato_strdup(value, 0));
        }
}

bool ato_base_var(const char *name, const char **value)
{
        ato_ListNode *node = ato_lst_find(_vars, name);
        if (node != NULL) {
                if (value) {
                        ATO_ASSERT_ISNOTALLOCATED(value);
                        *value = ato_lst_value(node);
                }
                return TRUE;
        }
        return FALSE;
}
/*********************************************************************************/
static char *_construct_logkey(const char *library, const char *module)
{
        return ato_strcat(library, ".", module);
}

ato_eLoglevel ato_loglevel(const char *library, const char *module)
{
        ato_eLoglevel level = ATO_LOG_WARN;
        ato_ListNode *node = NULL;
        char *key = NULL;
        assert(library != NULL); assert(module != NULL);

        key = _construct_logkey(library, module);
        node = ato_lst_find(_libmodules, key);
        if (node != NULL) {
                ato_LibModule *moddefn = ato_lst_value(node);
                level = moddefn->level;
        }
        free(key);
        return level;
}

bool ato_logfiledump(const char *library, const char *module)
{
        bool filedump = FALSE;
        ato_ListNode *node = NULL;
        char *key = NULL;
        assert(library != NULL); assert(module != NULL);

        key = _construct_logkey(library, module);
        node = ato_lst_find(_libmodules, key);
        if (node != NULL) {
                ato_LibModule *moddefn = ato_lst_value(node);
                filedump = moddefn->filedump;
        }
        free(key);
        return filedump;
}

static void _callmodulelogfiledump(ato_LibModule *moddefn, bool filedump)
{
        if (moddefn != NULL) {
                moddefn->filedump = filedump;
                if (moddefn->setlogfiledumpfn != NULL)
                        moddefn->setlogfiledumpfn(filedump);
        }
}

void ato_setlogfiledump(const char *library, unsigned long moduleidmask, bool filedump)
{
        ato_ListNode *node = NULL;
        ato_LibModule *moddefn = NULL;
        while ((node = ato_lst_next(_libmodules, node)) != NULL) {
                moddefn = ato_lst_value(node);
                if (moddefn == NULL)
                        continue;
                if (library == NULL) {
                        _callmodulelogfiledump(moddefn, filedump);
                        continue;
                }
                if (!ato_streq(library, moddefn->library))
                        continue;
                if (moduleidmask == 0) {
                        _callmodulelogfiledump(moddefn, filedump);
                        continue;
                }
                if (moddefn->moduleid & moduleidmask) {
                        _callmodulelogfiledump(moddefn, filedump);
                        continue;
                }
        }
}

static void _callmoduleloglevel(ato_LibModule *moddefn, ato_eLoglevel level)
{
        if (moddefn != NULL) {
                moddefn->level = level;
                if (moddefn->setloglevelfn != NULL)
                        moddefn->setloglevelfn(level);
        }
}

void ato_setloglevel(const char *library, unsigned long moduleidmask, ato_eLoglevel level)
{
        ato_ListNode *node = NULL;
        ato_LibModule *moddefn = NULL;
        while ((node = ato_lst_next(_libmodules, node)) != NULL) {
                moddefn = ato_lst_value(node);
                if (moddefn == NULL)
                        continue;
                if (library == NULL) {
                        _callmoduleloglevel(moddefn, level);
                        continue;
                }
                if (!ato_streq(library, moddefn->library))
                        continue;
                if (moduleidmask == 0L) {
                        _callmoduleloglevel(moddefn, level);
                        continue;
                }
                if (moddefn->moduleid & moduleidmask) {
                        _callmoduleloglevel(moddefn, level);
                        continue;
                }
        }
}

/*
void ato_setloglevel2(const char *library, const char *module, ato_eLoglevel level)
{
        ato_ListNode *node = NULL;
        _ato_LibModule *moddefn = NULL;
        while ((node = ato_lst_next(_libmodules, node)) != NULL) {
                moddefn = ato_lst_value(node);
                if (moddefn == NULL)
                        continue;
                if (library != NULL && !ato_streq(library, moddefn->library))
                        continue;
                if (library != NULL && module != NULL && !ato_streq(module, moddefn->module))
                        continue;
                _callmoduleloglevel(moddefn, level);
        }
}
*/

ato_LibModule *ato_initfnloglevel(const char *library, const char *module, unsigned long moduleid, ato_eLoglevel level, ato_setloglevelfn setloglevelfn)
{
        ato_LibModule *moddefn = NULL;
        ato_ListNode *node = NULL;
        char *key = NULL;
        assert(library != NULL); assert(module != NULL);

        key = _construct_logkey(library, module);
        node = ato_lst_find(_libmodules, key);
        if (node != NULL) {
                moddefn = ato_lst_value(node);
        } else {
                moddefn = calloc(1, sizeof(ato_LibModule));
                assert(module != NULL);
                moddefn->library = library;
                moddefn->module = module;
                moddefn->moduleid = moduleid;
                ato_lst_add(_libmodules, key, moddefn);
        }
        moddefn->level = level;
        moddefn->setloglevelfn = setloglevelfn;
        free(key);
        return moddefn;
}

void ato_initfnlogfiledump(ato_LibModule *moddefn, bool filedump, ato_setlogfiledumpfn setlogfiledumpfn)
{
        assert(moddefn != NULL);
        moddefn->filedump = filedump;
        moddefn->setlogfiledumpfn = setlogfiledumpfn;
}

/*********************************************************************************/
#define _ATO_ADD_ERRDEF(code, name, msg) ato_errdef_add(code, _library, name, msg)
static void _load_errdefs(void)
{
        _ATO_ADD_ERRDEF(ATO_ERR_OK, "ATO_ERR_OK", "");
        _ATO_ADD_ERRDEF(ATO_ERR_UNKNOWN, "ATO_ERR_UNKNOWN", "Unknown error - catch-all or unanticipated error not covered by other error codes");
        _ATO_ADD_ERRDEF(ATO_ERR_CFGNOMGR, "ATO_ERR_CFGNOMGR", "Deprecated");
        _ATO_ADD_ERRDEF(ATO_ERR_MEMORY, "ATO_ERR_MEMORY", "Memory allocaton error - currently this results in an assert");
        _ATO_ADD_ERRDEF(ATO_ERR_CFGNOTFOUND, "ATO_ERR_CFGNOTFOUND", "Configuration setting not found");
        _ATO_ADD_ERRDEF(ATO_ERR_INIT, "ATO_ERR_INIT", "Failing to initialise library");
        _ATO_ADD_ERRDEF(ATO_ERR_VERSION, "ATO_ERR_VERSION", "Version mismatch between client and library");
        _ATO_ADD_ERRDEF(ATO_ERR_VERSION_DEP, "ATO_ERR_VERSION_DEP", "Version mismatch with dependant library");
        _ATO_ADD_ERRDEF(ATO_ERR_B64, "ATO_ERR_B64", "Failed to decode from or encode to base64");
        _ATO_ADD_ERRDEF(ATO_ERR_NET, "ATO_ERR_NET", "Network communication error");
        _ATO_ADD_ERRDEF(ATO_ERR_FILEOPEN, "ATO_ERR_FILEOPEN", "Cannot open file");
        _ATO_ADD_ERRDEF(ATO_ERR_FILESTAT, "ATO_ERR_FILESTAT", "Cannot determine file size");
        _ATO_ADD_ERRDEF(ATO_ERR_FILEREAD, "ATO_ERR_FILEREAD", "Cannot read file");
        _ATO_ADD_ERRDEF(ATO_ERR_FILEWRITE, "ATO_ERR_FILEWRITE", "Cannot write file");
        _ATO_ADD_ERRDEF(ATO_ERR_P7, "ATO_ERR_P7", "Failed to access P7 (certificate)");
        _ATO_ADD_ERRDEF(ATO_ERR_P8, "ATO_ERR_P8", "Failed to access P8 (private key)");
        _ATO_ADD_ERRDEF(ATO_ERR_DER, "ATO_ERR_DER", "Failed to DER encode value");
        _ATO_ADD_ERRDEF(ATO_ERR_DUPLICATE_ERRDEF, "ATO_ERR_DUPLICATE_ERRDEF", "Internal/SDK error - duplicate error definition entry detected");
        _ATO_ADD_ERRDEF(ATO_ERR_DUPLICATE_ERRHANDLER, "ATO_ERR_DUPLICATE_ERRHANDLER", "Failed to add error handler for SDK - possible duplicate");
        _ATO_ADD_ERRDEF(ATO_ERR_XMLPARSE, "ATO_ERR_XMLPARSE", "Invalid XML");
        _ATO_ADD_ERRDEF(ATO_ERR_XMLROOT, "ATO_ERR_XMLROOT", "Failed to find or load root node");
        _ATO_ADD_ERRDEF(ATO_ERR_CFGDUPLICATE, "ATO_ERR_CFGDUPLICATE", "Possible duplicate confuration entry");
        _ATO_ADD_ERRDEF(ATO_ERR_XMLNODENOTFOUND, "ATO_ERR_XMLNODENOTFOUND", "Failed to find or load expected XML node");
        _ATO_ADD_ERRDEF(ATO_ERR_NET_TIMEOUT, "ATO_ERR_NET_TIMEOUT", "Network timeout connected to remote services");
        _ATO_ADD_ERRDEF(ATO_ERR_XML, "ATO_ERR_XML", "Error processing XML");
        _ATO_ADD_ERRDEF(ATO_ERR_NET_SOAP, "ATO_ERR_NET_SOAP", "Network SOAP error returned by remote client");
        _ATO_ADD_ERRDEF(ATO_ERR_HASH, "ATO_ERR_HASH", "Failed to generate crytographic hash");
        _ATO_ADD_ERRDEF(ATO_ERR_XMLXPCONTEXT, "ATO_ERR_XMLXPCONTEXT", "Failed to set XML Xpath context");
        _ATO_ADD_ERRDEF(ATO_ERR_INVALIDSOFTWAREINFO, "ATO_ERR_INVALIDSOFTWAREINFO", "Invalid or missing Software Info");
        _ATO_ADD_ERRDEF(ATO_ERR_CRYPTO, "ATO_ERR_CRYPTO", "General error from crypto routines - see message");
        _ATO_ADD_ERRDEF(ATO_ERR_SDK_EXPIRED, "ATO_ERR_SDK_EXPIRED", "SDK expiry date has been reached; please get an updated SDK");
}
