/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atoutilcore.h"

#include "atolst.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_LST;
static unsigned long _moduleid = ATO_BASE_MODULEID_LST;

static int _loglevel = ATO_LOG_WARN;

struct _ato_ListNode {
    char *key;
    void *value;
    void const *valueconst;
    bool isdeleted;
    ato_ListNode *prev;
    ato_ListNode *next;
    ato_List *lst;
};

struct _ato_List {
    size_t count;
    ato_ListNode *start;
    ato_ListNode *tail;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
int ato__lst_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}
void ato__lst_deinit(void)
{
}
/*********************************************************************************/

void ato_lst_create(ato_List **lst)
{
    ATO_ASSERT_ISNOTALLOCATED(lst);
    *lst = calloc(1, sizeof(ato_List));
}

static void _freenode(ato_ListNode *curr)
{
    curr->key = ato_free(curr->key);
    free(curr);
}

void ato_lst_clear(ato_List *lst)
{
    ato_ListNode *curr = NULL;

    if (lst == NULL)
        return;

    curr = lst->start;
    while (curr != NULL) {
        ato_ListNode *next = curr->next;
        _freenode(curr);
        curr = next;
    }
    lst->start = lst->tail = NULL;
    lst->count = 0;
}

void *ato_lst_free(ato_List *lst)
{
    if (lst == NULL) { return NULL; }
    ato_lst_clear(lst);
    free(lst);
    return NULL;
}

static void _createnode(ato_ListNode **node, const char *key, void *value, const void *valueconst)
{
    *node = calloc(1, sizeof(ato_ListNode));
    assert(*node != NULL);
    (*node)->key = ato_strdup(key, 0);
    (*node)->value = value;
    (*node)->valueconst = valueconst;
}

// Return value of NULL indicates value already exists.
static ato_ListNode *_lst_add(ato_List *lst, const char *key, void *value, const void *valueconst)
{
    ato_ListNode *node = NULL;
    assert(lst != NULL);
    if (ato_lst_find(lst, key) != NULL)
        return NULL;
    _createnode(&node, key, value, valueconst);
    if (lst->start == NULL) {
        lst->tail = lst->start = node;
    } else {
        lst->tail->next = node;
        node->prev = lst->tail;
        lst->tail = node;
    }
    node->lst = lst;
    lst->count++;
    return node;
}

ato_ListNode *ato_lst_add(ato_List *lst, const char *key, void *value) {
    return _lst_add(lst, key, value, NULL);
}

ato_ListNode *ato_lst_addconst(ato_List *lst, const char *key, const void *value) {
    return _lst_add(lst, key, NULL, value);
}

size_t ato_lst_count(ato_List *lst)
{
    assert(lst != NULL);
    return lst->count;
}

ato_ListNode *ato_lst_next(ato_List *lst, ato_ListNode *node)
{
    assert(lst != NULL);
    if (node == NULL)
        node = lst->start;
    else
        node = node->next;
    for (; node != NULL; node = node->next)
        if (!node->isdeleted)
            break;
    return node && !node->isdeleted ? node : NULL;
}

/*
ato_ListNode *ato_lst_item(ato_List *lst, size_t index)
{
    ato_ListNode *node = NULL;
    size_t ndx = 0;
    assert(lst != NULL);
    for (node = lst->start; node != NULL; node = node->next) {
        if (!node->isdeleted)
            ndx++;
        if (ndx == index+1)
            break;
    }
    return ndx == index+1 ? node : NULL;
}
*/

void *ato_lst_value(ato_ListNode *node) {
    if (node == NULL)
        return NULL;
    assert(!node->isdeleted);
    return node->value;
}

const void *ato_lst_valueconst(ato_ListNode *node) {
    if (node == NULL)
        return NULL;
    assert(!node->isdeleted);
    return node->valueconst;
}
const void *ato_lst_valueorconst(ato_ListNode *node) {
    const void *v = ato_lst_value(node);
    if (v == NULL) { v = ato_lst_valueconst(node); }
    return v;
}

void ato_lst_setvalue(ato_ListNode *node, void *value) {
    if (node == NULL)
        return;
    assert(!node->isdeleted);
    node->value = value;
    node->valueconst = value;
}
void ato_lst_setvalueconst(ato_ListNode *node, const void *value) {
    if (node == NULL)
        return;
    assert(!node->isdeleted);
    node->valueconst = value;
}

const char *ato_lst_key(ato_ListNode *node)
{
    if (node == NULL)
        return NULL;
    assert(!node->isdeleted);
    return node->key;
}

ato_ListNode *ato_lst_find(ato_List *lst, const char *key)
{
    ato_ListNode *node = NULL;
    assert(lst != NULL);
    if (key == NULL)
        return NULL;
    for (node = lst->start; node != NULL; node = node->next)
        if (!node->isdeleted && strcmp(node->key, key) == 0)
            break;
    return node;
}

void ato_lst_delete(ato_ListNode *node)
{
    if (node != NULL) {
        node->isdeleted = TRUE;
        assert(node->lst != NULL);
        node->lst->count--;
        node->value = NULL;
        node->valueconst = NULL;
        node->key = ato_free(node->key);
    }
}

const void *ato_lst_findvalueconst(ato_List *lst, const char *key)
{
    const void *value = NULL;
    ato_ListNode *node = ato_lst_find(lst, key);
    if (node) {
        value = ato_lst_valueconst(node);
    }
    return value;
}

/*
int ato_lst_delete(ato_ListNode *node)
{
    if (node != NULL) {
        ato_List *lst = node->lst;
        if (lst->count == 1) {
            lst->start = lst->tail = NULL;
        } else if (node == lst->tail) {
            node->prev->next = NULL;
            lst->tail = node->prev;
        } else if (node == lst->start) {
            node->next->prev = NULL;
            lst->start = node->next;
        } else {
            if (node->next != NULL)
                node->next->prev = node->prev;
            if (node->prev != NULL)
                node->prev->next = node->next;
        }
        lst->count--;
        _freenode(node);
    }
    return ATO_ERR_OK;
}
*/
