#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atoexcept.h"
#include "atostr.h"
#include "atoerr.h"
#include "atolog.h"
#include "atoutil.h"
#include "atoctx.h"
#include "atoutil.h"
#include "atopbe.h"

#include "atopk.h"

#include "atobcrypto.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_PK;
static unsigned long _moduleid = ATO_BASE_MODULEID_PK;

#define _ATO_KdfIterationCount 10000
#define _ATO_KdfLength 50
#define _ATO_KdfOid 256

/*********************************************************************************/
struct _ato_PwdKdf {
    const char *rpwd;
    ato_String *bsalt;
    char *dpwd;
    ato_Pbe *pbe;
};

/*********************************************************************************/
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}
/*********************************************************************************/

int ato__pk_init(void) {
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__pk_deinit(void) {
    static bool invoked = FALSE;
    if (invoked) return;
    invoked = TRUE;
}
/*********************************************************************************/
void *ato_pk_free(ato_PwdKdf *pk) {
    if (pk == NULL) return NULL;
    pk->rpwd = NULL;
    pk->dpwd = ato_freeclr(pk->dpwd);
    pk->pbe = ato_pbe_free(pk->pbe);
    pk->bsalt = ato_str_free(pk->bsalt);
    free(pk);
    return NULL;
}

ato_PwdKdf *ato_pk_create(ato_Ctx *ctx, ato_PwdKdf **obj, const char *pwd, const char *salt, const ato_Pbe *pbe) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_pk_create";
    int errcode = ATO_ERR_OK;
    ato_PwdKdf *pk = NULL;
    ATO_IGNORE(ctx);
    ATO_ASSERT_ISNOTALLOCATED(obj);
    assert(!ato_isnullorempty(pwd));
    *obj = pk = calloc(1, sizeof(ato_PwdKdf));
    assert(pk != NULL);
    pk->rpwd = pwd;
    if (ato_base64decode_array(ctx, &(pk->bsalt), salt, strlen(salt)) != ATO_ERR_OK) {
        errcode = ATO_CTX_NEWERR(ctx, ATO_ERR_CRYPTO, "Cannot decode salt.");
        goto finally;
    }

    ato_pbe_create(ctx, &(pk->pbe), ato_pbe_str(pbe));

finally:
    if (errcode != ATO_ERR_OK) {
        ato_pk_free(pk);
        Throw errcode;
    }

    return pk;
}

/*********************************************************************************/
const char *ato_pk_rpwd(ato_PwdKdf *pk) {
    assert(pk != NULL);
    return pk->rpwd;
}

const char *ato_pk_dpwd(ato_Ctx *ctx, ato_PwdKdf *pk) {
    assert(pk != NULL);
    if (pk->dpwd == NULL) {
        ato_crypto_kdf(ctx, &(pk->dpwd), pk->rpwd, pk->bsalt, _ATO_KdfIterationCount, _ATO_KdfLength, _ATO_KdfOid);
    }
    return pk->dpwd;
}

const char *ato_pk_pwd(ato_Ctx *ctx, ato_PwdKdf *pk) {
    assert(pk != NULL);
    if (ato_pbe_isPbe2(pk->pbe)) { return ato_pk_dpwd(ctx, pk); }
    return pk->rpwd;
}

void ato_pk_rpwd_set(ato_PwdKdf *pk, const char *rpwd) {
    assert(pk != NULL);
    pk->rpwd = rpwd;
}

void ato_pk_pbe_set(ato_Ctx *ctx, ato_PwdKdf *pk, const ato_Pbe *pbe) {
    assert(pk != NULL);
    pk->pbe = ato_pbe_free(pk->pbe);
    ato_pbe_create(ctx, &(pk->pbe), ato_pbe_str(pbe));
}

const ato_Pbe *ato_pk_pbe(ato_PwdKdf *pk) {
    assert(pk != NULL);
    return pk->pbe;
}

bool ato_pk_isPbe2(ato_PwdKdf *pk) {
    assert(pk != NULL);
    return ato_pbe_isPbe2(pk->pbe);
}
