#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>

#include <curl/curl.h>

#if ! (defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64))
#include "valgrind-memcheck.h"
#endif

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"


#include "atointernal.h"
#include "atotypes.h"
//#include "atoerrsoap.h"
#include "atonetd.h"
#include "atobnet.h"

static const char *_library = ATO_BNET_LIBRARY;
static const char *_module = ATO_BNET_MODULE_NET;
static unsigned long _moduleid = ATO_BNET_MODULEID_NET;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;
static bool _logdumpfile = FALSE;

static unsigned int _debug_flag = ATO_BNET_DEBUG_FLAG_TEXT; // ATO_BNET_DEBUG_FLAG_ALL ATO_BNET_DEBUG_FLAG_TEXT

static ato_curloptfunction *_curloptfunction;

//=============================================================================
struct _ato_Net {
    CURL *curl;
    char errbuf[CURL_ERROR_SIZE];
    char *proxyurl;
    char *proxyuserpwd;
    char *proxyexcludehosts;
    char *host;
    char *url;
    char *contenttype;
    size_t timeout;
    size_t timeoutconnect;
    FILE *fp;
    bool resp_init;
    int resp_errcode;
    int resp_http;
    ato_eNetDataType resp_type;
    ato_eNetDataTypeEnc resp_encoding;
    char *resp_contenttype;
    char *resp_boundary;
    ato_NetData *netdata;
    ato_NetDataItem *nditem;
};

#define ATO_NET_HTTPRESP_NONE 0
#define ATO_NET_HTTPRESP_IGNORE 1
#define ATO_NET_HTTPRESP_OK 2
#define ATO_NET_HTTPRESP_BAD 3
#define ATO_NET_HTTPRESP_OTHER 4

#define ATO_NET_ERR_NO_CONTENT_TYPE             1
#define ATO_NET_ERR_UNSUPPORTED_CONTENT_TYPE    2

//=============================================================================
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}
static void _setlogdumpfile(bool logdumpfile)
{
    _logdumpfile = logdumpfile;
}

//=============================================================================
static void *_clear_resp_vars(ato_Net *net)
{
    if (!net) return NULL;

    net->errbuf[0] = '\0';
    net->resp_init = FALSE;
    net->resp_http = ATO_NET_HTTPRESP_IGNORE;
    net->resp_type = ATO_BNETD_TYPE_UNKNOWN;
    net->resp_errcode = 0;
    net->resp_boundary = ato_free(net->resp_boundary);
    net->resp_contenttype = ato_free(net->resp_contenttype);
    net->nditem = NULL;
    ato_bnetd_free(net->netdata);
    net->netdata = NULL;
    return NULL;
}

static void _reset_resp_vars(ato_Net *net)
{
    _clear_resp_vars(net);
    ato_bnetd_create(&(net->netdata));
}

static bool _ato_bnet_ignore_ssl_error(ato_Ctx *ctx)
{
    ATO_IGNORE(ctx);
    return FALSE;
}

static long _ato_bnet_proxy_authtype(ato_Ctx *ctx)
{
    const char *authtype = NULL;
    ATO_IGNORE(ctx);
    if (!ato_isnullorempty(authtype)) {
        if (ato_streqi(authtype, "basic"))
            return CURLAUTH_BASIC;
        if (ato_streqi(authtype, "digest"))
            return CURLAUTH_DIGEST;
        if (ato_streqi(authtype, "ntlm"))
            return CURLAUTH_NTLM;
    }
    return CURLAUTH_ANY;
}

//=============================================================================
static void _free_proxy(ato_Net *net)
{
    net->proxyurl = ato_free(net->proxyurl);
    net->proxyuserpwd = ato_free(net->proxyuserpwd);
    net->proxyexcludehosts = ato_free(net->proxyexcludehosts);
}

//=============================================================================
static void _adjustfornsprefix(char *buf, char **ns, char **value)
{
    char *s = strchr(buf, ':');
    if (!s) {
        *ns = ato_strdup("", 0);
        *value = ato_strdup(buf, 0);
    } else {
        *s = '\0';
        *ns = ato_strdup(buf, 0);
        *value = ato_strdup(s+1, 0);
    }
    free(buf);
}

static void _set_soapns(ato_Ctx *ctx, ato_Xml *xml)
{
    assert(xml != NULL);
    ato_xml_namespace_register(ctx, xml, "s", "http://www.w3.org/2003/05/soap-envelope");
}

static bool _has_soaperrors(ato_Ctx *ctx, ato_ErrSoap **err, ato_String *buffer)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *ResponseError_Fault = "//s:Body/s:Fault";
    static const char *ResponseError_Value = "./s:Code//s:Value";
    static const char *ResponseError_Node = "s:Node";
    static const char *ResponseError_Detail = "s:Detail";
    static const char *ResponseError_Reason = "s:Reason/s:Text";
    int errcode = ATO_ERR_OK;
    bool contains_soaperr = FALSE;
    void *xlist = NULL;
    size_t i = 0, count = 0;
    ato_Xml *xml = NULL;
    ato_ErrSoap *error = NULL;
    void *pxnode = NULL;

    char *node = NULL, *detail = NULL, *reason = NULL;

    // If it's not XML  or fails to parse, clear exception and just return false;
    Try {
        ato_xml_create(ctx, &xml, buffer);
    } Catch (errcode) {
    }
    if (errcode != ATO_ERR_OK) {
        ato_ctx_err_clear(ctx);
        return FALSE;
    }

    Try {
        _set_soapns(ctx, xml);

        pxnode = ato_xml_findnode(xml, NULL, ResponseError_Fault, NULL);
        if (pxnode) {
            contains_soaperr = TRUE;

            ato_xml_nodevalue(xml, pxnode, ResponseError_Node, &node, ATO_XML_RAW_FALSE);
            ato_xml_nodevalue(xml, pxnode, ResponseError_Detail, &detail, ATO_XML_RAW_INNER);
            ato_xml_nodevalue(xml, pxnode, ResponseError_Reason, &reason, ATO_XML_RAW_FALSE);

            xlist = ato_xml_listload(xml, pxnode, ResponseError_Value, NULL);
            count = ato_xml_listcount(xml, xlist);

            ato_errsoap_createassign(ctx, &error, detail, node, reason, count);
            for (i = 0; i < count; i++) {
                char *buf = NULL, *value = NULL, *nsprefix = NULL, *ns = NULL;
                void *psubnode = NULL;
                psubnode = ato_xml_listitem(xml, xlist, i);
                assert(psubnode != NULL);
                ato_xml_nodevalue(xml, psubnode, ".", &buf, FALSE);
                _adjustfornsprefix(buf, &nsprefix, &value);
                ato_errsoap_code_addassign(ctx, error, value, ato_xml_findnamespace(xml, psubnode, nsprefix, &ns), i);
                if (nsprefix) free(nsprefix);
            }
            ato_xml_listfree(xml, xlist);
        }

    } Catch (errcode) {
    }

    ato_xml_free(xml);
    *err = error;
    ATO_RETHROW_ONERR(errcode);
    return contains_soaperr;
}

//=============================================================================

static bool _inlist(const char *value, const char *values)
{
    const char *delim = ",;";
    bool match = FALSE;
    char **elts = NULL;
    size_t count = 0, i = 0;

    assert(value != NULL); assert(values != NULL);

    count = ato_strtok2array(&elts, values, delim);
    for (i = 0; i < count; i++) {
        match = ato_streqi(value, elts[i]);
        if (match) break;
    }
    ato_strtokfree(elts);
    return match;
}

static bool _useProxy(ato_Net *net)
{
    return net->proxyurl != NULL && (net->proxyexcludehosts == NULL || !_inlist(net->host, net->proxyexcludehosts));
}

static const char *_getfmtbuf(char *fmtbuf, const char *hdr, size_t size)
{
        //sprintf(fmtbuf, "%s:%%.%" PRINTF_SIZET "us", hdr, size); // below is clearer.
        char fmtbufsize[10];
        sprintf(fmtbufsize, "%" PRINTF_SIZET "u", size);
        strcat(strcat(strcat(strcpy(fmtbuf, hdr), ":%."), fmtbufsize), "s");
        return fmtbuf;
}

static int _curl_logger(CURL *curl, curl_infotype itype, char *text, size_t size, ato_Ctx *ctx)
{
    const char *function = "_curl_logger";

    ATO_IGNORE(curl);
    if (size == 0)
        return 0;
    else {
        ato_Log *log = NULL;
        char fmtbuf[20];

        log = ato_ctx_log(ctx);
        switch (itype) {
            case CURLINFO_TEXT:
                if (_debug_flag & ATO_BNET_DEBUG_FLAG_TEXT) {
                    size_t i = 0;
                    char *textbuf = ato_memdup(text, size, 1);
                    textbuf[size] = '\0';
                    for (; size > 0 && textbuf[size-1] == '\n'; size--)
                        textbuf[size-1] = '\0';
                    for (i = 0; i < size; i++)
                        if (textbuf[i] == '\n') textbuf[i] = ' ';

                    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, _getfmtbuf(fmtbuf, "TEXT", size), textbuf);
                    free(textbuf);
                }
                break;
            case CURLINFO_HEADER_IN:
                if (_debug_flag & ATO_BNET_DEBUG_FLAG_HDR_IN) {
                    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, _getfmtbuf(fmtbuf, "HDR_IN", size), text);
                }
                break;
            case CURLINFO_HEADER_OUT:
                if (_debug_flag & ATO_BNET_DEBUG_FLAG_HDR_OUT) {
                    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, _getfmtbuf(fmtbuf, "HDR_OUT", size), text);
                }
                break;
            case CURLINFO_DATA_IN:
                if (_debug_flag & ATO_BNET_DEBUG_FLAG_DATA_IN) {
                    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, _getfmtbuf(fmtbuf, "DATA_IN", size), text);
                }
                break;
            case CURLINFO_DATA_OUT:
                if (_debug_flag & ATO_BNET_DEBUG_FLAG_DATA_OUT) {
                    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, _getfmtbuf(fmtbuf, "DATA_OUT", size), text);
                }
                break;
            case CURLINFO_SSL_DATA_IN:
                if (_debug_flag & ATO_BNET_DEBUG_FLAG_SSL_IN) {
                    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, _getfmtbuf(fmtbuf, "SSL_IN", size), text);
                }
                break;
            case CURLINFO_SSL_DATA_OUT:
                if (_debug_flag & ATO_BNET_DEBUG_FLAG_SSL_OUT) {
                    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, _getfmtbuf(fmtbuf, "SSL_OUT", size), text);
                }
                break;
            case CURLINFO_END:
                break;
        }
    }
    return 0;
}

static ato_eNetDataType _map_mimetype(char **mimestring, char **boundary, const char *chbuf)
{
    static const char *MimeType_xopxml = "application/xop+xml";
    static const char *MimeType_soapxml = "application/soap+xml";
    static const char *MimeType_multipart = "multipart/related";
    ato_eNetDataType mtypecode = ATO_BNETD_TYPE_UNKNOWN;
    const char *chbuf_end = chbuf;

    assert(chbuf != NULL);
    ATO_ASSERT_ISNOTALLOCATED(mimestring); ATO_ASSERT_ISNOTALLOCATED(boundary);
    while (*chbuf_end != '\r' && *chbuf_end != '\0' && *chbuf_end != ';')
        chbuf_end++;

    {
    size_t diff = (size_t)(chbuf_end - chbuf);
    *mimestring = ato_memdup(chbuf, diff, 1);
    assert(*mimestring != NULL);
    (*mimestring)[diff] = '\0';
    }

    if (ato_strstartswithi(chbuf, MimeType_soapxml)) {
        mtypecode = ATO_BNETD_TYPE_SOAP;
    } else if (ato_strstartswithi(chbuf, MimeType_xopxml)) {
        mtypecode = ATO_BNETD_TYPE_XOP;
    } else if (ato_strstartswithi(chbuf, MimeType_multipart)) {
        const char *name = chbuf + strlen(MimeType_multipart) + 2;
        mtypecode = ATO_BNETD_TYPE_MULTI;
        if (ato_strstartswithi(name, "boundary")) {
            char *boundary_start = strchr(name, '"');
            char *boundary_end = NULL;
            boundary_end = strchr(++boundary_start, '"');
            *boundary_end = '\0';
            if (boundary)
                *boundary = ato_strdup(boundary_start, 0);
        }
    } else {
        mtypecode = ATO_BNETD_TYPE_OTHER;
    }

    return mtypecode;
}

static ato_eNetDataTypeEnc _map_encoding(const char *chbuf)
{
    static const char *Enc_qp = "quoted-printable";
    static const char *Enc_b64 = "base64";
    ato_eNetDataTypeEnc enc = ATO_BNETD_ENC_NONE;

    assert(chbuf != NULL);

    if (ato_strstartswithi(chbuf, Enc_b64)) {
        enc = ATO_BNETD_ENC_BASE64;
    } else if (ato_strstartswithi(chbuf, Enc_qp)) {
        enc = ATO_BNETD_ENC_QUOTED;
    } else {
        enc = ATO_BNETD_ENC_NONE;
    }

    return enc;
}

static void _addcontent(ato_NetDataItem *nditem, const char *content, size_t len)
{
    ato_String *buf = NULL;
    char *chbuf = NULL;
    assert(nditem != NULL);
    chbuf = calloc(len+1, sizeof(char));
    assert(chbuf != NULL);
    memcpy(chbuf, content, len);
    chbuf[len] = '\0';
    ato_str_create(&buf, chbuf, len, TRUE);
    ato_bnetd_addcontent(nditem, buf);
}

/*
    A segment consists of an header block followed by an data block.

    An entire message consists of one "global" segment - i.e. a global header block plus
    a global data block.

    The global header block is processed by _write_hdr_response(). This header determines
    if the message is single or multi part mime. _write_data_response() processes the
    global data block.

    With single-part mime, no futher processing is required.

    With multi-part mime there are multiple "sub-segments" within the global data block
    seperated by a boundaryid. The boundaryid was supplied in the global header block.
    This means that for multi-part mime, _write_data_response() must switch between a
    header block processing mode and a data block processing mode.

    This method is called repeatedly by libcurl during the processing of header or data
    blocks therefore we can make no assunmptions about received complete blocks. We
    do however assume that complete HTML header lines are received, and where
    multiple lines are received they start with crlf (as per protocol).
*/
static size_t _write_data_response(ato_Net *net, const char *buffer, size_t len)
{
    if (buffer == NULL || len == 0)
        return 0;

    if (net->fp) {
        fwrite(buffer, sizeof(char), len, net->fp);
    }

    if (!net->resp_init) {
        net->nditem = ato_bnetd_additem(net->netdata);
        ato_bnetd_settype(net->nditem, net->resp_type);
        ato_bnetd_setcontenttype(net->nditem, net->resp_contenttype);
        ato_bnetd_setencoding(net->nditem, net->resp_encoding);
        if (net->resp_boundary)
            ato_bnetd_setcid(net->nditem, net->resp_boundary);
        net->resp_init = TRUE;
    }

    _addcontent(net->nditem, buffer, len);

    return len;
}

// must refactor this as per data processing to read entire header from libcurl BEFORE processing it, instead of DURING.
static size_t _write_hdr_response(ato_Net *net, const char *buffer, size_t len)
{
    static const char *ContentType = "Content-Type: ";
    static const char *ContentEncoding = "Content-Transfer-Encoding: ";

    if (buffer == NULL || len == 0)
        return 0;

    if (net->fp) {
        fwrite(buffer, sizeof(char), len, net->fp);
    }

    if (ato_strstartswithi(buffer, "HTTP")) {
        if (strstr(buffer, " 200 Connection") != NULL)
            net->resp_http = ATO_NET_HTTPRESP_IGNORE;
        else if (strstr(buffer, " 407") != NULL)
            net->resp_http = ATO_NET_HTTPRESP_IGNORE;
        else if (strstr(buffer, " 200 OK") != NULL)
            net->resp_http = ATO_NET_HTTPRESP_OK;
        else if (strstr(buffer, " Bad Request") != NULL)
            net->resp_http = ATO_NET_HTTPRESP_BAD;
        else
            net->resp_http = ATO_NET_HTTPRESP_OTHER;
    } else {
        if (net->resp_http != ATO_NET_HTTPRESP_NONE && net->resp_http != ATO_NET_HTTPRESP_IGNORE) {
            if (ato_strstartswithi(buffer, ContentType)) {
                net->resp_type = _map_mimetype(&(net->resp_contenttype), &(net->resp_boundary), buffer + strlen(ContentType));
                if (net->resp_type == ATO_BNETD_TYPE_UNKNOWN) {
                    net->resp_errcode = ATO_NET_ERR_UNSUPPORTED_CONTENT_TYPE;
                    len = 0;
                }
            } else if (ato_strstartswithi(buffer, ContentEncoding)) {
                net->resp_encoding = _map_encoding(buffer + strlen(ContentEncoding));
            }
        }
    }

    return len;
}

static size_t _write_data_callback(char *buffer, size_t size, size_t nmemb, ato_Net *net)
{
#if ! (defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64))
    VALGRIND_MAKE_MEM_DEFINED(&nmemb, sizeof(nmemb));
    VALGRIND_MAKE_MEM_DEFINED(buffer, size * nmemb);
#endif
    return _write_data_response(net, buffer, size * nmemb);
}

static size_t _write_hdr_callback(char *buffer, size_t size, size_t nmemb, ato_Net *net)
{
#if ! (defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64))
    VALGRIND_MAKE_MEM_DEFINED(&nmemb, sizeof(nmemb));
    VALGRIND_MAKE_MEM_DEFINED(buffer, size * nmemb);
#endif
    return _write_hdr_response(net, buffer, size * nmemb);
}

static CURLcode _process_request_init(ato_Ctx *ctx, ato_Net *net, struct curl_slist **slistaddr, const char *contenttype)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_process_request_init";
    //int errcode = ATO_ERR_OK;
    CURLcode ret = CURLE_OK;
    struct curl_slist *slist = NULL;
    char sbuf[1000];
    ato_Log *log = ato_ctx_log(ctx);

    _reset_resp_vars(net);

    sprintf(sbuf, "Content-Type: %s", contenttype);
    *slistaddr = slist = curl_slist_append(slist, sbuf);
    if (slist == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_NET, "Failed to initialise internal network buffer");
    // *slistaddr = slist = curl_slist_append(slist, "Transfer-Encoding: chunked");
    *slistaddr = slist = curl_slist_append(slist, "Connection: Keep-Alive");
    *slistaddr = slist = curl_slist_append(slist, "Expect:"); //  100-continue
    sprintf(sbuf, "Host: %s", net->host);
    // *slistaddr = slist = curl_slist_append(slist, sbuf);
    // *slistaddr = slist = curl_slist_append(slist, "AtoTransactionId:\"\"");

    ret = curl_easy_setopt(net->curl, CURLOPT_NOSIGNAL, TRUE);
    //ret = curl_easy_setopt(net->curl, CURLOPT_FAILONERROR, 1L);
    ret = curl_easy_setopt(net->curl, CURLOPT_ERRORBUFFER, net->errbuf);
    ret = curl_easy_setopt(net->curl, CURLOPT_VERBOSE, 1L);
    //ret = curl_easy_setopt(net->curl, CURLOPT_HEADER, 1L);
    ret = curl_easy_setopt(net->curl, CURLOPT_URL, net->url);
    ret = curl_easy_setopt(net->curl, CURLOPT_POST, 1L);
    ret = curl_easy_setopt(net->curl, CURLOPT_TIMEOUT, net->timeout);
    ret = curl_easy_setopt(net->curl, CURLOPT_CONNECTTIMEOUT, net->timeoutconnect);
    ret = curl_easy_setopt(net->curl, CURLOPT_HTTPHEADER, slist);
    ret = curl_easy_setopt(net->curl, CURLOPT_WRITEDATA, net);
    ret = curl_easy_setopt(net->curl, CURLOPT_WRITEFUNCTION, (curl_write_callback) _write_data_callback);
    ret = curl_easy_setopt(net->curl, CURLOPT_WRITEHEADER, net);
    ret = curl_easy_setopt(net->curl, CURLOPT_HEADERFUNCTION, (curl_write_callback) _write_hdr_callback);
    ret = curl_easy_setopt(net->curl, CURLOPT_DEBUGDATA, ctx);
    ret = curl_easy_setopt(net->curl, CURLOPT_DEBUGFUNCTION , (curl_debug_callback) _curl_logger);

    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Resetting timeouts: " "Timeout=%" PRINTF_SIZET "u" ", " "ConnectionTimeout=%" PRINTF_SIZET "u", net->timeout, net->timeoutconnect);

    if (_useProxy(net)) {
        const char *up = net->proxyuserpwd;
        if (up != NULL)
            ret = curl_easy_setopt(net->curl, CURLOPT_PROXYUSERPWD, up);
        ret = curl_easy_setopt(net->curl, CURLOPT_PROXY, net->proxyurl);
        ret = curl_easy_setopt(net->curl, CURLOPT_PROXYAUTH, _ato_bnet_proxy_authtype(ctx));
    }

    if (_ato_bnet_ignore_ssl_error(ctx)) {
        ret = curl_easy_setopt(net->curl, CURLOPT_SSL_VERIFYHOST, 0L);
        ret = curl_easy_setopt(net->curl, CURLOPT_SSL_VERIFYPEER, 0L);
    } else {
        ret = curl_easy_setopt(net->curl, CURLOPT_SSL_VERIFYHOST, 2L);
        ret = curl_easy_setopt(net->curl, CURLOPT_SSL_VERIFYPEER, 2L);
    }

    {
        const char *ca_bundle = getenv("CURL_CA_BUNDLE");
        if (ca_bundle != NULL) {
            ret = curl_easy_setopt(net->curl, CURLOPT_CAINFO, ca_bundle);
        }
    }

    if (_curloptfunction != NULL) { ret = _curloptfunction(net->curl); }

    if (_logdumpfile)   {
        char filename[1000];
        strcpy(filename, "net-response-");
        //ato_new_pseudo_guid(ctx, filename + strlen(filename));
        strcat(filename, net->host);
        strcat(filename, ".txt");
        net->fp = fopen(filename, "wb");
    }

    return ret;
}

static CURLcode _process_request(ato_Ctx *ctx, ato_Net *net)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_process_request";
    CURLcode ret = CURLE_OK;

    ret = curl_easy_perform(net->curl);
    if (net->fp) { fclose(net->fp); net->fp = NULL; }
    if (net->resp_errcode != 0) {
        if (net->resp_errcode == ATO_NET_ERR_NO_CONTENT_TYPE)
            Throw ATO_CTX_NEWERR(ctx, ATO_ERR_NET, "Unable to determine response Content-Type");
        else if (net->resp_errcode == ATO_NET_ERR_UNSUPPORTED_CONTENT_TYPE)
            Throw ATO_CTX_NEWERR(ctx, ATO_ERR_NET, "Content-Type not supported");
    }

    return ret;
}

static CURLcode _process_requestsp(ato_Ctx *ctx, ato_Net *net, ato_String *reqdata)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    CURLcode ret = CURLE_OK;
    struct curl_slist *slist = NULL;
    const char *data = ato_str_value(reqdata);
    size_t len = ato_str_len(reqdata);

    Try {

        if (_logdumpfile)   {
            char filename[1000];
            strcpy(filename, "net-request-");
            //ato_new_pseudo_guid(ctx, filename + strlen(filename));
            strcat(filename, net->host);
            strcat(filename, ".txt");
            net->fp = fopen(filename, "wb");
            if (net->fp) {
                fwrite(data, sizeof(char), len, net->fp);
                fclose(net->fp);
                net->fp = NULL;
            }
        }

        ret = _process_request_init(ctx, net, &slist, net->contenttype);
        ret = curl_easy_setopt(net->curl, CURLOPT_POSTFIELDS, data);
        ret = curl_easy_setopt(net->curl, CURLOPT_POSTFIELDSIZE, len);
        ret = _process_request(ctx, net);

    } Catch (errcode) {
    }

    if (net->fp) { fclose(net->fp); net->fp = NULL; }
    curl_easy_reset(net->curl);
    if (slist != NULL) curl_slist_free_all(slist);
    ATO_RETHROW_ONERR(errcode);
    return ret;
}

static struct curl_httppost *_mpdata(ato_Ctx *ctx, ato_NetData *netdata, struct curl_slist **slistarray)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_mpdata";
    struct curl_httppost* post = NULL;
    struct curl_httppost* last = NULL;
    struct curl_slist *slist = NULL;
    ato_NetDataItem *nditem = ato_bnetd_firstv(netdata);
    ato_String *content = NULL;
    char sbuf[1000];
    CURLFORMcode retcode = CURL_FORMADD_OK;
    size_t count = ato_bnetd_count(netdata), i = 0;

    assert(i < count); ATO_IGNORE(count);
    content = ato_bnetd_content(nditem);
    slistarray[i++] = slist = curl_slist_append(NULL, "Content-Transfer-Encoding: binary");
    retcode = curl_formadd(&post, &last,
        CURLFORM_COPYNAME, "xml",
        CURLFORM_NAMELENGTH, 0,
        CURLFORM_PTRCONTENTS, ato_str_value(content),
        CURLFORM_CONTENTSLENGTH, ato_str_len(content),
        CURLFORM_CONTENTTYPE, "application/xop+xml; charset=UTF-8; type=\"application/soap+xml\"",
        CURLFORM_CONTENTHEADER, slist,
        CURLFORM_END);
    if (retcode != CURL_FORMADD_OK)
        Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET, 20, "Failed adding xml to multi-part mime - network failure code %d", (int)retcode);

    while ((nditem = ato_bnetd_nextv(netdata)) != NULL) {
        assert(i < count);
        content = ato_bnetd_content(nditem);
        slist = curl_slist_append(NULL, "Content-Transfer-Encoding: binary");
        sprintf(sbuf, "Content-ID: <%s>", ato_bnetd_cidnoprefix(nditem));
        slistarray[i++] = slist = curl_slist_append(slist, sbuf);
        retcode = curl_formadd(&post, &last,
            CURLFORM_COPYNAME, "attachment",
            CURLFORM_NAMELENGTH, 0,
            CURLFORM_PTRCONTENTS, ato_str_value(content),
            CURLFORM_CONTENTSLENGTH, ato_str_len(content),
            CURLFORM_CONTENTTYPE, "application/pdf",
            CURLFORM_CONTENTHEADER, slist,
            CURLFORM_END);
        if (retcode != CURL_FORMADD_OK)
            Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET, 20, "Failed adding attachment to multi-part mime - network failure code %d", (int)retcode);
    }

    return post;
}

static CURLcode _process_requestmp(ato_Ctx *ctx, ato_Net *net, ato_NetData *netdata)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_process_requestmp";
    int errcode = ATO_ERR_OK;
    CURLcode ret = CURLE_OK;
    struct curl_slist *slist = NULL;
    //ato_Log *log = NULL;
    struct curl_httppost *mpdata = NULL;
    struct curl_slist **mpslistarray = NULL;
    size_t mpslistcount = 0, i = 0;

    ATO_CTX_FN_START(ctx);

    //log = ato_ctx_log(ctx);

    Try {

        mpslistcount = ato_bnetd_count(netdata);
        mpslistarray = calloc(mpslistcount+1, sizeof(void *));
        assert(mpslistarray != NULL);
        mpdata = _mpdata(ctx, netdata, mpslistarray);

        //multipart/related; start-info=\"application/soap+xml\"; type=\"application/xop+xml\"
        ret = _process_request_init(ctx, net, &slist, "multipart/related; start-info=\"application/soap+xml\"; type=\"application/xop+xml\"");
        ret = curl_easy_setopt(net->curl, CURLOPT_HTTPPOST, mpdata);
        ret = _process_request(ctx, net);

    } Catch (errcode) {
        ;
    }

    if (mpdata != NULL) curl_formfree(mpdata);
    for (i = 0; i < mpslistcount; i++) {
        if (mpslistarray[i])
            curl_slist_free_all(mpslistarray[i]);
    }
    free(mpslistarray);

    curl_easy_reset(net->curl);
    if (slist != NULL) curl_slist_free_all(slist);
    ATO_CTX_FN_END(ctx, errcode);
    ATO_RETHROW_ONERR(errcode);
    return ret;
}

static void _process_response(ato_Ctx *ctx, ato_Net *net, CURLcode curlcode)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_process_response";
    ato_String *response = NULL;

    ato_bnetd_content(net->nditem);

    if (net->resp_type == ATO_BNETD_TYPE_MULTI) {
        ato_bnetd_processmpresponse(ctx, net->netdata, net->resp_boundary);
    }

    response = ato_bnetd_content(ato_bnetd_item_primary(net->netdata));
    if (response == NULL || ato_str_len(response) == 0) {
        Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET, 20, "Empty response or failed to parse response - network failure code %d", (int)curlcode);
    } else if (net->resp_http != ATO_NET_HTTPRESP_OK) {
        Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET, 20, "Network failure HTTP code %d", net->resp_http);
    } else {
        int errcode = ATO_ERR_OK;
        ato_ErrSoap *error = NULL;
        bool hassoaperrors = FALSE;

        // Capture any leaked exceptions from this call so they'll be wrapped as inner exceptions of ATO_ERR_NET
        Try {
            hassoaperrors = _has_soaperrors(ctx, &error, response);
        } Catch (errcode) {
        }
        if (hassoaperrors) {
            errcode = ato_ctx_err_new(ctx, _library, _module, function,  __LINE__, ATO_ERR_NET_SOAP, 0, error, "SOAP response error - please see SOAP error details.");
        }
        if (errcode != ATO_ERR_OK) { Throw errcode; }
    }
}

//=============================================================================
//=============================================================================
ato_NetData *ato_bnet_request(ato_Ctx *ctx, ato_Net *net, ato_String *data)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_bnet_request";
    int errcode = ATO_ERR_OK;
    CURLcode curlcode = CURLE_OK;
    size_t errarglen = strlen("Network failure code %d");

    ATO_CTX_FN_START(ctx);

    Try {
        curlcode = _process_requestsp(ctx, net, data);
        if (curlcode != CURLE_OK && curlcode != CURLE_HTTP_RETURNED_ERROR) {
            if (curlcode == CURLE_OPERATION_TIMEDOUT)
                Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET_TIMEOUT, errarglen, "Network failure code %d", (int)curlcode);
            Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET, errarglen, "Network failure code %d", (int)curlcode);
        } else {
            _process_response(ctx, net, curlcode);
        }
    } Catch (errcode) {
    }

    ATO_CTX_FN_END(ctx, errcode);
    ATO_RETHROW_ONERR(errcode);
    return net->netdata;
}

ato_NetData *ato_bnet_requestmp(ato_Ctx *ctx, ato_Net *net, ato_NetData *netdata)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_bnet_requestmp";
    CURLcode curlcode = CURLE_OK;
    size_t errarglen = strlen("Network failure code %d");
    ato_NetDataItem *nditem = ato_bnetd_firstv(netdata);

    if (ato_bnetd_type(nditem) == ATO_BNETD_TYPE_XOP) {
        curlcode = _process_requestmp(ctx, net, netdata);
    } else {
        curlcode = _process_requestsp(ctx, net, ato_bnetd_content(nditem));
    }

    if (curlcode != CURLE_OK && curlcode != CURLE_HTTP_RETURNED_ERROR) {
        if (curlcode == CURLE_OPERATION_TIMEDOUT)
            Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET_TIMEOUT, errarglen, "Network failure code %d", (int)curlcode);
        Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_NET, errarglen, "Network failure code %d", (int)curlcode);
    } else {
        _process_response(ctx, net, curlcode);
    }
    return net->netdata;
}

size_t ato_bnet_settimeout(ato_Net *net, size_t timeout)
{
    return net->timeout = timeout;
}

size_t ato_bnet_settimeoutconnect(ato_Net *net, size_t timeout)
{
    return net->timeoutconnect = timeout;
}

size_t ato_bnet_timeout(ato_Net *net)
{
    return net->timeout;
}

size_t ato_bnet_timeoutconnect(ato_Net *net)
{
    return net->timeoutconnect;
}

const char *ato_bnet_curl_version(void)
{
    curl_version_info_data *vinfo = curl_version_info(CURLVERSION_NOW);
    return vinfo->version;
}

void ato_bnet_setdebugflag(unsigned int flag)
{
    _debug_flag = flag;
}

unsigned int ato_bnet_debugflag(void)
{
    return _debug_flag;
}

unsigned int ato_bnet_names2flag(const char *names)
{
    unsigned int flag = 0;
    if (ato_isnullorempty(names))                        return ATO_BNET_DEBUG_FLAG_OFF;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_OFF_NAME))      return ATO_BNET_DEBUG_FLAG_OFF;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_ALL_NAME))      return ATO_BNET_DEBUG_FLAG_ALL;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_TEXT_NAME))     flag |= ATO_BNET_DEBUG_FLAG_TEXT;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_HDR_IN_NAME))   flag |= ATO_BNET_DEBUG_FLAG_HDR_IN;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_HDR_OUT_NAME))  flag |= ATO_BNET_DEBUG_FLAG_HDR_OUT;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_DATA_IN_NAME))  flag |= ATO_BNET_DEBUG_FLAG_DATA_IN;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_DATA_OUT_NAME)) flag |= ATO_BNET_DEBUG_FLAG_DATA_OUT;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_SSL_IN_NAME))   flag |= ATO_BNET_DEBUG_FLAG_SSL_IN;
    if (strstr(names, ATO_BNET_DEBUG_FLAG_SSL_OUT_NAME))  flag |= ATO_BNET_DEBUG_FLAG_SSL_OUT;
    return flag;
}

void ato_bnet_setproxy(ato_Net *net, const char *url, const char *userpwd, const char *excludehosts)
{
    _free_proxy(net);
    if (url != NULL) {
        net->proxyurl = ato_strdup(url, 0);
        if (userpwd != NULL)
            net->proxyuserpwd = ato_strdup(userpwd, 0);
        if (excludehosts != NULL)
            net->proxyexcludehosts = ato_strdup(excludehosts, 0);
    }
}

int ato__net_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;

    if (curl_global_init(CURL_GLOBAL_ALL) != CURLE_OK)
        return ATO_ERR_INIT;

    invoked = TRUE;

    {
        ato_LibModule *lm = ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        ato_initfnlogfiledump(lm, _logdumpfile, _setlogdumpfile);
    }

    return ATO_ERR_OK;
}

void ato__net_deinit(void)
{
    static bool invoked = FALSE;
    if (invoked) return;
    invoked = TRUE;

    curl_global_cleanup();
}
void ato_bnet_setcurloptfn(ato_curloptfunction curloptfunction)
{
    _curloptfunction = curloptfunction;
}

size_t ato_bnet_timeout_default(ato_Ctx *ctx)
{
    ATO_IGNORE(ctx);
    return 60;
}

size_t ato_bnet_timeoutconnect_default(ato_Ctx *ctx)
{
    ATO_IGNORE(ctx);
    return 20;
}

void ato_bnet_create(ato_Ctx *ctx, ato_Net **obj, const char *url, const char *contenttype)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_bnet_create";
    int errcode = ATO_ERR_OK;
    ato_Net *net = NULL;
    ato_Log *log = NULL;

    ATO_CTX_FN_START(ctx);

    log = ato_ctx_log(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);
    assert(url != NULL);

    *obj = net = calloc(1, sizeof(ato_Net));
    assert(net != NULL);

    net->curl = curl_easy_init();
    if (net->curl == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_UNKNOWN, "Failed to initialise network connection");

    (void)ato_host(&(net->host), url);
    net->url = ato_strdup(url, 0);
    net->timeout = ato_bnet_timeout_default(ctx);
    net->timeoutconnect = ato_bnet_timeoutconnect_default(ctx);
    net->contenttype = ato_strdup(contenttype, 0);

    {
        const char *fmt = "url=%s" ", " "contenttype=%s" ", " "Timeout=%" PRINTF_SIZET "u" ", " "ConnectionTimeout=%" PRINTF_SIZET "u";
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, fmt, net->url, net->contenttype, net->timeout, net->timeoutconnect);
    }

    ATO_CTX_FN_END(ctx, errcode);
}


void ato_bnet_free(ato_Net *net)
{
    if (net == NULL) return;

    if (net->curl != NULL) { curl_easy_cleanup(net->curl); net->curl = NULL; }
    net->host = ato_free(net->host);
    net->url = ato_free(net->url);
    net->contenttype = ato_free(net->contenttype);
    _clear_resp_vars(net);
    _free_proxy(net);
    free(net);
}
