#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"


#include "atointernal.h"
#include "atotypes.h"
//#include "atoerrsoap.h"
#include "atonetd.h"

static const char *_library = ATO_BNET_LIBRARY;
static const char *_module = ATO_BNET_MODULE_NETD;
static unsigned long _moduleid = ATO_BNET_MODULEID_NETD;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
struct _ato_NetDataItem {
    ato_List *dataset;
    size_t totallen;
    ato_eNetDataType type;
    ato_eNetDataTypeEnc encoding;
    ato_String *content;
    char *contenttype;
    char *cid;
};

struct _ato_NetData {
    ato_List *itemlist;
    ato_Iterator *iter;
    int mode;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void *_freedataset(ato_NetDataItem *nditem)
{
    ato_ListNode *node = NULL;
    if (!nditem) return NULL;
    if (!nditem->dataset) return NULL;

    while ((node = ato_lst_next(nditem->dataset, node)) != NULL) {
        ato_str_free(ato_lst_value(node));
    }
    ato_lst_free(nditem->dataset);
    nditem->dataset = NULL;
    return NULL;
}

static void *_freeitem(ato_NetDataItem *nditem)
{
    if (nditem == NULL) return NULL;

    ato_free(nditem->cid);
    ato_free(nditem->contenttype);
    _freedataset(nditem);
    ato_str_free(nditem->content);
    free(nditem);
    return NULL;
}

/*********************************************************************************/
int ato__netd_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__netd_deinit(void)
{
}

void *ato_bnetd_free(ato_NetData *netdata)
{
    if (netdata != NULL) {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(netdata->itemlist, node)) != NULL)
            _freeitem(ato_lst_value(node));
        ato_lst_free(netdata->itemlist);
        ato_iter_free(netdata->iter);
        free(netdata);
    }
    return NULL;
}

void ato_bnetd_create(ato_NetData **obj)
{
    ato_NetData *netdata = NULL;

    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = netdata = calloc(1, sizeof(ato_NetData));
    assert(netdata != NULL);
    ato_lst_create(&(netdata->itemlist));
    netdata->iter = ato_iter_create_lst(netdata->itemlist);
}

/*********************************************************************************/
size_t ato_bnetd_count(ato_NetData *netdata)
{
    if (netdata == NULL) return 0;
    return ato_lst_count(netdata->itemlist);
}

void ato_bnetd_iterator(ato_NetData *netdata, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (netdata == NULL) return;
    *iter = ato_iter_create_lst(netdata->itemlist);
}

ato_NetDataItem *ato_bnetd_firstv(ato_NetData *netdata)
{
    if (netdata == NULL) return NULL;
    return ato_iter_firstv(netdata->iter);
}
ato_NetDataItem *ato_bnetd_nextv(ato_NetData *netdata)
{
    if (netdata == NULL) return NULL;
    return ato_iter_nextv(netdata->iter);
}

ato_NetDataItem *ato_bnetd_item(ato_NetData *netdata, ato_eNetDataType type)
{
    ato_ListNode *node = NULL;
    if (netdata == NULL) return NULL;
    while ((node = ato_lst_next(netdata->itemlist, node)) != NULL) {
        ato_NetDataItem *nditem = ato_lst_value(node);
        if (nditem && nditem->type == type)
            return nditem;
    }
    return NULL;
}

ato_NetDataItem *ato_bnetd_item_primary(ato_NetData *netdata)
{
    ato_ListNode *node = NULL;
    if (netdata == NULL) return NULL;
    while ((node = ato_lst_next(netdata->itemlist, node)) != NULL) {
        ato_NetDataItem *nditem = ato_lst_value(node);
        if (nditem && (nditem->type == ATO_BNETD_TYPE_SOAP || nditem->type == ATO_BNETD_TYPE_XOP || nditem->type == ATO_BNETD_TYPE_OTHER))
            return nditem;
    }
    return NULL;
}

/*********************************************************************************/
ato_NetDataItem *ato_bnetd_additem(ato_NetData *netdata)
{
    ato_NetDataItem *ndi = calloc(1, sizeof(ato_NetDataItem));
    assert(ndi != NULL);
    ato_lst_create(&(ndi->dataset));
    ato_lst_add(netdata->itemlist, NULL, ndi);
    return ndi;
}


/*********************************************************************************/
ato_eNetDataType ato_bnetd_type(ato_NetDataItem *nditem) { assert(nditem != NULL); return nditem->type; }
const char *ato_bnetd_contenttype(ato_NetDataItem *nditem) { assert(nditem != NULL); return nditem->contenttype; }
ato_eNetDataTypeEnc ato_bnetd_encoding(ato_NetDataItem *nditem) { assert(nditem != NULL); return nditem->encoding; }
const char *ato_bnetd_cid(ato_NetDataItem *nditem) { assert(nditem != NULL); return nditem->cid; }

const char *ato_bnetd_cidnoprefix(ato_NetDataItem *nditem)
{
    assert(nditem != NULL);
    if (ato_strstartswithi(nditem->cid, "cid:"))
        return nditem->cid + 4;
    else
        return nditem->cid;
}

void ato_bnetd_setcid(ato_NetDataItem *nditem, const char *cid)
{
    size_t len = 0;
    const char *end = strchr(cid, '>');
    if (end)
        len = (size_t)(end - cid);
    else
        len = strlen(cid);

    assert(nditem != NULL);
    nditem->cid = ato_free(nditem->cid);
    if (ato_strstartswithi(cid, "cid:")) {
        nditem->cid = ato_strdup(cid, 1);
        nditem->cid[len] = '\0';
    } else {
        nditem->cid = ato_strdup("cid:", len+1);
        memcpy(nditem->cid + 4, cid, len);
        nditem->cid[len+4] = '\0';
    }
    len = 0;
}

void ato_bnetd_settype(ato_NetDataItem *nditem, ato_eNetDataType type)
{
    assert(nditem != NULL);
    nditem->type = type;
}

void ato_bnetd_setcontenttype(ato_NetDataItem *nditem, const char *contenttype)
{
    assert(nditem != NULL);
    nditem->contenttype = ato_strdup(contenttype, 0);
}

void ato_bnetd_setencoding(ato_NetDataItem *nditem, ato_eNetDataTypeEnc encoding)
{
    assert(nditem != NULL);
    nditem->encoding = encoding;
}

void ato_bnetd_addcontent(ato_NetDataItem *nditem, ato_String *content)
{
    assert(nditem != NULL);
    nditem->totallen += ato_str_len(content);
    ato_lst_add(nditem->dataset, NULL, content);
}

static void _ato_bnetd_setcontent(ato_NetDataItem *nditem, const char *content, size_t len)
{
    char *lcontent = calloc(len+1, sizeof(char));
    memcpy(lcontent, content, len);
    ato_str_create(&(nditem->content), lcontent, len, FALSE);
}

ato_String *ato_bnetd_content(ato_NetDataItem *nditem)
{
    size_t offset = 0;
    ato_ListNode *node = NULL;
    char *chresponse = NULL;

    if (!nditem) return NULL;

    if (nditem->content)
        return nditem->content;

    if (!nditem->dataset)
        return NULL;

    chresponse = calloc(nditem->totallen + 1, sizeof(char));
    assert(chresponse != NULL);
    while ((node = ato_lst_next(nditem->dataset, node)) != NULL) {
        size_t len = 0;
        ato_String *datafragment = ato_lst_value(node);
        assert(datafragment != NULL);
        len = ato_str_len(datafragment);
        memcpy(chresponse + offset, ato_str_value(datafragment), len);
        offset += len;
        assert(offset <= nditem->totallen);
        ato_str_free(datafragment);
        ato_lst_setvalue(node, NULL);
    }

    assert(offset == nditem->totallen);
    ato_str_create(&(nditem->content), chresponse, nditem->totallen, FALSE);

    return nditem->content;
}

void ato_bnetd_clearcontent(ato_NetDataItem *nditem)
{
    _freedataset(nditem);
    ato_str_free(nditem->content);
    nditem->content = NULL;
}

//==================================================================================

// If there is more than one netdata item, then we have a multi-part mime, which in our
// case, means MTOM attachments indicated by ATO_BNETD_TYPE_MULTI_OTHER. For each attachment
// find its "cid" node and corresponding parent in the XML, delete the existing node,
// and add the attachment content as content to the parent. Before adding the content
// check if it needs converting to base64. ATO_BNETD_ENC_NONE means the original data
// encoding scheme was used which for attachments is binary (its possible for the
// data to already be base64 encoded by the sender although this should not happen in
// our case).
bool ato_bnetd_mtom_convertfrom(ato_Ctx *ctx, ato_NetData *netdata, ato_Xml *xml)
{
    static const char *xop = "//xop:Include[@href=\"$\"]";
    ato_NetDataItem *nditem = NULL;
    bool response_altered = FALSE;

    for (nditem = ato_bnetd_firstv(netdata); nditem; nditem = ato_bnetd_nextv(netdata)) {
        if (ato_bnetd_type(nditem) == ATO_BNETD_TYPE_MULTI_OTHER && ato_bnetd_cid(nditem) != NULL) {
            void *xnode = ato_xml_findnode(xml, NULL, xop, ato_bnetd_cid(nditem));
            ato_String *content = ato_bnetd_content(nditem);
            ato_eNetDataTypeEnc encoding = ato_bnetd_encoding(nditem);
            if (xnode && content) {
                ato_String *b64 = NULL;
                void *xpnode = ato_xml_nodeparent(xml, xnode);
                ato_xml_deletenode(xml, xnode);
                if (encoding == ATO_BNETD_ENC_NONE) {
                    ato_base64encode(ctx, &b64, content);
                    ato_xml_setvaluestr(xml, xpnode, b64);
                    ato_str_free(b64);
                } else {
                    ato_xml_setvalue(xml, xpnode, ato_str_value(content));
                }
                ato_bnetd_clearcontent(nditem);
                response_altered = TRUE;
            }
        }
    }
    ATO_IGNORE(ctx);
    return response_altered;
}

bool ato_bnetd_mtom_convertto(ato_Ctx *ctx, ato_NetData **nddataarg, ato_Xml *xml, const char *mtomxpath, size_t mtomthreshhold, const char *contenttypexpath)
{
    ato_String *content = NULL;
    ato_NetData *nddata = NULL;
    ato_NetDataItem *nditemprimary = NULL, *nditem = NULL;
    void *xlist = NULL;
    size_t count = 0, i = 0;
    ato_eNetDataType ndoutertype = ATO_BNETD_TYPE_SOAP;

    ato_bnetd_create(&nddata);
    assert(nddata != NULL);
    nditemprimary = ato_bnetd_additem(nddata);
    assert(nditemprimary != NULL);

    ATO_IGNORE(mtomxpath);
    xlist = ato_xml_listload(xml, NULL, mtomxpath, NULL);
    if (xlist != NULL && mtomthreshhold > 0) {
        count = ato_xml_listcount(xml, xlist);
        for (i = 0; i < count; i++) {
            char *b64data = NULL;
            ato_String *b64datastr = NULL;
            void *xnode = ato_xml_listitem(xml, xlist, i);

            ato_xml_value(xml, xnode, &b64data);
            ato_str_create(&b64datastr, b64data, ato_strlen(b64data), TRUE);
            if (ato_str_len(b64datastr) >= mtomthreshhold) {
                char *contenttype = NULL;
                char *xop = NULL;
                ato_String *binary = NULL;
                char cid[100];

                ndoutertype = ATO_BNETD_TYPE_XOP;

                assert(xnode != NULL);
                nditem = ato_bnetd_additem(nddata);
                strcpy(cid, "cid:");
                ato_new_pseudo_guid(ctx, cid+4);
                ato_bnetd_setcid(nditem, cid);
                ato_bnetd_settype(nditem, ATO_BNETD_TYPE_MULTI_OTHER);
                ato_xml_nodevalue(xml, xnode, contenttypexpath, &contenttype, ATO_XML_RAW_FALSE);
                assert(contenttype != NULL);
                ato_bnetd_setcontenttype(nditem, contenttype);
                free(contenttype);
                assert(b64data != NULL);
                ato_base64decode(ctx, &binary, b64datastr);
                ato_bnetd_addcontent(nditem, binary);
                ato_xml_setvalue(xml, xnode, NULL);
                xop = ato_strcat("<xop:Include xmlns:xop=\"http://www.w3.org/2004/08/xop/include\" href=\"", cid, "\"/>");  //e.g. <xop:Include href="cid:8b502a7b-fed1-4f86-be34-c585faff822c"/>
                ato_xml_createnodevalueraw(ctx, xml, xnode, NULL, NULL, xop);
                free(xop);
                //ato_xml_createnodevalueraw2(ctx, xml, xnode, xop, "xop", "http://www.w3.org/2004/08/xop/include");
            }

            ato_str_free(b64datastr);
        }

    }

    ato_xml_listfree(xml, xlist);

    ato_xml_save(ctx, xml, &content);

    ato_bnetd_settype(nditemprimary, ndoutertype);

#if defined(DEBUG) || defined(_DEBUG)
    {
        ato_String *debugbuf = NULL;
        ato_fileload(&debugbuf, "debugcsrrequest.xml");
        if (debugbuf) {
            ato_str_free(content);
            content = debugbuf;
        }
    }
#endif

    ato_bnetd_addcontent(nditemprimary, content);

    *nddataarg = nddata;

    return ndoutertype == ATO_BNETD_TYPE_XOP;
}

//======================================================================================
typedef enum {
    ATO_NET_MPBOUNDARY_FALSE,
    ATO_NET_MPBOUNDARY_TRUE,
    ATO_NET_MPBOUNDARY_TERMINATOR
} ato_eNetMPMimeBoundary;

static ato_eNetMPMimeBoundary _isboundary(const char *buffer, const char *boundary)
{
    if (*buffer == '-' && *(buffer+1) == '-' && ato_strstartswithi(buffer+2, boundary)) {
        const char c = buffer[strlen(boundary)+2];
        return (c == '-') ? ATO_NET_MPBOUNDARY_TERMINATOR : ATO_NET_MPBOUNDARY_TRUE;
    }
    return ATO_NET_MPBOUNDARY_FALSE;
}

#define ATO_NET_DP_MODE_HDR 0
#define ATO_NET_DP_MODE_DATA 1
#define ATO_NET_DP_MODE_DONE 2

static const char *_eoln(const char *line, int maxlen)
{
    int len = maxlen;
    while (len > 0 && *line != '\r' && *line != '\n') { line++; len--; }
    return (len > 0) ? line : NULL;
}

static bool _block_seperator(const char *line)
{
    return line && *line == '\r' && *(line+1) == '\n' && *(line+2) == '\r' && *(line+3) == '\n';
}

static ato_eNetDataType _map_nested_mimetype(ato_Ctx *ctx, char **mimestring, const char *chbuf)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_map_nested_mimetype";
    static const char *MimeType_xopxml = "application/xop+xml";
    static const char *MimeType_soapxml = "application/soap+xml";
    static const char *MimeType_multipart = "multipart/related";
    ato_eNetDataType mtypecode = ATO_BNETD_TYPE_UNKNOWN;
    const char *chbuf_end = chbuf;

    assert(chbuf != NULL);
    ATO_ASSERT_ISNOTALLOCATED(mimestring);
    while (*chbuf_end != '\r' && *chbuf_end != '\0' && *chbuf_end != ';')
        chbuf_end++;

    {
    size_t diff = (size_t)(chbuf_end - chbuf);
    *mimestring = ato_memdup(chbuf, diff, 1);
    assert(*mimestring != NULL);
    (*mimestring)[diff] = '\0';
    }

    if (ato_strstartswithi(chbuf, MimeType_soapxml)) {
        mtypecode = ATO_BNETD_TYPE_SOAP;
    } else if (ato_strstartswithi(chbuf, MimeType_xopxml)) {
        mtypecode = ATO_BNETD_TYPE_XOP;
    } else if (ato_strstartswithi(chbuf, MimeType_multipart)) {
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_NET, "Nested MultiPart MIME has been unexpectedly detected but is not supported.");
    } else {
        mtypecode = ATO_BNETD_TYPE_MULTI_OTHER;
    }

    return mtypecode;
}

static ato_eNetDataTypeEnc _map_encoding(const char *chbuf)
{
    static const char *Enc_qp = "quoted-printable";
    static const char *Enc_b64 = "base64";
    ato_eNetDataTypeEnc enc = ATO_BNETD_ENC_NONE;

    assert(chbuf != NULL);

    if (ato_strstartswithi(chbuf, Enc_b64)) {
        enc = ATO_BNETD_ENC_BASE64;
    } else if (ato_strstartswithi(chbuf, Enc_qp)) {
        enc = ATO_BNETD_ENC_QUOTED;
    } else {
        enc = ATO_BNETD_ENC_NONE;
    }

    return enc;
}

static const char *_processhdr(ato_Ctx *ctx, ato_NetData *netdata, ato_NetDataItem *nditem, const char *buffer, const char *boundary, size_t len, size_t *offset)
{
    static const char *ContentID = "Content-ID: <";
    static const char *ContentType = "Content-Type: ";
    static const char *ContentEncoding = "Content-Transfer-Encoding: ";
    const char *line = buffer, *lineend = NULL;
    ato_eNetMPMimeBoundary boundaryflag;

    while (*line == '\r' || *line == '\n') line++;
    boundaryflag = _isboundary(line, boundary);
    assert(boundaryflag != ATO_NET_MPBOUNDARY_FALSE);
    line = _eoln(line, 500);
    assert(line != NULL);
    while (!_block_seperator(line)) {
        while (*line == '\r' || *line == '\n') line++;
        lineend = _eoln(line, 500);
        assert(lineend != NULL);
        if (lineend - line > 0) {
            if (ato_strstartswithi(line, ContentType)) {
                char *mimestring = NULL;
                ato_bnetd_settype(nditem, _map_nested_mimetype(ctx, &mimestring, line + strlen(ContentType)));
                ato_bnetd_setcontenttype(nditem, mimestring);
                ato_free(mimestring);
            } else if (ato_strstartswithi(line, ContentEncoding)) {
                ato_bnetd_setencoding(nditem, _map_encoding(line + strlen(ContentEncoding)));
            } else if (ato_strstartswithi(line, ContentID)) {
                ato_bnetd_setcid(nditem, line + strlen(ContentID));
            }
        }
        line = lineend;
    }

    netdata->mode = ATO_NET_DP_MODE_DATA;

    assert(line - buffer >= 0);
    *offset = len - (size_t)(line - buffer);
    return line;
}

static const char *_processdata(ato_NetData *netdata, ato_NetDataItem *nditem, const char *buffer, const char *boundary, size_t len, size_t *offset)
{
    ato_eNetMPMimeBoundary boundaryflag = ATO_NET_MPBOUNDARY_FALSE;
    const char *content = buffer, *contentend = NULL;

    while (*content == '\r' || *content == '\n') content++;

    contentend = content;
    while (boundaryflag == ATO_NET_MPBOUNDARY_FALSE) {
        while (*contentend != '\r' && *contentend != '\n') contentend++;
        while (*contentend == '\r' || *contentend == '\n') contentend++;
        boundaryflag = _isboundary(contentend, boundary);
    }
    contentend -= 2;

    assert(contentend - content > 0);
    _ato_bnetd_setcontent(nditem, content, (size_t)(contentend - content));

    if (boundaryflag == ATO_NET_MPBOUNDARY_TERMINATOR)
        netdata->mode = ATO_NET_DP_MODE_DONE;
    else
        netdata->mode = ATO_NET_DP_MODE_HDR;

    *offset = len - (size_t)(contentend - buffer);
    return contentend;
}

void ato_bnetd_processmpresponse(ato_Ctx *ctx, ato_NetData *netdata, const char *boundaryid)
{
    ato_NetDataItem *nditem = ato_lst_value(ato_lst_next(netdata->itemlist, NULL));
    const char *buffer = NULL;
    size_t len = 0;
    size_t offset = 0;

    buffer = ato_str_value(nditem->content);
    len = ato_str_len(nditem->content);

    while (netdata->mode != ATO_NET_DP_MODE_DONE) {
        if (netdata->mode == ATO_NET_DP_MODE_HDR) {
            ato_bnetd_content(nditem);
            nditem = ato_bnetd_additem(netdata);
            buffer = _processhdr(ctx, netdata, nditem, buffer, boundaryid, len, &offset);
        } else if (netdata->mode == ATO_NET_DP_MODE_DATA) {
            buffer = _processdata(netdata, nditem, buffer, boundaryid, len, &offset);
        }
    }
}
