#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>
#include <ctype.h>
#include <sys/stat.h>

#include <curl/curl.h>

#if defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64)
#include <windows.h>
#define SLEEP(x) Sleep((x)*1000)
#else
#include <unistd.h>
#define SLEEP(x) sleep((x))
#endif

#include "akversion.h"

#include "atobase/all.h"
#include "atobnet/all.h"
#include "atoakm/all.h"
#include "atostm/all.h"

#include "../common/util.h"
#include "../common/log.h"

#include "sys.h"
#include "misc.h"

ato_Ctx *g_ctx = NULL;
ato_Log *g_log = NULL;
ato_eLoglevel arg_loglevel = ATO_LOG_ALL;

static const char *_DEFAULT_ksdir = "."; // current folder
static const char *_DEFAULT_kspwd = "Password1!";
static const char *_DEFAULT_ksalias = "ABRP:27809366375_10000022";

#if defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64)
#else
#endif

static const char *arg_organisation = NULL;
static const char *arg_installprefix = NULL;
static bool arg_logfd = FALSE;
static const char *arg_logpath = NULL;
static const char *arg_logcr = NULL;
static const char *arg_netlogflag = NULL;
static const char *arg_proxyurl = NULL;
static const char *arg_proxyauthtype = NULL;
static const char *arg_proxybypass = NULL;
static const char *arg_proxyuser = NULL;
static const char *arg_proxypwd = NULL;
static const char *arg_proxypwdfile = NULL;
static const char *arg_sslversion = NULL;
static const char *arg_sslcainfo = NULL;

static const char *arg_ksdir = NULL;
static const char *arg_ksfile = NULL;
static const char *arg_kspwd = NULL;
static const char *arg_ksalias = NULL;
static const char *arg_chpwd = NULL;
static size_t arg_creds = 3;
static bool arg_crypto = FALSE;
static const char *arg_crexpiryperiod = NULL;
static const char *arg_akmtests = NULL;

static int arg_exptime = 0;
static size_t arg_nettimeout = 120;
static size_t arg_connecttimeout = 30;
static bool arg_forcestmerror = FALSE;
static const char *arg_securitytokenfile = NULL;
static const char *arg_stmvars = NULL;
static int arg_stmrepeat = 0;
static const char *arg_stmtemplate = NULL;
static const char *arg_stmservice = NULL;
static const char *arg_relyingparty = NULL;
static const char *arg_stm_rp_p8 = NULL;
static bool arg_stm_forceretry = FALSE;
static const char *arg_stm_headermin_adj = "0";
static const char *arg_stm_headermin_ttl = "5";
static const char *arg_stm_bodymin_adj = "0";
static const char *arg_stm_bodymin_ttl = "5";

static void _addargs(void)
{
        ato_arg_add("organisation", TRUE, NULL, NULL, "Name of the organisation or person running this");
        ato_arg_add("installprefix", FALSE, NULL, NULL, "Where to look for dependent library resources, if any");
        ato_arg_add("logfd", FALSE, NULL, NULL, "Some modules will dump data to files for debugging purposes");
        ato_arg_add("logpath", FALSE, "messages.log", NULL, "Set the log path");
        ato_arg_add("loglevel", FALSE, "d", "off|fatal|err|warn|info|debug|trace|all", "Set the loglevel - only need first letter");
        ato_arg_add("logcr", FALSE, "d", "off|fatal|err|warn|info|debug|trace|all", "Set the loglevel for the credential module");
        ato_arg_add("netlogflag", FALSE, NULL, NULL, "the network library has additional loglevel settings");
        ato_arg_add("proxyurl", FALSE, NULL, NULL, "proxy server url");
        ato_arg_add("proxyauthtype", FALSE, NULL, NULL, "proxy server authentication type: any (default); basic; digest; ntlm");
        ato_arg_add("proxybypass", FALSE, NULL, NULL, "proxy server bypass list");
        ato_arg_add("proxyuser", FALSE, NULL, NULL, "proxy server user name - if not supplied read the settings from the current config file context");
        ato_arg_add("proxypwd", FALSE, NULL, NULL, "proxy server user password");
        ato_arg_add("proxypwdfile", FALSE, NULL, NULL, "read proxyuser (1st line) and proxypwd (2nd line) from this file");
        ato_arg_add("sslversion", FALSE, NULL, NULL, "CURL options for SSL/TLS(DEFAULT,TLSv1,SSLv2,SSLv3,TLSv1_0,TLSv1_1,TLSv1_2)");
        ato_arg_add("sslcainfo", FALSE, NULL, NULL, "Path to CURL CA store or 'off' to diable SSL - see CURL option CURLOPT_CAINFO");

        ato_arg_add("ksdir", FALSE, ".", NULL, "Folder to look for the kestore");
        ato_arg_add("ksfile", FALSE, ato_akm_default_ksfile(), NULL, "Filename of the keystore");
        ato_arg_add("kspwd", FALSE, "Password1!", NULL, "Password for the keystore");
        ato_arg_add("ksalias", FALSE, NULL, NULL, "Alias to load from the keystore");
        ato_arg_add("chpwd", FALSE, NULL, NULL, "keystore password to change to");
        ato_arg_add("creds", FALSE, "3", NULL, "Number of credentials to list");
        ato_arg_add("crypto", FALSE, NULL, NULL, "Dump private key details");
        ato_arg_add("crexpiryperiod", FALSE, NULL, NULL, "Credential expiry period in months - minimum 14");
        ato_arg_add("akmtests", FALSE, NULL, NULL, "AKM tests to run: ',dumpks,loadsaveks,renewcred,changepwd,copycred,deleteallcrs,' or all if not supplied");

        ato_arg_add("exptime", FALSE, NULL, NULL, "Expiry time for SecurityTokens");
        ato_arg_add("nettimeout", FALSE, "120", NULL, "Total seconds to wait for transaction to complete");
        ato_arg_add("connecttimeout", FALSE, "30", NULL, "Total seconds to wait for a connection");
        ato_arg_add("forcestmerror", FALSE, NULL, NULL, "Force an error in the SecurityToken request");
        ato_arg_add("securitytokenfile", FALSE, NULL, NULL, "security token file used to load token into cache");
        ato_arg_add("stmvars", FALSE, NULL, NULL, "Template vars to use");
        ato_arg_add("stmrepeat", FALSE, NULL, NULL, "Repeat call to STS n times");
        ato_arg_add("stmtemplate", FALSE, ato_stm_default_templatefile(), NULL, "STM template file");
        ato_arg_add("stmservice", FALSE, NULL, NULL, "URL for STS service");
        ato_arg_add("relyingparty", FALSE, NULL, NULL, "Relying Party");
        ato_arg_add("stm.rp_p8", FALSE, "rstr.p8.pem", NULL, "Relying Party decryption key to descrypt the STM RSTR");
        ato_arg_add("stm.forceRetry", FALSE, NULL, NULL, "The very first RST is replayed repeatedly");
        ato_arg_add("stm.headermin_adj", FALSE, "0", NULL, "Adjust header created time");
        ato_arg_add("stm.bodymin_adj", FALSE, "0", NULL, "Adjust body created time");
        ato_arg_add("stm.headermin_ttl", FALSE, "5", NULL, "Header expiry time in minutes");
        ato_arg_add("stm.bodymin_ttl", FALSE, "5", NULL, "Life of STM token in minutes");
}

static void _initargs(void)
{
        arg_organisation = ato_arg_value("organisation");
        arg_installprefix = ato_arg_value("installprefix");;
        arg_logfd = ato_arg_value_bool("logfd");
        arg_logpath = ato_arg_value("logpath");
        arg_loglevel = ato_log_name2level(ato_arg_value("loglevel"));
        arg_logcr = ato_arg_value("logcr");
        arg_netlogflag = ato_arg_value("netlogflag");
        arg_proxyurl = ato_arg_value("proxyurl");
        arg_proxyauthtype = ato_arg_value("proxyauthtype");
        arg_proxybypass = ato_arg_value("proxybypass");
        arg_proxyuser = ato_arg_value("proxyuser");
        arg_proxypwd = ato_arg_value("proxypwd");
        arg_proxypwdfile = ato_arg_value("proxypwdfile");
        arg_sslversion = ato_arg_value("sslversion");
        arg_sslcainfo = ato_arg_value("sslcainfo");

        arg_ksdir = ato_arg_value("ksdir");
        arg_ksfile = ato_arg_value("ksfile");
        arg_kspwd = ato_arg_value("kspwd");
        arg_ksalias = ato_arg_value("ksalias");
        arg_chpwd = ato_arg_value("chpwd");
        arg_creds = (size_t) ato_arg_value_int("creds");
        arg_crypto = ato_arg_value_bool("crypto");
        arg_crexpiryperiod = ato_arg_value("crexpiryperiod");
        arg_akmtests = ato_arg_value("akmtests");

        arg_exptime = ato_arg_value_int("exptime");
        arg_nettimeout = (size_t) ato_arg_value_int("nettimeout");
        arg_connecttimeout = (size_t) ato_arg_value_int("connecttimeout");
        arg_forcestmerror = ato_arg_value_bool("forcestmerror");
        arg_securitytokenfile = ato_arg_value("securitytokenfile");
        arg_stmvars = ato_arg_value("stmvars");
        arg_stmrepeat = ato_arg_value_int("stmrepeat");
        arg_stmtemplate = ato_arg_value("stmtemplate");
        arg_stmservice = ato_arg_value("stmservice");
        arg_relyingparty = ato_arg_value("relyingparty");
        arg_stm_rp_p8 = ato_arg_value("stm.rp_p8");
        arg_stm_forceretry = ato_arg_value_bool("stm.forceretry");
        arg_stm_headermin_adj = ato_arg_value("stm.headermin_adj");
        arg_stm_headermin_ttl = ato_arg_value("stm.headermin_ttl");
        arg_stm_bodymin_adj = ato_arg_value("stm.bodymin_adj");
        arg_stm_bodymin_ttl = ato_arg_value("stm.bodymin_ttl");
}

static bool _streqi(const char *src, const char *src2)
{
    if (src == NULL && src2 == NULL)
        return TRUE;
    if ((src == NULL && src2 != NULL) || (src != NULL && src2 == NULL))
        return FALSE;
    if (strlen(src) != strlen(src2))
        return FALSE;
    while (*src2 != '\0') {
        if (tolower((int)*src) != tolower((int)*src2))
            return FALSE;
        src++; src2++;
    }
    return TRUE;
}

static int curloptfunction(void *curl)
{
    static bool logged = FALSE;
    CURLcode rc = CURLE_OK;
    if (curl == NULL) { return rc; }
    if (!logged) printf("Checking CURL options: arg_sslversion=[%s] arg_sslcainfo=[%s]\n", arg_sslversion, arg_sslcainfo);
    if (arg_sslversion != NULL) {
        if (!logged) printf("Setting CURLOPT_SSLVERSION to '%s'\n", arg_sslversion);
        if (_streqi(arg_sslversion, "default")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_DEFAULT);
        } else if (_streqi(arg_sslversion, "tlsv1")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1);
        } else if (_streqi(arg_sslversion, "sslv2")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv2);
        } else if (_streqi(arg_sslversion, "sslv3")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv3);
        } else if (_streqi(arg_sslversion, "tlsv1_0")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_0);
        } else if (_streqi(arg_sslversion, "tlsv1_1")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_1);
        } else if (_streqi(arg_sslversion, "tlsv1_2")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);
        } else {
            printf("***Setting CURLOPT_SSLVERSION FAILED (ignoring)***");
        }
    }
    if (arg_sslcainfo != NULL) {
        if (strcmp(arg_sslcainfo, "off") == 0) {
            if (!logged) printf("%s\n", "Disabling SSL validation");
            rc = curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
            rc = curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
        } else {
            // careful - we're not catering to platform independence here
            // so the arg_sslcainfo path is OS specific.
            // Don't bother setting if the file doesn't exist
            struct stat sb;
            if (stat(arg_sslcainfo, &sb) == 0) {
                rc = curl_easy_setopt(curl, CURLOPT_CAINFO, arg_sslcainfo);
            }
        }
    }
    logged = TRUE;
    return rc;
}

static bool _getcrinfo(int *ecode, ato_Ctx *ctx, ato_Credential *cr, ato_String **certificate, ato_String **privatekey, const char *pwd)
{
    int errcode = ATO_ERR_OK;
    bool ok = FALSE;

    if (ato_cr_isrenewable(ctx, cr)) {
        //if ato_cr_renew(ctx, cr, pwd) != ATO_ERR_OK) {
        //  // Ignore any renewal errors and continue.
        //  printf("Warning: renewal for '%s' failed\n", alias);
        //}
    }

    if ((errcode = ato_cr_certificate(ctx, cr, certificate)) != ATO_ERR_OK ||
            (errcode = ato_cr_privatekey(ctx, cr, privatekey, pwd)) != ATO_ERR_OK)
        goto finally;

    ok = TRUE;
finally:
    *ecode = errcode;
    return ok;
}

static void _process_error(ato_Ctx *ctx)
{
    ato_Err *err = ato_ctx_err(ctx);
    int errcode = ato_err_code(err);
    if (errcode == ATO_ERR_OK)
        return;
    printf("\n");
    printf("-----------------------------------------\n");

    switch (errcode) {
        case ATO_STM_ERR_GENERAL:
            printf("STM General error: ");
            break;
        case ATO_STM_ERR_NETSENDER:
            printf("STM Sender error: ");
            break;
        case ATO_STM_ERR_NETRECEIVER:
            printf("STM Receiver error: ");
            break;
        case ATO_STM_ERR_NETUNAVAILABLE:
            printf("STM Remote Service not available: ");
            break;
        case ATO_STM_ERR_NETCOMMS:
            printf("STM Network error: ");
            break;
        case ATO_STM_ERR_NETTIMEOUT:
            printf("STM Remote Service timeout: ");
            break;
    }

    for (; err; err = ato_err_inner(err)) {
        errcode = ato_err_code(err);
        printf("%s\n", ato_err_msg(err));
        if (errcode == ATO_ERR_NET_SOAP) {
            ato_ErrSoap *soaperr = ato_err_custom(err);
            if (soaperr && ato_errsoap_issoaperr(soaperr)) {
                ato_ErrSoapCode *code = NULL;
                printf("  node=[%s]\n", ato_errsoap_node(soaperr));
                printf("  reason=[%s]\n", ato_errsoap_reason(soaperr));
                printf("  detail=[%s]\n", ato_errsoap_detail(soaperr));
                for (code = ato_errsoap_code_firstv(soaperr); code; code = ato_errsoap_code_nextv(soaperr)) {
                    printf("  code value=[%s] ns=[%s]\n", ato_errsoap_code_value(code), ato_errsoap_code_ns(code));
                }
            } else {
                printf("  SOAP error details not available\n");
            }
        }
        printf("-----------------------------------------\n");
    }
    printf("\n");
}

static bool _stm_var_extract(char *ssid, char *secondparty)
{
    size_t len = 0;
    char *end = strchr(arg_stmvars, ',');
    if (end == NULL) { return FALSE; }
    len = end - arg_stmvars;
    strncpy(ssid, arg_stmvars, len);
    ssid[len] = '\0';
    strcpy(secondparty, (end+1));
    return TRUE;
}

static int _stm_load(ato_Stm **stm, ato_Credential *cr)
{
    static char id[50];
    static char ssid[50];
    static char firstparty[50];
    static char secondparty[50];
    int errcode = ATO_ERR_OK;
    char *buf = NULL;

    printf("Loading template %s\n", arg_stmtemplate);
    if ((errcode = ato_stm_loadtemplate(g_ctx, &buf, arg_stmtemplate, NULL)) != ATO_ERR_OK)
        goto finally;
    printf("Creating STM for %s\n", arg_stmservice);
    if ((errcode = ato_stm_create(g_ctx, stm, arg_stmservice, buf)) != ATO_ERR_OK)
        goto finally;
    if (arg_stmvars != NULL) {
        // Seperated just for test
        const char *vars1[] = {
            "dg_OSPfor_ssid", ssid,
            "dg_OSPfor_firstparty", firstparty,
            NULL, NULL
        };
        const char *vars2[] = {
            "dg_OSPfor_id", id,
            "dg_OSPfor_secondparty", secondparty,
            NULL, NULL
        };
        ato_new_pseudo_guid(g_ctx, id);
        strcpy(firstparty, ato_cr_abn(cr));
        if (!_stm_var_extract(ssid, secondparty)) {
            printf("\narg_stmvar format must be ssid,secondparty\n");
            errcode = -1;
            goto finally;
        }

        printf("Setting template vars\n");
        //printf("%s=%s\n", vars1[0], vars1[1]);
        //printf("%s=%s\n", vars1[2], vars1[3]);
        //printf("%s=%s\n", vars2[0], vars2[1]);
        //printf("%s=%s\n", vars2[2], vars2[3]);
        if ((errcode = ato_stm_set_template_vars(g_ctx, *stm, vars1)) != ATO_ERR_OK)
            goto finally;
        if ((errcode = ato_stm_set_template_vars(g_ctx, *stm, vars2)) != ATO_ERR_OK)
            goto finally;
    }

finally:
    ato_free(buf);
    return errcode;
}

static int _mark_secs(void) {
    time_t now = time(NULL);
    struct tm *tms = gmtime(&now);
    return tms->tm_sec + tms->tm_min * 60 + tms->tm_hour * 60 * 60;
}

static bool _run_test(void)
{
    bool ok = FALSE;
    int errcode = ATO_ERR_OK;
    ato_StmTkn *st = NULL;
    ato_Stm *stm = NULL;
    ato_Keystore *ks = NULL;
    ato_Credential *cr = NULL;
    ato_String *certificate = NULL;
    ato_String *privatekey = NULL;
    const char *ksdir = NULL, *kspwd = NULL, *ksalias = NULL;
    size_t nettimeout = 0;
    size_t connecttimeout = 0;
    int stmrepeat = arg_stmrepeat;

    printf("\n\n");

    ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
    kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
    ksalias = (arg_ksalias != NULL ? arg_ksalias : _DEFAULT_ksalias);

    if ((errcode = sys_load(g_ctx, &ks, ksdir, arg_ksfile)) != ATO_ERR_OK) {
        goto finally;
    }

    cr = ato_ks_credential(ks, ksalias);
    if (cr == NULL) {
        printf("Failed to find alias=%s\n", ksalias);
        goto finally;
    }

    if (!_getcrinfo(&errcode, g_ctx, cr, &certificate, &privatekey, kspwd)) {
        goto finally;
    }

    if ((errcode = _stm_load(&stm, cr)) != ATO_ERR_OK) {
        goto finally;
    }

    //printf("ato_stm_default_templatedir is: %s\n", ato_stm_default_templatedir());
    printf("ato_stm_default_templatefile is: %s\n", ato_stm_default_templatefile());
    //printf("ato_stm_default_templatepath is: %s\n", ato_stm_default_templatepath());

    nettimeout = ato_stm_timeout(g_ctx, stm);
    connecttimeout = ato_stm_timeoutconnect(g_ctx, stm);
    printf("Total network time out value is %" PRINTF_SIZET "u seconds, connect time out is %" PRINTF_SIZET "u.\n", nettimeout, connecttimeout);
    ato_stm_settimeout(g_ctx, stm, arg_nettimeout);
    nettimeout = ato_stm_timeout(g_ctx, stm);
    ato_stm_settimeoutconnect (g_ctx, stm, arg_connecttimeout);
    connecttimeout = ato_stm_timeoutconnect(g_ctx, stm);
    if(nettimeout != arg_nettimeout || connecttimeout != arg_connecttimeout){
        printf("time outs failed to stick!\n");
        goto finally;
    }
    printf("Total network time out value is NOW %" PRINTF_SIZET "u seconds and connect time out %" PRINTF_SIZET "u.\n", nettimeout, connecttimeout);

    if (arg_securitytokenfile != NULL && *arg_securitytokenfile != '\0') {
        char *buf = NULL;
        ato_String *atorstr = NULL;
        size_t len = utl_loadfile(&buf, arg_securitytokenfile, FALSE);
        assert(len > 0);
        ato_str_create(&atorstr, buf, len, FALSE);
        ato_stm_loadst(g_ctx, stm, &st, atorstr);
        ato_str_free(atorstr);
    } else {
        int start = _mark_secs();

        //ato_stm_setexpiryinterval(stm, 7); //Will set expiry time minutes.
        printf("New sts issue request for RP %s\n", arg_relyingparty);
        if (arg_forcestmerror) {
            if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, "ardvark")) != ATO_ERR_OK)
                goto finally;
        } else {
            if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, arg_relyingparty)) != ATO_ERR_OK)
                goto finally;
        }
        fflush(NULL);
        printf("\n\n");

        for (; stmrepeat > 0; stmrepeat--) {
            printf("------------- RST repeat countdown %d\n", stmrepeat);

            ////This will reuse cached value - based on hash(template+certificate+relyingpartyurl)
            //printf("Try cache: sts issue request. This should use a cached token. See log.\n");
            //if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, NULL)) != ATO_ERR_OK)
            //    goto finally;
            //printf("Purge cache.\n");
            //ato_stm_purgecache(g_ctx, TRUE);
            //fflush(NULL);
            //printf("\n\n");

            st = NULL;
            //printf("Cache cleared: sts issue request. This should request a new token. See log.\n");
            if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, arg_relyingparty)) != ATO_ERR_OK)
                goto finally;
        }
        fflush(NULL);

        printf("Duration (secs): %d\n", _mark_secs() - start);


        printf("\n\n");

        {
            // Write "SecurityToken" to file - this shows it is "(de)seriablizable".
            ato_String *pt = ato_stmtkn_xml(st);
            const char *ato_rstr_filename = "AtoSecurityToken.xml";
            FILE *fp = fopen(ato_rstr_filename, "w");
            if (fp != NULL) {
                fwrite(ato_str_value(pt), sizeof(char), ato_str_len(pt), fp);
                fclose(fp);
            }
            ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "Serialised ATO RSTR written to file %s", ato_rstr_filename);
        }
    }

    ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "stkey=%s", ato_stmtkn_key(st));
    ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "SAMLID=%s", ato_stmtkn_samlid(st));

    {
        struct tm *time_exp;
        char buffer[50];
        time_t now;
        time_t tt = ato_stmtkn_expirytime(st);
        time(&now);
        memset(buffer, 0, 50);
        time_exp = localtime(&tt);
        strftime(buffer, 50, "%Y%m%d-%H%M%S", time_exp);
        printf("ST expiry time is: %s  (%d)\n", buffer, (int)((tt - now) / 60));
    }

    if((arg_exptime > 0) && (arg_exptime<70)){
        printf("Now waiting %d minutes for the token to expire.... or not. MIN expiry time is 5 and MAX is 30.\n", arg_exptime+4);
        SLEEP((unsigned int)(60*(arg_exptime+4)));
        printf("now doing ato_stmtkn_isexpired.\n");
        if(TRUE == ato_stmtkn_isexpired (st)){
        printf("Fourth sts request again with same identity etc.. Check for a log entry for new sts token.\n");
        if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, NULL)) != ATO_ERR_OK)
            goto finally;
        }
        else{
            printf("The ato_stmtkn_isexpired check did not return TRUE.\n");
            printf("Attempting ato_stm_stsissue - will it use the expired token?\n");
            if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, NULL)) != ATO_ERR_OK)
            goto finally;
        }
    }

    ok = TRUE;
finally:
    ato_str_free(certificate);
    ato_str_free(privatekey);
    ato_stm_free(g_ctx, stm);
    sys_free(g_ctx, ks);
    if (errcode != ATO_ERR_OK) {
        _process_error(g_ctx);
        ato_errh_handler(g_ctx, errcode);
    }
    return ok;
}

static bool _init_tests(void)
{
    int errcode = ATO_ERR_OK;

        if (arg_organisation == NULL || arg_organisation[0] == '\0') { errcode = -1; goto finally; }
        ato_si_set(arg_organisation, _EXE_DESCRIPTION, _EXE_VERSION, _EXE_TIMESTAMP, _EXE_SRC, _EXE_HASH);
        ato_ctx_create(&g_ctx,  NULL);
        errcode = init(g_ctx);

    if (errcode != ATO_ERR_OK) {
        ato_Err *err = ato_ctx_err(g_ctx);
        printf("Error: %d (%s) %s\n", ato_err_code(err), ato_err_codename(err), ato_err_msg(err));
        goto finally;
    }

    ato_log_setlogfn(customlogger); //ato_log_setlogfn(verysimplelogger);

    g_log = ato_ctx_log_set(g_ctx, "", "2", NULL, "w");

    // Set log level for all modules
    ato_setloglevel(NULL, 0, arg_loglevel);
    // Override CR as it can generate too much if there are many credentials
    ato_setloglevel(ATO_AKM_LIBRARY, ATO_AKM_MODULEID_CR, ATO_LOG_WARN);
    // Some modules dump extra files useful for debugging
    ato_setlogfiledump(NULL, 0, arg_logfd);
    // Extra log settings for application network traffic
    if (arg_netlogflag) {
        ato_bnet_setdebugflag(ato_bnet_names2flag(arg_netlogflag));
    }

    ato_ctx_cachec_set(g_ctx, "stm.rp_p8", arg_stm_rp_p8);
    ato_ctx_cachec_set(g_ctx, "stm.headermin_adj", arg_stm_headermin_adj);
    ato_ctx_cachec_set(g_ctx, "stm.bodymin_adj", arg_stm_bodymin_adj);
    ato_ctx_cachec_set(g_ctx, "stm.headermin_ttl", arg_stm_headermin_ttl);
    ato_ctx_cachec_set(g_ctx, "stm.bodymin_ttl", arg_stm_bodymin_ttl);

    if (arg_stm_forceretry) {
        ato_ctx_cachec_set(g_ctx, "stm.forceretry", "1");
    }

finally:
    return errcode == ATO_ERR_OK;
}

static void _deinit_tests(void)
{
    deinit(g_ctx);
}

static bool _run_tests(void)
{
    bool ok = FALSE;
    bool inited = FALSE;

    if ((inited = _init_tests()) == FALSE)
        goto finally;

    ato_log_start(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", ato_si_product());

    ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "BASE Version code=%s", ato_base_version());
    ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "BASE Version code=%s", ato_bnet_version());
    ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "AKM Version code=%s", ato_akm_version());
    ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "STM Version code=%s", ato_stm_version());

    ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "",
        "SoftwareInfo organisation='%s', product='%s', version='%s', timestamp='%s', source='%s'",
        ato_si_organisation(), ato_si_product(), ato_si_version(), ato_si_timestamp(), ato_si_source());
    printf("%s\n", ato_si_info());

    // this overrides the default search behaviour when loading the STM template file - forces it to look here.
    //ato_stm_setinstallprefix(arg_installprefix);

    if (!_run_test())
        goto finally;

    ok = TRUE;
finally:
    if (inited) ato_log_end(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", FALSE, ato_si_product());
    _deinit_tests();
    return ok;
}

static void _forceemptyline(void)
{
    printf("\n"); printf("\n"); printf(" \n");
}

bool useCurlFn = TRUE;
static bool _setargs(int argc, const char *argv[])
{
    bool argres = FALSE;
    _addargs();
    if (argc > 1) {
        argres = ato_arg_process(argc, argv);
        if (argres) {
            _initargs();
            _forceemptyline();
            argres = ato_arg_check();
            _forceemptyline();
        }
                argres = (ato_arg_value_bool("help") == FALSE);
    }

    if (useCurlFn) {
        ato_bnet_setcurloptfn(curloptfunction);
    }
    return argres;
}

int main(int argc, const char *argv[])
{
    bool ok = FALSE;
    setbuf(stdout, NULL);
    if (_setargs(argc, argv)) {
        ato_arg_dump();
        _forceemptyline();
        ok = _run_tests();
        if (!ok) {
            _forceemptyline();
            printf("An error occurred - please check logs if any\n");
        }
    } else {
        ato_arg_usage(_EXE_NAME);
    }
    ato_arg_free();
    _forceemptyline();
    return ok == TRUE ? 0: -1;
}
