macro(AK_HALTONERR ISERR MSG)
    if (${ISERR})
        message(STATUS " ")
        message(STATUS "***ERROR*** ${MSG}.")
        message(STATUS " ")
        message(FATAL_ERROR " ")
    endif()
endmacro(AK_HALTONERR)

macro(AK_HALT MSG)
    AK_HALTONERR(TRUE "${MSG}")
endmacro(AK_HALT)

macro(AK_ENSURE_OUT_OF_SOURCE_BUILD)
    STRING(COMPARE EQUAL "${CMAKE_SOURCE_DIR}" "${CMAKE_BINARY_DIR}" insource)
    GET_FILENAME_COMPONENT(PARENTDIR ${CMAKE_SOURCE_DIR} PATH)
    STRING(COMPARE EQUAL "${CMAKE_SOURCE_DIR}" "${PARENTDIR}" insourcesubdir)
    IF(insource OR insourcesubdir)
        AK_HALTONERR(TRUE "This project requires an out of source build")
    ENDIF(insource OR insourcesubdir)
endmacro(AK_ENSURE_OUT_OF_SOURCE_BUILD)

# Adapted from From http://www.cmake.org/pipermail/cmake/2010-November/041034.html
# Don't care about format
macro(NOW RESULT)
    if (WIN32)
        execute_process(COMMAND "cmd" "/C date /T" OUTPUT_VARIABLE tvar OUTPUT_STRIP_TRAILING_WHITESPACE)
        set(${RESULT} "WIN32: ${tvar}")
        execute_process(COMMAND "cmd" "/C time /T" OUTPUT_VARIABLE tvar OUTPUT_STRIP_TRAILING_WHITESPACE)
        set(${RESULT} "${${RESULT}} ${tvar}")
    elseif (UNIX OR MSYS)
        execute_process(COMMAND "date" "-u" OUTPUT_VARIABLE tvar OUTPUT_STRIP_TRAILING_WHITESPACE)
        set(${RESULT} "UNIX: ${tvar}")
    else()
        execute_process(COMMAND "date" "-u" OUTPUT_VARIABLE tvar OUTPUT_STRIP_TRAILING_WHITESPACE)
        set(${RESULT} "XXX: ${tvar}")
        #message(SEND_ERROR "date not implemented")
        #set(${RESULT} "XXX: 000000")
    endif()
endmacro(NOW)

# https://stackoverflow.com/questions/32183975/how-to-print-all-the-properties-of-a-target-in-cmake
execute_process(COMMAND cmake --help-property-list OUTPUT_VARIABLE AK_PROPERTY_LIST)
STRING(REGEX REPLACE ";" "\\\\;" AK_PROPERTY_LIST "${AK_PROPERTY_LIST}")
STRING(REGEX REPLACE "\n" ";" AK_PROPERTY_LIST "${AK_PROPERTY_LIST}")

function(AK_check_env)
    execute_process(COMMAND bash 7z RESULT_VARIABLE AK_RESULT OUTPUT_QUIET)
    if (NOT ${AK_RESULT} EQUAL 0)
        AK_HALT("RESULT=${AK_RESULT}: 7z (p7zip) not found or invalid version")
    endif()
endfunction(AK_check_env)

function(AK_print_properties_global)
    set(props ${AK_PROPERTY_LIST})
    foreach (prop ${props})
        if (NOT "${${prop}}" STREQUAL "")
            message(STATUS "${prop}=${${prop}}")
        endif()
    endforeach()
endfunction(AK_print_properties_global)

function(AK_print_properties_dir)
    get_cmake_property(props VARIABLES)
    foreach (prop ${props})
        if (NOT "${prop}" STREQUAL "AK_PROPERTY_LIST")
            if (NOT "${${prop}}" MATCHES ".*\n.*" AND NOT "${${prop}}" STREQUAL "")
                message(STATUS "${prop}=${${prop}}")
            endif()
        endif()
    endforeach()
endfunction(AK_print_properties_dir)

function(AK_print_properties_target tgt)
    if(NOT TARGET ${tgt})
      message("There is no target named '${tgt}'")
      return()
    endif()
    foreach (prop ${AK_PROPERTY_LIST})
        string(REPLACE "<CONFIG>" "${CMAKE_BUILD_TYPE}" prop ${prop})
        if(prop STREQUAL "LOCATION" OR prop MATCHES "^LOCATION_" OR prop MATCHES "_LOCATION$")
            continue()
        endif()
        get_property(propval TARGET ${tgt} PROPERTY ${prop})
        if (propval)
            get_target_property(propval ${tgt} ${prop})
            message ("${tgt} ${prop} = ${propval}")
        endif()
    endforeach(prop)
endfunction(AK_print_properties_target)

function(AK_Path2Nix result path)
  string(REGEX REPLACE "^(.):" "/\\1" newpath "${path}")
  string(REGEX REPLACE "\\\\" "/" newpath "${newpath}")
  set(${result} ${newpath} PARENT_SCOPE)
endfunction()

# Find files recursively and add to existing list if any
macro(AK_FindFiles RESULT FILEGLOB)
    file(GLOB_RECURSE TMP ${FILEGLOB})
    if (NOT "${${RESULT}}" STREQUAL "")
        set(TMP ";${TMP}")
    endif()
    set(${RESULT} ${${RESULT}}${TMP})
endmacro(AK_FindFiles)

#macro(AK_zip archive cwd)
#    file(GLOB src LIST_DIRECTORIES false RELATIVE "${cwd}" "${cwd}/*")
#    message(STATUS "cd ${cwd}")
#    message(STATUS "cmd=tar cfv ${archive} --format=zip ${src}")
#    execute_process(COMMAND ${CMAKE_COMMAND} -E tar cfv "${archive}" --format=zip ${src} WORKING_DIRECTORY "${cwd}" RESULT_VARIABLE AK_RESULT)
#    if (NOT ${AK_RESULT} EQUAL 0)
#        message(FATAL_ERROR "RESULT=${AK_RESULT}")
#    endif()
#endmacro(AK_zip)

macro(AK_7zip archive cwd)
    file(GLOB_RECURSE src "${cwd}/*")
    message(STATUS "cmd=tar cfv ${archive} --format=7zip ${src}")
    execute_process(COMMAND ${CMAKE_COMMAND} -E tar cfv "${archive}" --format=7zip ${src} RESULT_VARIABLE AK_RESULT)
    if (NOT ${AK_RESULT} EQUAL 0)
        message(FATAL_ERROR "RESULT=${AK_RESULT}")
    endif()
endmacro(AK_7zip)

macro(AK_zip archive cwd) # pwd)
    set(opt_argv ${ARGN})
    list(LENGTH opt_argv opt_argc)
    if (${opt_argc} GREATER 0)
        list(GET opt_argv 0 pwd)
        if (NOT "${pwd}" STREQUAL "")
            set(pwd "-p${pwd}")
            set(pwdmask "-p***")
        endif()
    endif()
    message(STATUS "cd ${cwd}")
    #message(STATUS "cmd=tar xfv ${archive}")
    message(STATUS "cmd=7z a -aoa -bb1 ${pwdmask} ${archive}")
    execute_process(COMMAND bash 7z a -aoa -bb1 ${pwd} "${archive}" WORKING_DIRECTORY "${cwd}" RESULT_VARIABLE AK_RESULT)
    if (NOT ${AK_RESULT} EQUAL 0)
        message("RESULT=${AK_RESULT}")
        message(FATAL_ERROR "Ensure p7zip (7z) is installed")
    endif()
endmacro(AK_zip)

macro(AK_unzip archive cwd) # pwd)
    set(opt_argv ${ARGN})
    list(LENGTH opt_argv opt_argc)
    if (${opt_argc} GREATER 0)
        list(GET opt_argv 0 pwd)
        if (NOT "${pwd}" STREQUAL "")
            set(pwd "-p${pwd}")
            set(pwdmask "-p***")
        endif()
    endif()
    message(STATUS "cd ${cwd}")
    #message(STATUS "cmd=tar xfv ${archive}")
    message(STATUS "cmd=7z x -aoa -bb1 ${pwdmask} ${archive}")
    execute_process(COMMAND bash 7z x -aoa -bb1 ${pwd} "${archive}" WORKING_DIRECTORY "${cwd}" RESULT_VARIABLE AK_RESULT)
    if (NOT ${AK_RESULT} EQUAL 0)
        message("RESULT=${AK_RESULT}")
        message(FATAL_ERROR "Ensure p7zip (7z) is installed")
    endif()
endmacro(AK_unzip)

macro(AK_unzipif archivedir)
    if (NOT EXISTS "${archivedir}")
        file(MAKE_DIRECTORY "${archivedir}")
        AK_unzip("${archivedir}.7z" "${archivedir}/..")
        message("Extracted ${archivedir}.7z")
    endif()
endmacro(AK_unzipif)

macro(AK_unzip_folders_tobedone)
set(MSG "")
file(GLOB_RECURSE ZIPFILES RELATIVE "${AK_DIR_PROJECT}/7zips"  "7zips/*.7z")
foreach(ZIPFILE ${ZIPFILES})
    string(REGEX REPLACE ".7z$" "" UNZIPDIR "${ZIPFILE}")
    set(ZIPFILE "7zips/${ZIPFILE}")
    set(MSG "${MSG}\n${ZIPFILE} -> ${UNZIPDIR}")
    if (NOT EXISTS "${UNZIPDIR}")
        file(MAKE_DIRECTORY "${UNZIPDIR}")
        AK_unzip("${AK_DIR_PROJECT}/${ZIPFILE}" "${UNZIPDIR}/..")
    else()
        set(MSG "${MSG} (skipped)")
    endif()
endforeach()
endmacro(AK_unzip_folders_tobedone)

function(AK_ghfetch repo asset user tag)
    if ("${asset}" STREQUAL "" OR "${asset}" STREQUAL "x")
        set(asset "binaries.7z") # value is "org" or "user"
    endif()
    if ("${user}" STREQUAL "" OR "${user}" STREQUAL "x")
        set(user "org") # value is "org" or "user"
    endif()
    if ("${tag}" STREQUAL "" OR "${tag}" STREQUAL "x")
        set(tag ">=0.1.0")
    endif()
    #####git_LATEST_V=$(curl -s "https://$git_token:@api.github.com/repos/ato-dis/xxxx.be/releases/latest" | sed -n 's/.*tag_name": "\([^"]*\).*/\1/p')
    #gh_fetch --github-oauth-token=$git_token --repo="https://github.com/ato-dis/xxxx.be" --tag=">=v1.0.0" --release-asset="prebuilt.7z" .
    set(ENV{GITHUB_OAUTH_TOKEN} $ENV{git_token})
    set(ARGS "--repo=https://github.com/$ENV{git_${user}}/${repo}" "--tag=${tag}" "--release-asset=${asset}" ".")
    string(REGEX REPLACE ";" " " XARGS "${ARGS}" )
    message("ghfetch ${XARGS} WORKING_DIRECTORY .")
    execute_process(COMMAND ghfetch ${ARGS} WORKING_DIRECTORY "." RESULT_VARIABLE AK_RESULT OUTPUT_STRIP_TRAILING_WHITESPACE)
    if (NOT ${AK_RESULT} EQUAL 0)
        message(FATAL_ERROR "RESULT=${AK_RESULT}")
    endif()
endfunction(AK_ghfetch)

function(AK_ghfetch_latest repo asset)
    AK_ghfetch("${repo}" "${asset}" "" "")
endfunction(AK_ghfetch_latest)

#zippwd must be supplied but is ignored if blank
function(AK_ghfetch_unzip unzip_dst fetch_dst fetch_repo asset zippwd)
    string(REPLACE "." "/" assetdir "dep.${asset}")
    string(APPEND asset ".7z")
    message("================================================================")
    message("${fetch_repo}/${asset}")
    message("  -> ${fetch_dst}/7zips/${asset}")
    message("  -> ${unzip_dst}/${assetdir}")

    if (NOT EXISTS "${fetch_dst}/7zips/${asset}")
        file(MAKE_DIRECTORY "${fetch_dst}/7zips")
        AK_ghfetch_latest("${fetch_repo}" "${asset}")
        file(RENAME "${asset}" "${fetch_dst}/7zips/${asset}")
        message("  downloaded")
    else()
        message("  already downloaded")
    endif()

    if (EXISTS "${fetch_dst}/7zips/${asset}")
        if (NOT EXISTS "${unzip_dst}/${assetdir}")
            file(MAKE_DIRECTORY "${unzip_dst}/${assetdir}")
            AK_unzip("${fetch_dst}/7zips/${asset}" "${unzip_dst}/${assetdir}/.." "${zippwd}")
            message("  unzipped")
        else()
            message("  '${unzip_dst}/${assetdir}' already exists")
        endif()
    else()
        message("  local 7z not found")
    endif()
endfunction(AK_ghfetch_unzip)

# Fetch and unzip dependency to dep repo
function(AK_ghfetch_unzip_dep asset)
    if ("${asset}" MATCHES ".*\.private")
        set(opt_argv ${ARGN})
        list(LENGTH opt_argv opt_argc)
        if (${opt_argc} GREATER 0)
            list(GET opt_argv 0 zippwd)
        endif()
        AK_ghfetch_unzip("${AK_DIR_DEP}" "${AK_DIR_DEP}" "ausid.csc" "${asset}" "${zippwd}")
    else()
        AK_ghfetch_unzip("${AK_DIR_DEP}" "${AK_DIR_DEP}" "ausid.dep" "${asset}" "")
    endif()
endfunction(AK_ghfetch_unzip_dep)

# Fetch and unzip dependency to dep repo then copy it to folder with parent basedir
function(AK_ghfetch_unzip_dep_copy asset basedir)
    string(REPLACE "." "/" assetdir "dep.${asset}")
    if (NOT EXISTS "${basedir}/${assetdir}")
        AK_ghfetch_unzip_dep("${asset}")
        file(COPY "${AK_DIR_DEP}/${assetdir}" DESTINATION "${basedir}/${assetdir}/../")
        message("  Copied '${AK_DIR_DEP}/${assetdir}' -> '${basedir}/${assetdir}/../'")
    else()
        message("  '${basedir}/${assetdir}' already exists")
    endif()
endfunction(AK_ghfetch_unzip_dep_copy)

#[[
macro(patchFile path)
configure_file("${path}" "${PROJECT_SOURCE_DIR}/src/akversion.h" @ONLY)
    message(STATUS "path=${path}")
    foreach(arg ${ARGN})
        string(REGEX MATCH  "^(.+)=(.*)$" myArg "${arg}")
        if (${CMAKE_MATCH_COUNT} LESS 2)
            continue()
        endif()
        configure_file("${path}" "${PROJECT_SOURCE_DIR}/src/akversion.h" @ONLY)

        <replaceregexp flags="g" file="@{path}" match="@EXT_NAME_X@" replace="@{name}" unless:blank="name"/>
        message(STATUS "CMAKE_MATCH_1=${CMAKE_MATCH_1}")
        message(STATUS "CMAKE_MATCH_2=${CMAKE_MATCH_2}")
    endforeach()
endmacro(patchFile)
#]]
