#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intmsgparty.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_MSGP;
static unsigned long _moduleid = SBR_CSR_MODULEID_MSGP;
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

static const char *xPartySender = ".//data:Sender";
static const char *xPartyReceiver = ".//data:Receiver";
static const char *xDesignation = "data:IdentificationDetails.IdentifierDesignation.Text";
static const char *xScheme = "data:IdentificationDetails.IdentifierName.Text";

/*********************************************************************************/
struct _sbr_MsgParty {
    sbr_eMsgPartyType type;
    char *iddesignation;
    char *idname;
    const char *xParty;
    ato_Xml *xml;
    void *xnode;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void _ato_free(sbr_MsgParty *party)
{
    if (party == NULL) return;

    party->iddesignation = ato_free(party->iddesignation);
    party->idname = ato_free(party->idname);
}

static const char *_xparty(sbr_eMsgPartyType partytype)
{
    assert(partytype == SBR_MSG_RECEIVER || partytype == SBR_MSG_SENDER);
    if (partytype == SBR_MSG_RECEIVER)
        return xPartyReceiver;
    else
        return xPartySender;
}

static sbr_MsgParty *_create(sbr_eMsgPartyType partytype)
{
    const char *xParty = _xparty(partytype);
    sbr_MsgParty *party = NULL;
    party = calloc(1, sizeof(sbr_MsgParty));
    assert(party != NULL);
    party->type = partytype;
    party->xParty = xParty;
    return party;
}
/*********************************************************************************/
int sbr__msgp_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void sbr__msgp_deinit(void)
{
}

void sbr__msgp_create(ato_Ctx *ctx, sbr_MsgParty **obj,
                        sbr_eMsgPartyType partytype,
                        ato_Xml *xml, void *pxnode)
{
    const char *function = "sbr__msgp_create";
    int errcode = ATO_ERR_OK;
    sbr_MsgParty *party = NULL;

    assert(ctx != NULL);
    ATO_ASSERT_ISNOTALLOCATED(obj);

    ATO_CTX_FN_START(ctx);

    if (!xml || !pxnode) {
        *obj = party = _create(partytype);
        assert(party != NULL);
    } else {
        void *xnode = ato_xml_findnode(xml, pxnode, _xparty(partytype), NULL);
        if (xnode) {
            *obj = party = _create(partytype);
            assert(party != NULL);
            party->xnode = xnode;
            party->xml = xml;
            ato_xml_nodevalue(party->xml, party->xnode, xDesignation, &(party->iddesignation), FALSE);
            ato_xml_nodevalue(party->xml, party->xnode, xScheme, &(party->idname), FALSE);
        }
    }
    ATO_CTX_FN_END(ctx, errcode);
}

void sbr__msgp_free(sbr_MsgParty *party)
{
    if (party == NULL) return;
    _ato_free(party);
    free(party);
}

void sbr__msgp_set(ato_Ctx *ctx, sbr_MsgParty *party,
                            const char *iddesignation,
                            const char *idname
                            )
{
    const char *function = "sbr__msgp_set";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    assert(ctx != NULL); assert(party != NULL);
    assert(party->type == SBR_MSG_RECEIVER || party->type == SBR_MSG_SENDER);

    if (!ato_streq(party->iddesignation, iddesignation)) {
        ato_free(party->iddesignation);
        party->iddesignation = ato_strdup(iddesignation, 0);
    }
    if (!ato_streq(party->idname, idname)) {
        ato_free(party->idname);
        party->idname = ato_strdup(idname, 0);
    }

    ATO_CTX_FN_END(ctx, errcode);
}

void sbr__msgp_render(ato_Ctx *ctx, sbr_MsgParty *party, ato_Xml *xml, void *pxnode)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "sbr__msgp_render";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    if (party) {
        if (!party->xnode) {
            assert(xml != NULL); assert(pxnode != NULL);
            party->xnode = ato_xml_findnode(xml, pxnode, party->xParty, NULL);
        }
        if (!party->xnode) {
            const char *s = sbr_msgp_typestr(party->type);
            Throw ATO_CTX_VNEWERR(ctx, SBR_CSR_ERR_GENERAL, strlen(s), "Failed to add %s party.", s);
        }
        party->xml = xml;
        ato_xml_createnodevalue(party->xml, party->xnode, NULL, xDesignation, party->iddesignation);
        ato_xml_createnodevalue(party->xml, party->xnode, NULL, xScheme, party->idname);
    }

    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
const char *sbr_msgp_iddesignation(sbr_MsgParty *party)
{
    assert(party != NULL);
    return party->iddesignation;
}

const char *sbr_msgp_idname(sbr_MsgParty *party)
{
    assert(party != NULL);
    return party->idname;
}

sbr_eMsgPartyType sbr_msgp_type(sbr_MsgParty *party)
{
    assert(party != NULL);
    return party->type;
}

const char *sbr_msgp_typestr(sbr_eMsgPartyType msgpartytype)
{
    switch (msgpartytype) {
        case SBR_MSG_SENDER: return "SBR_MSG_SENDER";
        case SBR_MSG_RECEIVER: return "SBR_MSG_RECEIVER";
        default: return "";
    }
}
