#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intmsgts.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_MSGTS;
static unsigned long _moduleid = SBR_CSR_MODULEID_MSGTS;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;

static const char *xTimestamps = ".//data:MessageTimestamps";
static const char *xTimestamp = "./data:MessageTimestamp";
static const char *xCode = "data:Message.Timestamp.GenerationSource.Code";
static const char *xDatetime = "data:Message.Timestamp.Generation.Datetime";

/*********************************************************************************/
struct _sbr_MsgTimestamp {
    void *xnode;
    char *source;
    char *datetime;
};

struct _sbr_MsgTimestamps {
    ato_Xml *xml;
    void *xnode; // container node
    ato_List *itemlist;
    ato_Iterator *iter;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void _freeitem(sbr_MsgTimestamp *timestamp)
{
    if (timestamp == NULL) return;

    timestamp->source = ato_free(timestamp->source);
    timestamp->datetime = ato_free(timestamp->datetime);
    free(timestamp);
}

static void _createitem(ato_Ctx *ctx, sbr_MsgTimestamp **obj, ato_Xml *xml, void *xnode)
{
    sbr_MsgTimestamp *timestamp = NULL;

    assert(ctx != NULL); ATO_IGNORE(ctx);
    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = timestamp = calloc(1, sizeof(sbr_MsgTimestamp));
    assert(timestamp != NULL);
    timestamp->xnode = xnode;

    if (xnode) {
        ato_xml_nodevalue(xml, xnode, xCode, &(timestamp->source), FALSE);
        ato_xml_nodevalue(xml, xnode, xDatetime, &(timestamp->datetime), FALSE);
    }

}

/*********************************************************************************/
int sbr__msgts_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void sbr__msgts_deinit(void)
{
}

void sbr__msgts_free(sbr_MsgTimestamps *timestamps)
{
    if (timestamps == NULL) return;

    {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(timestamps->itemlist, node)) != NULL)
            _freeitem(ato_lst_value(node));
    }
    ato_lst_free(timestamps->itemlist);
    ato_iter_free(timestamps->iter);
    free(timestamps);
}

void sbr__msgts_create(ato_Ctx *ctx, sbr_MsgTimestamps **obj, ato_Xml *xml, void *pxnode)
{
    const char *function = "sbr__msgts_create";
    int errcode = ATO_ERR_OK;
    sbr_MsgTimestamps *timestamps = NULL;
    void *xnode = NULL;
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);

    xnode = ato_xml_findnode(xml, pxnode, xTimestamps, NULL);
    if (xnode) {
        void *xlist = NULL;

        *obj = timestamps = calloc(1, sizeof(sbr_MsgTimestamps));
        assert(timestamps != NULL);
        timestamps->xml = xml;
        timestamps->xnode = xnode;

        xlist = ato_xml_listload(xml, xnode, xTimestamp, NULL);
        if (xlist != NULL) {
            size_t i = 0, count = 0;
            ato_lst_create(&(timestamps->itemlist));

            count = ato_xml_listcount(xml, xlist);
            for (i = 0; i < count; i++) {
                void *psubnode = NULL;
                sbr_MsgTimestamp *item = NULL;
                psubnode = ato_xml_listitem(xml, xlist, i);
                assert(psubnode != NULL);
                _createitem(ctx, &item, xml, psubnode);
                ato_lst_add(timestamps->itemlist, NULL, item);
            }
            ato_xml_listfree(xml, xlist);
        }
        timestamps->iter = ato_iter_create_lst(timestamps->itemlist);
    }

    ATO_CTX_FN_END(ctx, errcode);
}

size_t sbr__msgts_count(sbr_MsgTimestamps *timestamps)
{
    if (timestamps == NULL) return 0;
    return ato_lst_count(timestamps->itemlist);
}

void sbr__msgts_iterator(sbr_MsgTimestamps *timestamps, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (timestamps == NULL) return;
    *iter = ato_iter_create_lst(timestamps->itemlist);
}
sbr_MsgTimestamp *sbr__msgts_firstv(sbr_MsgTimestamps *timestamps)
{
    if (timestamps == NULL) return NULL;
    return ato_iter_firstv(timestamps->iter);
}
sbr_MsgTimestamp *sbr__msgts_nextv(sbr_MsgTimestamps *timestamps)
{
    if (timestamps == NULL) return NULL;
    return ato_iter_nextv(timestamps->iter);
}

void sbr__msgts_set(ato_Ctx *ctx, sbr_MsgTimestamp *timestamp, const char *source, const char *datetime)
{
    const char *function = "sbr__msgts_set";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    assert(ctx != NULL); assert(timestamp != NULL);

    if (!ato_streq(timestamp->source, source)) {
        ato_free(timestamp->source);
        timestamp->source = ato_strdup(source, 0);
    }
    if (!ato_streq(timestamp->datetime, datetime)) {
        ato_free(timestamp->datetime);
        timestamp->datetime = ato_strdup(datetime, 0);
    }

    ATO_CTX_FN_END(ctx, errcode);
}

/*
void sbr__msgts_render(ato_Ctx *ctx, sbr_MsgTimestamp *timestamp, ato_Xml *xml, void *pxnode)
{
    const char *function = "sbr__msgts_render";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    if (timestamp) {
        ato_xml_createnodevaluevar(xml, pxnode, xTimestamp, ato_itoa(buf, index+1), xCode, timestamp->code);
        ato_xml_createnodevaluevar(xml, pxnode, xTimestamp, ato_itoa(buf, index+1), xDatetime, party->datetime);
    }

    ATO_CTX_FN_END(ctx, errcode);
}
*/

/*********************************************************************************/
const char *sbr_msgts_source(sbr_MsgTimestamp *timestamp)
{
    assert(timestamp != NULL);
    return timestamp->source;
}

const char *sbr_msgts_datetime(sbr_MsgTimestamp *timestamp)
{
    assert(timestamp != NULL);
    return timestamp->datetime;
}

/*********************************************************************************/

