#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>
#include <sys/time.h>

#include "xmlsec/xmlsec.h"
#include "xmlsec/xmltree.h"
#include "xmlsec/xmldsig.h"
#include "xmlsec/crypto.h"
#include "xmlsec/parser.h"

#include "atobase/private/pall.h"
#include "atobnet/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intcsr.h"
#include "intmsgevent.h"
#include "intmsgts.h"
#include "intmsgparty.h"
#include "intmsgrec.h"
#include "intdoca.h"
#include "intdoc.h"
#include "intsbdm.h"
#include "intresponse.h"
#include "intrequest.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_REQUEST;
static unsigned long _moduleid = SBR_CSR_MODULEID_REQUEST;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;
static bool _logdumpfile = FALSE;

/*********************************************************************************/
struct _sbr_Request {
    sbr_eMsgSourceType msgsrctype;
    char *servicename;
    void *xservicenode;
    sbr_Csr *csr;
    ato_Net *net;
    char *contenttype;
    size_t MTOMthreshold;
    sbr_Response *response;
    ato_Xml *xml;
    ato_List *sbdms;
    ato_Iterator *iter;
    char *softwareSubscriptionId;
};

/*********************************************************************************/
static const char *Sig1Name = "/s:Envelope/s:Header/wsse:Security/sig:Signature[1]";
static const char *Sig2Name = "/s:Envelope/s:Header/wsse:Security/sig:Signature[2]";
static const char *EncryptedAssertionName = "/s:Envelope/s:Header/wsse:Security/saml:EncryptedAssertion";
static const char *BinarySecurityTokenName = "/s:Envelope/s:Header/wsse:Security/wsse:BinarySecurityToken";
static const char *xServiceName = "/s:Envelope/s:Body/service:$";
static const char *GenerationTSName = "/s:Envelope/s:Body/service:$/data:StandardBusinessDocumentMessage/data:StandardBusinessDocumentHeader/data:MessageTimestamps/data:MessageTimestamp/data:Message.Timestamp.Generation.Datetime";
static const char *OrgName = "/s:Envelope/s:Body/service:$/data:StandardBusinessDocumentMessage/data:StandardBusinessDocumentHeader/data:SoftwareInformation/data:OrganisationNameDetails.OrganisationalName.Text";
static const char *ProductName = "/s:Envelope/s:Body/service:$/data:StandardBusinessDocumentMessage/data:StandardBusinessDocumentHeader/data:SoftwareInformation/data:SoftwareInformation.ProductName.Text";
static const char *VersionName = "/s:Envelope/s:Body/service:$/data:StandardBusinessDocumentMessage/data:StandardBusinessDocumentHeader/data:SoftwareInformation/data:SoftwareInformation.ProductVersion.Text";

static const char *SoftwareSubscriptionIdName = "/s:Envelope/s:Header/wsse:Security/ssid:softwareSubscriptionId";

#define SBR_CORRECT_YEAR_OFFSET 100

/*********************************************************************************/
static void _req_clear(sbr_Request *request)
{
    if (request == NULL) return;

    sbr__res_free(request->response);
    request->response = NULL;
    ato_xml_free(request->xml);
    request->xml = NULL;
}

static void _ato_free(sbr_Request *request)
{
    ato_ListNode *node = NULL;
    if (request == NULL) return;

    _req_clear(request);
    request->servicename = ato_free(request->servicename);
    request->contenttype = ato_free(request->contenttype);
    ato_bnet_free(request->net);

    if (request->sbdms != NULL) {
        while ((node = ato_lst_next(request->sbdms, node)) != NULL)
            sbr__sbdm_free(ato_lst_value(node));
        ato_lst_free(request->sbdms);
    }
    ato_iter_free(request->iter);
    request->softwareSubscriptionId = ato_free(request->softwareSubscriptionId);
}

static void _setns(ato_Ctx *ctx, ato_Xml *xml, const char *servicens)
{
    assert(xml != NULL); assert(servicens != NULL);
    ato_xml_addnamespace(ctx, xml, "i", "http://www.w3.org/2001/XMLSchema-instance");
    ato_xml_addnamespace(ctx, xml, "s", "http://www.w3.org/2003/05/soap-envelope");
    ato_xml_addnamespace(ctx, xml, "d4p1", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
    ato_xml_addnamespace(ctx, xml, "d2p1", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
    ato_xml_addnamespace(ctx, xml, "xenc", "http://www.w3.org/2001/04/xmlenc#");
    ato_xml_addnamespace(ctx, xml, "wsse", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd");

    ato_xml_addnamespace(ctx, xml, "sig", "http://www.w3.org/2000/09/xmldsig#");
    //b = ato__xml_setns(ctx, xml, "wst", "http://docs.oasis-open.org/ws-sx/ws-trust/200512");
    ato_xml_addnamespace(ctx, xml, "saml", "urn:oasis:names:tc:SAML:2.0:assertion");
    ato_xml_addnamespace(ctx, xml, "data", "http://sbr.gov.au/comn/sbdm.02.data");
    ato_xml_addnamespace(ctx, xml, "service", servicens); //"tns"

    ato_xml_addnamespace(ctx, xml, "err", "http://sbr.gov.au/comn/sbrfaults.200809");
    ato_xml_addnamespace(ctx, xml, "nsmime", "http://www.w3.org/2005/05/xmlmime");

    ato_xml_addnamespace(ctx, xml, "ssid", "http://sbr.gov.au/identifier/softwareSubscriptionId");
}

static void _printstuff(ato_Ctx *ctx, ato_Log *log, ato_Xml *xml, const char *request_servicename)
{
    int errcode = ATO_ERR_OK;
    static const char *function = "_printstuff";
    char *GenerationTS = NULL, *BinarySecurityToken = NULL;
    char *si_name = NULL, *si_product = NULL, *si_version = NULL;

    ATO_CTX_FN_START(ctx);

    ato_xml_nodevaluevar(xml, NULL, GenerationTSName, request_servicename, &GenerationTS, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "GenerationTS=%s", GenerationTS);

    ato_xml_nodevalue(xml, NULL, BinarySecurityTokenName, &BinarySecurityToken, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "BinarySecurityToken=%s", BinarySecurityToken);

    ato_xml_nodevaluevar(xml, NULL, OrgName, request_servicename, &si_name, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "OrgName=%s", si_name);
    ato_xml_nodevaluevar(xml, NULL, ProductName, request_servicename, &si_product, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "ProductName=%s", si_product);
    ato_xml_nodevaluevar(xml, NULL, VersionName, request_servicename, &si_version, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "VersionName=%s", si_version);

    BinarySecurityToken = ato_free(BinarySecurityToken);
    GenerationTS = ato_free(GenerationTS);
    si_name = ato_free(si_name);
    si_product = ato_free(si_product);
    si_version = ato_free(si_version);

    ATO_CTX_FN_END(ctx, errcode);
}

static void _setsoftwareinfo(ato_Ctx *ctx, sbr_Request *request)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_setsoftwareinfo";
    int errcode = ATO_ERR_OK;
    void *encodedvalue = NULL;

    Try {
        encodedvalue = ato_xml_encodedvalue_create(request->xml, ato_si_organisation());
        if (!ato_xml_setnodevaluevar(request->xml, NULL, OrgName, request->servicename, encodedvalue))
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to set request softwareinfo organisation");
        encodedvalue = ato_xml_encodedvalue_free(request->xml, encodedvalue);
        encodedvalue = ato_xml_encodedvalue_create(request->xml, ato_si_product());
        if (!ato_xml_setnodevaluevar(request->xml, NULL, ProductName, request->servicename, encodedvalue))
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to set request softwareinfo product");
        encodedvalue = ato_xml_encodedvalue_free(request->xml, encodedvalue);
        if (!ato_xml_setnodevaluevar(request->xml, NULL, VersionName, request->servicename, ato_si_version()))
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to set request softwareinfo version");
    } Catch (errcode) {
        ato_xml_encodedvalue_free(request->xml, encodedvalue);
    }
    if (errcode != ATO_ERR_OK) Throw errcode;
}

//-------------------------------------------------------------------------------

static void _render_request(ato_Ctx *ctx, sbr_Request *request)
{
    sbr_Sbdm *sbdm = NULL;
    for (sbdm = sbr_req_sbdm_firstv(request); sbdm; sbdm = sbr_req_sbdm_nextv(request)) {
        sbr__sbdm_render(ctx, sbdm);
        if (sbdm) break; // we're only going to support one SBDM at the moment - just silently ignore any others
    }
}
//-------------------------------------------------------------------------------
// We're going to some milli value. Don't care about accuracy as it's just to distinguish relative values.
static int timeInMilliseconds(void) {
    struct timeval tv;
    gettimeofday(&tv,NULL);
    return tv.tv_usec;
}
#define CSR_TS_FMT_MILLIS "20%02d-%02d-%02dT%02d:%02d:%02d.%06dZ"
static void gettimemillis(char *buffer)
{
    time_t now = 0;
    struct tm tms;
    struct tm *t = NULL;
    int tm_year = 0;
    int millis = timeInMilliseconds();
    t = &tms;
    time(&now);
    ato_gmtime(&now, &tms);
    tm_year = t->tm_year - SBR_CORRECT_YEAR_OFFSET;
    sprintf(buffer, CSR_TS_FMT_MILLIS, tm_year, t->tm_mon+1, t->tm_mday, t->tm_hour, t->tm_min, t->tm_sec, millis);
}

static void _build_xml(ato_Ctx *ctx, sbr_Request *request, ato_String *encryptedtoken, ato_String *certificate)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_build_xml";
    int errcode = ATO_ERR_OK;
    char buffer[100];
    ato_String *b64 = NULL;

    ato_Log *log = ato_ctx_log(ctx);

    if (_loglevel >= ATO_LOG_DEBUG)
        _printstuff(ctx,log, request->xml, request->servicename);

    Try {
        if (ato_base64encode(ctx, &b64, certificate) != ATO_ERR_OK)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Possible bad certificate - failed to (base64) encode");

        if (!ato_xml_setnodevaluelen(request->xml, NULL, BinarySecurityTokenName, ato_str_value(b64), ato_str_len(b64)))
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to insert certificate into request");

        if (!ato_xml_createnodevalueraw(ctx, request->xml, NULL, EncryptedAssertionName, NULL, ato_str_value(encryptedtoken)))
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to insert token into request");

        gettimemillis(buffer);
        if (!ato_xml_setnodevaluevar(request->xml, NULL, GenerationTSName, request->servicename, buffer))
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to set request timestamp");

        _setsoftwareinfo(ctx, request);

        if (_loglevel >= ATO_LOG_DEBUG)
            _printstuff(ctx, log, request->xml, request->servicename);

    } Catch (errcode) {
    }

    ato_str_free(b64);
    if (errcode != ATO_ERR_OK) Throw errcode;
}

static void _loadtemplate(ato_Ctx *ctx, sbr_Request *request, ato_String *xmltemplate)
{
    _req_clear(request);
    ato_xml_create(ctx, &(request->xml), xmltemplate);
    _setns(ctx, request->xml, sbr_csr_servicenamespace(ctx, request->csr));
}

static void _setnet(ato_Ctx *ctx, sbr_Request *request, const char *url, size_t timeout, size_t timeoutconnect)
{
    ato_bnet_create(ctx, &(request->net), url, request->contenttype);
    ato_bnet_settimeout(request->net, timeout);
    ato_bnet_settimeoutconnect(request->net, timeoutconnect);
}

static void _settype(ato_Ctx *ctx, sbr_Request *request, const char *action, const char *url, size_t timeout, size_t timeoutconnect)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    // action="http://sbr.gov.au/services/prefill.02.service/PreFillReport"
    const char *ctbase = "application/soap+xml; charset=utf-8;";
    int errcode = ATO_ERR_OK;

    request->contenttype = ato_strdup(ctbase, ato_strlen(action) + 20);
    if (action != NULL) {
        strcat(request->contenttype, " action=");
        strcat(request->contenttype, action);
    }

    Try {
        _setnet(ctx, request, url, timeout, timeoutconnect);
    } Catch(errcode) {
    }

    ATO_RETHROW_ONERR(errcode);
}

static void _dumpxml(ato_Ctx *ctx, ato_Log *log, ato_Xml *xml, const char *msg, const char *filename)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_dumpxml";
    int errcode = ATO_ERR_OK;

    Try {
        if (_logdumpfile) {
            ATO_LOG_MSG(log, ATO_LOG_DEBUG, msg);
            ato_xml_savef(ctx, xml, filename);
        }
    } Catch(errcode) {
    }

    if (errcode != ATO_ERR_OK) Throw errcode;
}

static xmlSecKeyPtr _loadkey(ato_Ctx *ctx, ato_String *keydata, xmlSecKeyDataFormat format)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_loadkey";
    xmlSecKeyPtr key = NULL;

    if (format == xmlSecKeyDataFormatDer) {
        key = xmlSecCryptoAppKeyLoadMemory((const unsigned char *)ato_str_value(keydata), ato_str_len(keydata), format, NULL, NULL, NULL);
        if (key == NULL)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to load DER key");
    } else {
        xmlSecKeyDataId dataId = xmlSecKeyDataIdListFindByName(xmlSecKeyDataIdsGet(), (const xmlChar*)"hmac", xmlSecKeyDataUsageAny);
        if (dataId == xmlSecKeyDataIdUnknown) {
            key = NULL;
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to find hmac key");
        }
        else
            key = xmlSecKeyReadMemory(dataId, (const unsigned char *)ato_str_value(keydata), ato_str_len(keydata));
        if (key == NULL)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to load hmac key");
    }

    return key;
}

static void _sign(ato_Ctx *ctx, sbr_Request *request, const char *startxpath, ato_String *keydata, xmlSecKeyDataFormat format)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_sign";
    int errcode = ATO_ERR_OK;
    int ret = 0;
    xmlSecKeyPtr key = NULL;
    xmlNodePtr cur = NULL;
    xmlSecDSigCtx dsigCtx;
    xmlSecKeysMngrPtr gKeysMngr = NULL;
    ato_Log *log = ato_ctx_log(ctx);

    ATO_CTX_FN_START(ctx);

    Try {
        gKeysMngr = xmlSecKeysMngrCreate();
        if (gKeysMngr == NULL)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to setup keymanager");
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Key manager created");

        if (xmlSecCryptoAppDefaultKeysMngrInit(gKeysMngr) < 0)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to initialize keymanager");
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Key manager initialized");

        key = _loadkey(ctx, keydata, format);

        ret = xmlSecCryptoAppDefaultKeysMngrAdoptKey(gKeysMngr, key);
        if (ret != 0)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to load key");
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Key loaded");

        if (xmlSecDSigCtxInitialize(&dsigCtx, gKeysMngr) < 0)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to initialize signing context");
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Signing context initialized");

        cur = ato_xml_findnode(request->xml, NULL, startxpath, NULL);
        if (cur == NULL)
            Throw ATO_CTX_VNEWERR(ctx, SBR_CSR_ERR_GENERAL, strlen(startxpath), "Failed to locate node %s", startxpath);
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Root xml target found");

        cur = xmlSecFindNode(cur, xmlSecNodeSignature, xmlSecDSigNs);
        if (cur == NULL)
            Throw ATO_CTX_VNEWERR(ctx, SBR_CSR_ERR_GENERAL, strlen(startxpath), "Failed to locate signing node %s", startxpath);
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "xml signing node found");

        if(xmlSecDSigCtxSign(&dsigCtx, cur) < 0)
            Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to sign request");
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Signing complete");

    } Catch (errcode) {
    }

    if (gKeysMngr)
        xmlSecKeysMngrDestroy(gKeysMngr);
    xmlSecDSigCtxFinalize(&dsigCtx);
    ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Signing context deinitialized");
    if (errcode != ATO_ERR_OK) Throw errcode;
    ATO_CTX_FN_END(ctx, errcode);
}

static void _postsignhack(ato_Log *log, sbr_Request *request) {
    static const char *function = "_postsignxxxx";
    void *xnode = ato_xml_findnode(request->xml, NULL, SoftwareSubscriptionIdName, NULL);
    if (xnode == NULL) {
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "softwareSubscriptionId not found");
        return;
    }
    if (request->softwareSubscriptionId == NULL) {
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Removing softwareSubscriptionId");
        ato_xml_deletenode(request->xml, xnode);
    } else {
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Setting softwareSubscriptionId");
        ato_xml_setvalue(request->xml, xnode, request->softwareSubscriptionId);
    }
}

static void _submit(ato_Ctx *ctx, sbr_Request *request, ato_String *privatekey, ato_String *hmackey)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_submit";
    static const char *xBinaryObjects = "//data:Message.Attachment.Instance.BinaryObject";
    static const char *xContentType = "@nsmime:contentType";
    int errcode = ATO_ERR_OK;
    ato_NetData *ndresponse = NULL, *ndrequest = NULL;
    ato_Log *log = ato_ctx_log(ctx);

    ATO_CTX_FN_START(ctx);

    assert(request->xml != NULL);
    assert(privatekey != NULL); assert(hmackey != NULL);

    Try {
        _dumpxml(ctx, log, request->xml,  "Writing xpresign.xml", "xpresign.xml");

        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Signing with private key");
        _sign(ctx, request, Sig1Name, privatekey, xmlSecKeyDataFormatDer);
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Signing with hmac");
        _sign(ctx, request, Sig2Name, hmackey, 0);
        ato_xml_dropdtd(request->xml);
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Signing completed successfully");

        _postsignhack(log, request);

        _dumpxml(ctx, log, request->xml,  "Writing xsign.xml", "xsign.xml");

        if (ato_bnetd_mtom_convertto(ctx, &ndrequest, request->xml, xBinaryObjects, request->MTOMthreshold, xContentType))
            _dumpxml(ctx, log, request->xml,  "Writing xsign-mtom.xml", "xsign-mtom.xml");
        ndresponse = ato_bnet_requestmp(ctx, request->net, ndrequest);

        sbr__res_create(ctx, &(request->response), ndresponse, sbr_csr_servicename(ctx, request->csr), sbr_csr_servicenamespace(ctx, request->csr));

    } Catch(errcode) {
    }

    ato_bnetd_free(ndrequest);
    if (errcode != ATO_ERR_OK) Throw errcode;

    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}
static void _setlogdumpfile(bool logdumpfile)
{
    _logdumpfile = logdumpfile;
}

static void _process_request(ato_Ctx *ctx, sbr_Request *request)
{
    static const char *xSBDMs = ".//data:StandardBusinessDocumentMessage";
    void *xlist = NULL;
    size_t i = 0, count = 0;

    xlist = ato_xml_listload(request->xml, request->xservicenode, xSBDMs, NULL);
    if (xlist == NULL) return;

    count = ato_xml_listcount(request->xml, xlist);
    for (i = 0; i < count; i++) {
        sbr_Sbdm *sbdm = NULL;
        sbr__sbdm_create(ctx, &sbdm, request->msgsrctype, i, request->xml, request->xservicenode);
        ato_lst_add(request->sbdms, sbr_sbdm_iid(sbdm), sbdm);
    }
    ato_xml_listfree(request->xml, xlist);
}
/*********************************************************************************/
int sbr__req_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    {
        ato_LibModule *lm = ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        ato_initfnlogfiledump(lm, _logdumpfile, _setlogdumpfile);
    }

    return ATO_ERR_OK;
}

void sbr__req_deinit(void)
{
}

/*********************************************************************************/
void sbr__req_create(ato_Ctx *ctx, sbr_Request **obj, sbr_Csr *csr)
{
    const char *function = "sbr__req_create";
    int errcode = ATO_ERR_OK;
    sbr_Request *request = NULL;
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);
    assert(csr != NULL);

    *obj = request = calloc(1, sizeof(sbr_Request));
    assert(request != NULL);

    request->msgsrctype = SBR_REQUEST;
    request->csr = csr;
    request->servicename = ato_strdup("Request", strlen(sbr_csr_servicename(ctx, csr)));
    strcat(request->servicename, sbr_csr_servicename(ctx, csr));
    _settype(ctx, request, NULL, sbr_csr_url(ctx, csr), sbr_csr_timeout(ctx, csr), sbr_csr_timeoutconnect(ctx, csr));
    _loadtemplate(ctx, request, sbr_csr_template(ctx, request->csr));
    ato_lst_create(&(request->sbdms));

    request->xservicenode = ato_xml_findnode(request->xml, NULL, xServiceName, request->servicename);
    _process_request(ctx, request);
    request->iter = ato_iter_create_lst(request->sbdms);
    request->MTOMthreshold = SBR_MTOM_THRESHOLD_DEFAULT;

    ATO_CTX_FN_END(ctx, errcode);
}

void sbr__req_submit(ato_Ctx *ctx, sbr_Request *request, sbr_Response **response, ato_String *certificate, ato_String *privatekey, ato_String *prooftoken, ato_String *assertion)
{
    const char *function = "sbr__req_submit";
    int errcode = ATO_ERR_OK;

    assert(request != NULL);

    ATO_CTX_FN_START(ctx);

    sbr__res_free(request->response);
    *response = request->response = NULL;

    ATO_ASSERT_ISNOTALLOCATED(response);

    _render_request(ctx, request);
    _build_xml(ctx, request, assertion, certificate);
    _submit(ctx, request, privatekey, prooftoken);
    *response = request->response;

    ATO_CTX_FN_END(ctx, errcode);
}

void sbr__req_xml(ato_Ctx *ctx, sbr_Request *request, char **buffer)
{
    const char *function = "sbr__req_xml";
    ato_String *str = NULL;
    int errcode = ATO_ERR_OK;

    ATO_CTX_FN_START(ctx);
    ATO_ASSERT_ISNOTALLOCATED(buffer);

    *buffer = NULL;
    ato_xml_save(ctx, request->xml, &str);
    *buffer = ato_strdup(ato_str_value(str), 0);
    ato_str_free(str);

    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
void sbr_req_free(sbr_Request *request)
{
    if (request == NULL) return;
    _ato_free(request);
    free(request);
}

/*********************************************************************************/
int sbr_req_create(ato_Ctx *ctx, sbr_Request **request, sbr_Csr *csr)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_req_create";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        sbr__req_create(ctx, request, csr);
    } Catch (errcode) {
        errcode = sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

int sbr_req_submit(ato_Ctx *ctx, sbr_Request *request, sbr_Response **response, ato_String *certificate, ato_String *privatekey, ato_String *prooftoken, ato_String *assertion)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_req_submit";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        sbr__req_submit(ctx, request, response, certificate, privatekey, prooftoken, assertion);
    } Catch (errcode) {
        errcode = sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

int sbr_req_xml(ato_Ctx *ctx, sbr_Request *request, char **buffer)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_req_xml";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        sbr__req_xml(ctx, request, buffer);
    } Catch (errcode) {
        errcode = sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

void sbr_req_mtom_setsize(sbr_Request *request, size_t threshold)
{
    request->MTOMthreshold = threshold;
}

size_t sbr_req_mtom_size(sbr_Request *request)
{
    return request->MTOMthreshold;
}

void sbr_req_ssid_set(sbr_Request *request, const char *ssid)
{
    request->softwareSubscriptionId = ato_free(request->softwareSubscriptionId);
    request->softwareSubscriptionId = ato_strdup(ssid, 0);
}

const char *sbr_req_ssid(sbr_Request *request)
{
    return request->softwareSubscriptionId;
}

size_t sbr_req_sbdm_count(sbr_Request *request)
{
    assert(request != NULL);
    return ato_lst_count(request->sbdms);
}

void sbr_req_sbdm_iterator(sbr_Request *request, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (request == NULL) return;
    *iter = ato_iter_create_lst(request->sbdms);
}
sbr_Sbdm *sbr_req_sbdm_firstv(sbr_Request *request)
{
    if (request == NULL) return NULL;
    return ato_iter_firstv(request->iter);
}
sbr_Sbdm *sbr_req_sbdm_nextv(sbr_Request *request)
{
    if (request == NULL) return NULL;
    return ato_iter_nextv(request->iter);
}

/*
static void _req_sbdm_add(ato_Ctx *ctx, sbr_Request *request, sbr_Sbdm **sbdm)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_req_sbdm_add";
    ato_ListNode *node = NULL;
    sbr__sbdm_create(ctx, sbdm, request->msgsrctype, ato_lst_count(request->sbdms), request->xml, request->xservicenode);
    node = ato_lst_add(request->sbdms, sbr_sbdm_iid(*sbdm), *sbdm);
    if (node == NULL)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Error in adding SBDM into Request.");
}

int sbr_req_sbdm_add(ato_Ctx *ctx, sbr_Request *request, sbr_Sbdm **sbdm)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_req_sbdm_add";
    int errcode = ATO_ERR_OK;
    assert(ctx != NULL); assert(request != NULL);
    ATO_CTX_FN_START(ctx);
    Try {
        _req_sbdm_add(ctx, request, sbdm);
    } Catch (errcode) {
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

static void _req_sbdm_remove(ato_Ctx *ctx, sbr_Request *request, sbr_Sbdm *sbdm)
{
    ato_ListNode *node = NULL;
    node = ato_lst_find(request->sbdms, sbr_sbdm_iid(sbdm));
    if (node != NULL) {
        sbr__sbdm_free(ato_lst_value(node));
        ato_lst_delete(node);
    }
    ATO_IGNORE(ctx);
}

int sbr_req_sbdm_remove(ato_Ctx *ctx, sbr_Request *request, sbr_Sbdm *sbdm)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_req_sbdm_remove";
    int errcode = ATO_ERR_OK;
    assert(ctx != NULL); assert(request != NULL);
    ATO_CTX_FN_START(ctx);
    Try {
        _req_sbdm_remove(ctx, request, sbdm);
    } Catch (errcode) {
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}
*/
