#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intmsgevent.h"
#include "intmsgts.h"
#include "intmsgparty.h"
#include "intmsgrec.h"
#include "intdoca.h"
#include "intdoc.h"
#include "intsbdm.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_SBDM;
static unsigned long _moduleid = SBR_CSR_MODULEID_SBDM;
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

static const char *xSBDMs = ".//data:StandardBusinessDocumentMessage[$]";
static const char *xMaximumSeverityCode = ".//data:Message.Event.MaximumSeverity.Code";
static const char *xMessageTypeText = ".//data:Message.Type.Text";

/*********************************************************************************/
struct _sbr_Sbdm {
    ato_Xml *xml;
    void *xnode;
    sbr_eMsgSourceType msgsrctype;
    char guid[50];
    size_t index;
    char *messagetype;
    char *maxseveritycode;
    sbr_MsgParty *receiver;
    sbr_MsgParty *sender;
    sbr_MsgReceipt *receipt;
    sbr_MsgTimestamps *timestamps;
    sbr_MsgEventItems *eventitems;
    sbr_Docs *docs;
    size_t attachmentcount;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void _ato_free(sbr_Sbdm *sbdm)
{
    if (sbdm == NULL) return;

    sbr__msgp_free(sbdm->receiver);
    sbr__msgp_free(sbdm->sender);
    sbr__msgrec_free(sbdm->receipt);
    sbr__msgts_free(sbdm->timestamps);
    sbr__msgevent_free(sbdm->eventitems);
    sbr__doc_free(sbdm->docs);
    sbdm->messagetype = ato_free(sbdm->messagetype);
    sbdm->maxseveritycode = ato_free(sbdm->maxseveritycode);
}

static void _sbdm_setparty(ato_Ctx *ctx, sbr_Sbdm *sbdm, sbr_eMsgPartyType partytype, const char *iddesignation, const char *idname)
{
    if (partytype == SBR_MSG_RECEIVER) {
        if (!sbdm->receiver)
            sbr__msgp_create(ctx, &(sbdm->receiver), SBR_MSG_RECEIVER, NULL, NULL);
        sbr__msgp_set(ctx, sbdm->receiver, iddesignation, idname);
    } else {
        if (!sbdm->sender)
            sbr__msgp_create(ctx, &(sbdm->sender), SBR_MSG_SENDER, NULL, NULL);
        sbr__msgp_set(ctx, sbdm->sender, iddesignation, idname);
    }
}

static void *_process(ato_Ctx *ctx, sbr_Sbdm *sbdm, ato_Xml *xml, void *pxnode)
{
    char buf[50];
    void *node = NULL;

    node = ato_xml_findnode(xml, pxnode, xSBDMs, ato_itoa(buf, (int)(sbdm->index+1)));
    if (!node) {
        return NULL;// ato_xml_addelt(xml, pxnode, "StandardBusinessDocumentMessage");
    }
    ato_xml_nodevalue(xml, node, xMessageTypeText, &(sbdm->messagetype), FALSE);
    ato_xml_nodevalue(xml, node, xMaximumSeverityCode, &(sbdm->maxseveritycode), FALSE);
    sbr__msgp_create(ctx, &(sbdm->receiver), SBR_MSG_RECEIVER, xml, node);
    sbr__msgp_create(ctx, &(sbdm->sender), SBR_MSG_SENDER, xml, node);
    sbr__msgrec_create(ctx, &(sbdm->receipt), xml, node);
    sbr__msgts_create(ctx, &(sbdm->timestamps), xml, node);
    sbr__msgevent_create(ctx, &(sbdm->eventitems), xml, node);
    sbr__doc_create(ctx, &(sbdm->docs), sbdm->msgsrctype, xml, node, &(sbdm->attachmentcount));
    return node;
}

/*********************************************************************************/
int sbr__sbdm_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void sbr__sbdm_deinit(void)
{
}

void sbr__sbdm_create(ato_Ctx *ctx, sbr_Sbdm **obj, sbr_eMsgSourceType msgsrctype, size_t index, ato_Xml *xml, void *pxnode)
{
    const char *function = "sbr__sbdm_create";
    int errcode = ATO_ERR_OK;
    sbr_Sbdm *sbdm = NULL;
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);
    assert(ctx != NULL); assert(xml != NULL); assert(pxnode != NULL);

    *obj = sbdm = calloc(1, sizeof(sbr_Sbdm));
    assert(sbdm != NULL);

    ato_new_pseudo_guid(ctx, sbdm->guid);
    sbdm->msgsrctype = msgsrctype;
    sbdm->index = index;
    sbdm->xml = xml;
    sbdm->xnode = _process(ctx, sbdm, xml, pxnode);

    ATO_CTX_FN_END(ctx, errcode);
}

void sbr__sbdm_free(sbr_Sbdm *sbdm)
{
    if (sbdm == NULL) return;
    _ato_free(sbdm);
    free(sbdm);
}

void sbr__sbdm_render(ato_Ctx *ctx, sbr_Sbdm *sbdm)
{
    const char *function = "sbr__sbdm_render";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    assert(ctx != NULL);
    if (sbdm) {
        // Some time in the future, when we support multiple SBDMs, we need to added a new SBDM here if xnode NULL.
        if (sbdm->xnode != NULL) {
            ato_xml_createnodevalue(sbdm->xml, sbdm->xnode, NULL, xMessageTypeText, sbdm->messagetype);
            sbr__msgp_render(ctx, sbr_sbdm_party(sbdm, SBR_MSG_SENDER), sbdm->xml, sbdm->xnode);
            sbr__msgp_render(ctx, sbr_sbdm_party(sbdm, SBR_MSG_RECEIVER), sbdm->xml, sbdm->xnode);
            sbr__doc_render(ctx, sbdm->docs, sbdm->xml, sbdm->xnode);
        }
    }

    ATO_CTX_FN_END(ctx, errcode);
}
/*********************************************************************************/
const char *sbr_sbdm_iid(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbdm->guid;
}

sbr_eMsgSourceType sbr_sbdm_msgsourcetype(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbdm->msgsrctype;
}

int sbr_sbdm_setmessagetype(ato_Ctx *ctx, sbr_Sbdm *sbdm, const char *messagetype)
{
    ATO_IGNORE(ctx);
    assert(sbdm != NULL);
    if (sbdm->msgsrctype == SBR_RESPONSE)
        return -1;
    sbdm->messagetype = ato_free(sbdm->messagetype);
    sbdm->messagetype = ato_strdup(messagetype, 0);
    return ATO_ERR_OK;
}

const char *sbr_sbdm_messagetype(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbdm->messagetype;
}

int sbr_sbdm_setparty(ato_Ctx *ctx, sbr_Sbdm *sbdm, sbr_eMsgPartyType partytype, const char *iddesignation, const char *idname)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_sbdm_setparty";
    int errcode = ATO_ERR_OK;
    assert(sbdm != NULL);
    if (sbdm->msgsrctype == SBR_RESPONSE)
        return -1;
    ATO_CTX_FN_START(ctx);
    Try {
        _sbdm_setparty(ctx, sbdm, partytype, iddesignation, idname);
    } Catch (errcode) {
        errcode = sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

sbr_MsgParty *sbr_sbdm_party(sbr_Sbdm *sbdm, sbr_eMsgPartyType partytype)
{
    assert(partytype == SBR_MSG_RECEIVER || partytype == SBR_MSG_SENDER);
    if (partytype == SBR_MSG_RECEIVER)
        return sbdm->receiver;
    else
        return sbdm->sender;
}

size_t sbr_sbdm_doc_count(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbr__doc_count(sbdm->docs);
}
void sbr_sbdm_doc_iterator(sbr_Sbdm *sbdm, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (sbdm == NULL) return;
    sbr__doc_iterator(sbdm->docs, iter);
}
sbr_Doc *sbr_sbdm_doc_firstv(sbr_Sbdm *sbdm)
{
    if (sbdm == NULL) return NULL;
    return sbr__doc_firstv(sbdm->docs);
}
sbr_Doc *sbr_sbdm_doc_nextv(sbr_Sbdm *sbdm)
{
    if (sbdm == NULL) return NULL;
    return sbr__doc_nextv(sbdm->docs);
}

sbr_MsgReceipt *sbr_sbdm_receipt(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbdm->receipt;
}

const char *sbr_sbdm_maxseveritycode(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbdm->maxseveritycode;
}

size_t sbr_sbdm_eventitem_count(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbr__msgevent_count(sbdm->eventitems);
}
void sbr_sbdm_eventitem_iterator(sbr_Sbdm *sbdm, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    assert(sbdm != NULL);
    sbr__msgevent_iterator(sbdm->eventitems, iter);
}
sbr_MsgEventItem *sbr_sbdm_eventitem_firstv(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbr__msgevent_firstv(sbdm->eventitems);
}
sbr_MsgEventItem *sbr_sbdm_eventitem_nextv(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbr__msgevent_nextv(sbdm->eventitems);
}

size_t sbr_sbdm_timestamp_count(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbr__msgts_count(sbdm->timestamps);
}
void sbr_sbdm_timestamp_iterator(sbr_Sbdm *sbdm, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    assert(sbdm != NULL);
    sbr__msgts_iterator(sbdm->timestamps, iter);
}
sbr_MsgTimestamp *sbr_sbdm_timestamp_firstv(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbr__msgts_firstv(sbdm->timestamps);
}
sbr_MsgTimestamp *sbr_sbdm_timestamp_nextv(sbr_Sbdm *sbdm)
{
    assert(sbdm != NULL);
    return sbr__msgts_nextv(sbdm->timestamps);
}

static int _sbr_sbdm_doc_add(ato_Ctx *ctx, sbr_Sbdm *sbdm, sbr_Doc **doc,
                                        const char *validationuri,
                                        const char *governmentid,
                                        const char *businessid,
                                        time_t createtime
                                        )
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_sbdm_doc_add";
    int errcode = ATO_ERR_OK;
    assert(ctx != NULL); assert(sbdm != NULL);
    if (sbdm->msgsrctype == SBR_RESPONSE)
        return -1;
    ATO_CTX_FN_START(ctx);
    Try {
        if (!sbdm->docs)
            sbr__doc_create(ctx, &(sbdm->docs), sbdm->msgsrctype, NULL, NULL, &(sbdm->attachmentcount));
        *doc = sbr__doc_add(ctx, sbdm->docs);
        sbr_doc_setvalidationuri(ctx, *doc, validationuri);
        sbr_doc_setgovernmentid(ctx, *doc, governmentid);
        sbr_doc_setbusinessid(ctx, *doc, businessid);
        sbr_doc_setcreatetime(ctx, *doc, createtime);
    } Catch (errcode) {
        errcode = sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

int sbr_sbdm_doc_add(ato_Ctx *ctx, sbr_Sbdm *sbdm, sbr_Doc **doc,
                                        const char *validationuri,
                                        const char *governmentid,
                                        const char *businessid,
                                        time_t createtime
                                        )
{
    return _sbr_sbdm_doc_add(ctx, sbdm, doc, validationuri, governmentid, businessid, createtime);
}

/*
int sbr_sbdm_doc_remove(ato_Ctx *ctx, sbr_Sbdm *sbdm, sbr_Doc *doc)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_sbdm_doc_remove";
    int errcode = ATO_ERR_OK;
    assert(ctx != NULL); assert(sbdm != NULL);
    if (sbdm->msgsrctype == SBR_RESPONSE)
        return -1;
    ATO_CTX_FN_START(ctx);
    Try {
        sbr__doc_remove(ctx, sbdm->docs, doc);
    } Catch (errcode) {
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}
*/

