#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>

#include "atobase/all.h"
#include "atoakm/all.h"
#include "atostm/all.h"
#include "sbrcsr/all.h"

#include "../common/util.h"
#include "amend.h"

static size_t _strtok_count(const char *src, const char *delim)
{
    size_t count = 0;
    if (*src != '\0') {
        for (; *src != '\0'; src++) {
            if (strchr(delim, *src) != NULL)
                count++;
        }
        count++;
    }
    return count;
}
static size_t _strtok_len(const char *src, const char *delim)
{
    size_t len = 0;
    for (; *src != '\0'; src++) {
        if (strchr(delim, *src) != NULL)
            break;
        len++;
    }
    return len;
}
static const char *_strtok_dup(char *dst, const char *src, const char *delim)
{
    for (; *src != '\0'; src++, dst++) {
        if (strchr(delim, *src) != NULL)
            break;
        *dst = *src;
    }
    *dst = '\0';
    return *src == '\0' ? NULL : ++src;
}


static size_t _strtok2array(char ***dst, const char *src, const char *delim)
{
    size_t tindex = 0, tcount = 0; // token vars
    char **d = NULL;
    const char *s = NULL;
    assert(src != NULL); assert(delim != NULL);
    assert(dst != NULL);

    tcount = _strtok_count(src, delim);
    if (tcount == 0)
        return 0;

    d = calloc(tcount+1, sizeof(void *));
    s = src;
    while (s != NULL) {
        size_t size = _strtok_len(s, delim);
        d[tindex] = calloc(size+1, sizeof(char));
        s = _strtok_dup(d[tindex], s, delim);
        tindex++;
    }
    *dst = d;
    return tcount;
}

static void _strtokfree(char *dst[])
{
    size_t i = 0;
    if (dst == NULL)
        return;
    for (i = 0; dst[i] != NULL; i++)
        free(dst[i]);
    free(dst);
}

/*
// we don't support multiple SBDMs yet
static int amend_request_addsbdms(ato_Ctx *ctx, sbr_Request *request)
{
    int errcode = ATO_ERR_OK;
    sbr_Sbdm *sbdm = NULL;
    char buf[50], buf2[50], buf3[50], buf4[50];
    int maxcount = 3, i = 0;
    sbr_Doc *doc = NULL;
    time_t t;
    time(&t);

    // Add maxcount sbdms
    for (i = 1; i <= maxcount; i++) {
        sbr_DocA *attachment = NULL;
        *buf = *buf2 = *buf3 = *buf4 = '\0';

        sbdm = NULL;
        errcode = sbr_req_sbdm_add(ctx, request, &sbdm);
        if (errcode != ATO_ERR_OK) goto finally;

        sprintf(buf, "sbdm messagetype %d", i);
        sbr_sbdm_setmessagetype(ctx, sbdm, buf);

        sprintf(buf, "sbdm receiver designation %d", i);
        sprintf(buf2, "sbdm receiver scheme %d", i);
        errcode = sbr_sbdm_setparty(ctx, sbdm, SBR_MSG_RECEIVER, buf, buf2);
        if (errcode != ATO_ERR_OK) goto finally;
        sprintf(buf, "sbdm sender designation %d", i);
        sprintf(buf2, "sbdm sender scheme %d", i);
        errcode = sbr_sbdm_setparty(ctx, sbdm, SBR_MSG_SENDER, buf, buf2);
        if (errcode != ATO_ERR_OK) goto finally;

        sprintf(buf, "sbdm doc content %d", i);
        sprintf(buf2, "sbdm doc validation uri %d", i);
        sprintf(buf3, "sbdm doc governmentid %d", i);
        sprintf(buf4, "sbdm doc businessid %d", i);
        errcode = sbr_sbdm_doc_add(ctx, sbdm, &doc, buf, buf2, buf3, buf4, t);
        if (errcode != ATO_ERR_OK) goto finally;

        sprintf(buf, "sbdm doc attachment content %d", i);
        sprintf(buf2, "sbdm doc attachment description %d", i);
        sprintf(buf3, "sbdm doc attachment filename %d", i);
        sprintf(buf4, "sbdm doc attachment contenttype %d", i);
        errcode = sbr_doc_attachment_add(ctx, doc, &attachment, buf, buf2, buf3, buf4);
        if (errcode != ATO_ERR_OK) goto finally;
    }
}
*/

static int _amend_req_attachment(ato_Ctx *ctx, sbr_Doc *doc, const char *attachmentname)
{
    int errcode = ATO_ERR_OK;
    char *buffer = NULL;
    char *b64buffer = NULL;

    if (doc) {
        size_t len = 0;
        ato_String *b64 = NULL;
        sbr_DocA *attachment = NULL;

        if (attachmentname == NULL) { // use the const content
            const char *a_content = "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";
            len = strlen(a_content);
            errcode = sbr_doc_attachment_add(ctx, doc, &attachment, "SMP-PDF.pdf", "SMP-PDF.pdf", "application/pdf");
            if (errcode == ATO_ERR_OK) {
                ato_str_createconst(&b64, a_content, len, TRUE);
            }
        } else { // load binary from a file and encode to base64
            if ((len = utl_loadfile(&buffer, attachmentname, FALSE)) == 0) {
                // This isn't an API call that failed so we can't return an API errcode.
                // Just return something that indicates a problem (i.e. not OK) but which is not an SDK error.
                // Anything less than 0 will do.
                errcode = -9;
                goto finally;
            }
            errcode = sbr_doc_attachment_add(ctx, doc, &attachment, "Attachment", attachmentname, "application/pdf");
            if (errcode != ATO_ERR_OK) goto finally;
            errcode = ato_base64encode_array(ctx, &b64, buffer, len); // This allocates b64.
            if (errcode != ATO_ERR_OK) goto finally;
        }
        if (errcode == ATO_ERR_OK) {
            // ATO_PARAMACTION_FREE tells the attachment object to reference b64 and take "ownership" of it and free when doc/attachment is freed.
            errcode = sbr_doca_setcontent(ctx, attachment, b64, ATO_PARAMACTION_FREE);
        }
    }

finally:
    if (buffer) free(buffer);
    if (b64buffer) free(b64buffer);
    return errcode;
}

static int _add_doc_to_sbdm(ato_Ctx *ctx, const char *party_designation, const char *messagetype, sbr_Sbdm *sbdm, const char *busdocname, const char *validationuri, const char *governmentid, const char *businessid)
{
    int errcode = ATO_ERR_OK;
    sbr_Doc *doc = NULL;
    time_t t;
    char *docfilename = NULL;
    char *attachmentnames = NULL;
    char *buffer = NULL;
    char **attachmentnameslist = NULL;

    if (!sbdm) goto finally;

    assert(busdocname != NULL);
    docfilename = calloc(strlen(busdocname)+1, sizeof(char));
    assert(docfilename != NULL);
    strcpy(docfilename, busdocname);
    attachmentnames = strchr(docfilename, '(');
    if (attachmentnames) {
        char *s = strchr(docfilename, ')');
        *attachmentnames++ = '\0';
        if (s) *s = '\0';
    }

    if (utl_loadfile(&buffer, docfilename, FALSE) == 0) {
        // Return something that indicates a problem (i.e. not OK) but which is not an SDK error.
        // Anything less than 0 will do.
        errcode = -9;
        goto finally;
    }

    time(&t);

    printf("Adding document: %s\n", docfilename);
    errcode = sbr_sbdm_doc_add(ctx, sbdm, &doc, validationuri, governmentid, businessid, t);
    if (errcode != ATO_ERR_OK) goto finally;
    errcode = sbr_doc_setcontent(ctx, doc, buffer, ATO_PARAMACTION_COPY);
    if (errcode != ATO_ERR_OK) goto finally;

    if (strstr(party_designation, "asic") || strstr(messagetype, "asic")) {
        if (attachmentnames) {
            size_t i = 0, numattach = 0;
            numattach = _strtok2array(&attachmentnameslist, attachmentnames, ";");
            for (i = 0; i < numattach; i++) {
                printf("  Adding attachment %" PRINTF_SIZET "u name: %s\n", i,  attachmentnameslist[i]);
                if ((errcode = _amend_req_attachment(ctx, doc, attachmentnameslist[i])) != ATO_ERR_OK) {
                    printf("  Failed to add attachment %" PRINTF_SIZET "u name: %s\n", i,  attachmentnameslist[i]);
                    goto finally;
                }
            }
        }
    }
    if (errcode != ATO_ERR_OK) goto finally;

finally:
    if (docfilename) free(docfilename);
    if (buffer) free(buffer);
    _strtokfree(attachmentnameslist);
    return errcode;
}

// true - the following amend_request method should be broken up to make it more tidy.
int amend_request(ato_Ctx *ctx, sbr_Request *request, const char *messagetype, const char *agency,
                  const char *busdocnames, const char *valuris, const char *govgenid, const char *busgenid, bool forceerror)
{
    sbr_Sbdm *sbdm = NULL;
    sbr_MsgParty *party = NULL;
    const char *party_designation = NULL;
    int errcode = ATO_ERR_OK;
    char **nameslist = NULL;
    char **vurilist = NULL;

    if (!request) goto finally;
    sbdm = sbr_req_sbdm_firstv(request);
    assert(sbdm != NULL);
    if (messagetype)
        errcode = sbr_sbdm_setmessagetype(ctx, sbdm, messagetype);
    if (errcode != ATO_ERR_OK) goto finally;
    if (agency)
        errcode = sbr_sbdm_setparty(ctx, sbdm, SBR_MSG_RECEIVER, agency, SBR_PARTYIDNAME_AGENCY);
    if (errcode != ATO_ERR_OK) goto finally;
    party = sbr_sbdm_party(sbdm, SBR_MSG_RECEIVER);
    assert(party != NULL);
    party_designation = sbr_msgp_iddesignation(party);
    assert(party_designation != NULL);
    if (errcode != ATO_ERR_OK) goto finally;

    {
        const char *governmentid = (govgenid != NULL && strlen(govgenid) > 0 )? govgenid : NULL;
        const char *businessid = (busgenid != NULL && strlen(busgenid) > 0)?  busgenid : "66c6109b-f35e-44d1-8178-4a1b358a77e5";

        if (busdocnames != NULL) {
            size_t i;
            size_t numurls = 0;
            size_t numdocs = _strtok2array(&nameslist, busdocnames, ",");

            if (numdocs == 0){
                errcode = -9;
                printf("Failed to get busdocnames from %s.\n", busdocnames);
                goto finally;
            }
            numurls = _strtok2array(&vurilist, valuris, ",");
            if(numurls < numdocs){
                errcode = -9;
                    printf("Failed to get urls from %s. OR number is less than number of bus docs.\n", valuris);
                    goto finally;
            }
            for (i = 0; i < numdocs; i++) {
                if ((errcode = _add_doc_to_sbdm(ctx, party_designation, messagetype, sbdm, nameslist[i], vurilist[i], governmentid, businessid)) != ATO_ERR_OK){
                    printf("Failed to add %" PRINTF_SIZET "u bus doc. name: %s. vuri = %s", i,  nameslist[i], vurilist[i]);
                    goto finally;
                }
            }
        }
        if (errcode != ATO_ERR_OK) goto finally;

    }

    if (forceerror)
        errcode = sbr_sbdm_setparty(ctx, sbdm, SBR_MSG_RECEIVER, agency, "xxxxxxxxxxxxxxxxxxx");
    if (errcode != ATO_ERR_OK) goto finally;

finally:
    _strtokfree(nameslist);
    _strtokfree(vurilist);
    return errcode;
}

static int _checkcall(int errcode, const char *fn)
{
    const char *result = "Success";
    if (errcode != -1) result = "Failure";
    printf("%s: Expect call to %s to be ignored\n", result, fn);
    return errcode;
}

static int _amend_res_attachments(ato_Ctx *ctx, sbr_Doc *doc)
{
    int errcode = ATO_ERR_OK;
    if (doc) {
        //sbr_DocA *attachment = NULL;
        //errcode = sbr_doc_attachment_add(ctx, doc, &attachment, "", "", "", "");
        //if (_checkcall(errcode, "sbr_doc_attachment_add") != -1)
        //  goto finally;

        //attachment = (sbr_DocA *)sbr_doc_attachment_firstv(doc);
        //errcode = sbr_doc_attachment_remove(ctx, doc, attachment);
        //if (_checkcall(errcode, "sbr_doc_attachment_remove") != -1) goto finally;
    }
    ATO_IGNORE(ctx);
    return errcode;
}

int amend_response(ato_Ctx *ctx, sbr_Response *response)
{
    int errcode = ATO_ERR_OK;
    sbr_Sbdm *sbdm = NULL;
    sbr_Doc *doc = NULL;
    time_t t;
    time(&t);

    if (!response)
        goto finally;

    sbdm = (sbr_Sbdm *)sbr_res_sbdm_firstv(response);

    if (sbdm) {
        printf("Trying to amend Response..\n");
        errcode = sbr_sbdm_setmessagetype(ctx, sbdm, "");
        if (_checkcall(errcode, "sbr_sbdm_setmessagetype") != -1)
            goto finally;

        errcode = sbr_sbdm_setparty(ctx, sbdm, SBR_MSG_RECEIVER, "", "");
        if (_checkcall(errcode, "sbr_sbdm_setparty(SBR_MSG_RECEIVER)") != -1)
            goto finally;

        errcode = sbr_sbdm_setparty(ctx, sbdm, SBR_MSG_SENDER, "", "");
        if (_checkcall(errcode, "sbr_sbdm_setparty(SBR_MSG_SENDER)") != -1)
            goto finally;

        doc = NULL;
        errcode = sbr_sbdm_doc_add(ctx, sbdm, &doc, "", "", "", t);
        if (_checkcall(errcode, "sbr_sbdm_doc_add") != -1)
            goto finally;

        doc = sbr_sbdm_doc_firstv(sbdm);
        //errcode = sbr_sbdm_doc_remove(ctx, sbdm, doc);
        //if (_checkcall(errcode, "sbr_sbdm_doc_remove") != -1) goto finally;
    }

    errcode = _amend_res_attachments(ctx, doc);

finally:
    return errcode >= 0 ? errcode : ATO_ERR_OK; // convert < 0 into "no error";
}
