#ifndef __ATOTST_SYS_H__
#define __ATOTST_SYS_H__

#ifdef __cplusplus
extern "C" {
#endif

/**
Initialise the SDK. This must be called at application startup prior to any threads being started.

This initialises the ATO AKM and ATO Base SDKs (see also ato_akm_init(), ato_base_init()).
See the Dependencies for details on these.

Subsequent calls to this are ignored.

@return ATO_ERR_OK
*/
extern int init(ato_Ctx *ctx);

/**
Deinitialise the SDK. This must be called at application shutdown after threads are finished and
any allocated objects freed.

This deinitialises the ATO AKM, and ATO Base SDKs.
See the Dependencies for details on these.

@param ctx the global context to free during deinitialisation (ignored if NULL).
@param cfgmgr the global cfgmgr to free during deinitialisation (ignored if NULL).
*/
extern void deinit(ato_Ctx *ctx);

/**
Load a keystore from the indicated file into a newly created @ref ato_Keystore instance.

@param ctx the context to use. If NULL, default to the global context - see ato_sys_init().
@param ks the address of the object to create. *ks must be initialised to NULL and must be freed using
ato_ks_free().
@param keystorepath the relative or absolute path to the keystore.
A relative path is relative to the current working folder.
@return ATO_ERR_OK or an error code. Check for an error object using the macro ATO_CTX_ERR().
*/
extern int loadks(ato_Ctx *ctx, ato_Keystore **ks, const char *dirname, const char *filename);

/**
Save the keystore object contents to the indicated file.

@param ks the keystore object.
@param keystorepath the relative or absolute path to the keystore.
A relative path is relative to the current working folder.
@return ATO_ERR_OK or an error code. Check the error object using the macro ATO_ERR(ks) or .
*/
extern int saveks(ato_Ctx *ctx, ato_Keystore *ks, const char *dirname, const char *filename);

/**
Create an STM object using the template file and parameters.

@param ctx the context to use. If NULL, default to the global context - see ato_sys_init().
@param stm the address of the object to create. *stm must be initialised to NULL and must be freed using
ato_stm_free().
@param url the url of the STS service to use. If NULL, the value is read from the configuration file and context
specified in ato_sys_init().
@return ATO_ERR_OK or an error code. Check for an error object using the macro ATO_CTX_ERR().
*/
extern int loadstm(ato_Ctx *ctx, ato_Stm **stm, const char *url, const char *templatefile);

extern int loadcsr(ato_Ctx *ctx, sbr_Csr **csr, const char *url, const char *templatefile, const char *servicetype);

extern int sys_load(ato_Ctx *ctx, ato_Keystore **ks, const char *ksdirname, const char *ksfilename, ato_Stm **stm, const char *stsurl, const char *stmtemplatefile, sbr_Csr **csr, const char *csrurl, const char *csrtemplatefile, const char *csrservicetype);

/**
Convenience method to free the keystore, and STM objects in one go. Any NULL parameter is ignored.
@param ks the keystore object
@param stm the STM object
*/
extern void sys_free(ato_Ctx *ctx, ato_Keystore *ks, ato_Stm *stm, sbr_Csr *csr);

#ifdef __cplusplus
}
#endif

/*! @} */

#endif /*__ATOTST_SYS_H__*/
