#ifndef __ATO_AKM_KS_H__
#define __ATO_AKM_KS_H__

#include <stddef.h>


/** @addtogroup atoakm_api_ks

@{
*/

/**
The keystore object.
*/
typedef struct _ato_Keystore ato_Keystore;

#ifdef __cplusplus
extern "C" {
#endif

/**
Create a keystore object from the XML stored in buffer.

@param ctx the current thread local Context.
@param ks the address of the object to create. *obj must be initialised to NULL.
@param buffer the XML content. If null, an empty keystore is created.
@param properties optional properties to use for the keystore.
An internal copy of this is created, which means that ato_ks_properties() will return the internal copy.

@return ATO_ERR_OK
*/
ATO_AKM_EXPORT int ato_ks_create(ato_Ctx *ctx, ato_Keystore **ks, const char *buffer, const ato_ksProperties *properties);

/**
Free the keystore object if not NULL.

@param ks the keystore object. If NULL do nothing.
@return NULL
*/
ATO_AKM_EXPORT void *ato_ks_free(ato_Keystore *ks);

/**
get the properties object associated with the keystore.

@param ks the keystore object.
@return the properties object associated with the keystore.
*/
ATO_AKM_EXPORT ato_ksProperties *ato_ks_properties(ato_Keystore *ks);

/*
@deprecated { Use ato_ksprop_ver(). }

The version of the keystore.
You can only copy (add) credentials between keystores with the same schema version.

@param ks the keystore object.
@return version in the form 00.00.00.00 with at least 2 levels
*/
ATO_AKM_EXPORT const char *ato_ks_schemaversion(ato_Keystore *ks);

/**
Returns TRUE if the keystore or any of its credentials has been modified since the last created or loaded.

@param ks the keystore object.
@return TRUE or FALSE
*/
ATO_AKM_EXPORT bool ato_ks_ismodified(ato_Keystore *ks);

/**
Returns FALSE if the keystore is in invalid state due to a failed update operation.
This may occur if a change password works for some credentials but not others.
If this occurs, the in-memory keystore should be discarded and reloaded.

@param ks the keystore object.
@return TRUE or FALSE
*/
ATO_AKM_EXPORT bool ato_ks_isvalid(ato_Keystore *ks);

/*
@deprecated
Use ato_ks_ismodified().

@param ks the keystore object.
@return TRUE or FALSE
*/
ATO_AKM_EXPORT bool ato_ks_isdirty(ato_Keystore *ks);

/**
Save the keystore to a buffer.
This does not change the internal state meaning that ismodified() will still be true.

@param ctx the current thread local Context.
@param ks the keystore object.
@param buffer the buffer to allocate. This caller must free this using ato_str_free().

@return ATO_ERR_OK if successful, otherwise
- ATO_AKM_ERR_KEYSTORE_CORRUPTED where the keystore should be discarded and reloaded.
*/
ATO_AKM_EXPORT int ato_ks_save(ato_Ctx *ctx, ato_Keystore *ks, char **buffer);

/**
Return the number of credentials in the keystore matching the filter ato_eCredfilter.

@param ks the keystore object.
@param filter the filter to use - ATO_FILTER_ALL counts all records.

@return number of credentials.
*/
ATO_AKM_EXPORT size_t ato_ks_count(ato_Keystore *ks, ato_eCredfilter filter);

/**
Return the credential corresponding to alias or NULL if not found.

@param ks the keystore object.
@param alias the alias (i.e. xml id) for the credential.

@return the credential or NULL if the alias is not found.
*/
ATO_AKM_EXPORT ato_Credential *ato_ks_credential(ato_Keystore *ks, const char *alias);

/**
Return an allocated array of credentials according to the filter provided.

This is in the original order.

@param ks the keystore object.
@param creds the variable to allocate an array of credentials to. This must be freed by the caller.
Do not, however, free the credentials themselves, as they will be freed when the ato_ks_free() is called.
@param filter the type of credential to include.

@return the size of the array.
*/
ATO_AKM_EXPORT size_t ato_ks_credentials(ato_Keystore *ks, ato_Credential ***creds, ato_eCredfilter filter);

/**
Delete the credential corresponding to the alias or all credentials if alias is NULL.

@param ks the keystore object.
@param alias the alias (i.e. xml id) for the credential. If NULL, delete all credentials.

@return ATO_ERR_OK if successful
*/
ATO_AKM_EXPORT size_t ato_ks_delete(ato_Keystore *ks, const char *alias);

/**
Test if the password is correct without accessing any particular credential.

@param ctx the current thread local Context.
@param ks the keystore object.
@param pwd the password

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_PWD if the password is wrong
- ATO_AKM_ERR_PWDPOLICY if the keystore is empty and the password doesn't meet strength requirements
For any other error, also check for the presence of a nested inner error.
*/
ATO_AKM_EXPORT int ato_ks_checkcorrectpwd(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd);

/**
@deprecated { Use ato_ks_checkcorrectpwd(). }

Test if the password is correct without accessing any particular credential.

If the keystore is empty this will be true if it matches the password policy.

@param ctx the current thread local Context.
@param ks the keystore object.
@param correct the address of the boolean to set to TRUE or FALSE
@param pwd the password

@return ATO_ERR_OK or an errcode.
*/
ATO_AKM_EXPORT int ato_ks_iscorrectpwd(ato_Ctx *ctx, ato_Keystore *ks, bool *correct, const char *pwd);

/**
Change the password for all credentials in the keystore - see ato_ks_pwdpolicy_desc() for password rules.

If multi-threading, ensure that the keystore is not involved in any current ato_ks_copycredential operation.

@param ctx the current thread local Context.
@param ks the keystore
@param pwd the password
@param newpwd the new password

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_PWD if the password is wrong
- ATO_AKM_ERR_PWDPOLICY if the new password doesn't meet strength requirements
- ATO_AKM_ERR_GENERAL for any other error.
For any error, a nested inner error, if any, will contain more details.
*/
ATO_AKM_EXPORT int ato_ks_changepwd(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd, const char *newpwd);

/**
Check if the password matches the password policy - see ato_ks_pwdpolicy_desc().

It is possible this may vary in future versions of the keystore.

@param ks the keystore. If NULL, the latest policy is used.
@param pwd the password

@return TRUE or FALSE.
*/
ATO_AKM_EXPORT bool ato_ks_pwdpolicy_isvalid(ato_Keystore *ks, const char *pwd);

/**
Return the password policy as a string.

Currently a password must contain:
- no whitespace
- at least 1 uppercase character
- at least 1 lowercase character
- at least 1 digit OR 1 punctuation character
- at least 10 printable characters

@param ks the keystore. If NULL, the latest policy description (above) is returned.

@return The above content as a string - including embedded newlines.
*/
ATO_AKM_EXPORT const char *ato_ks_pwdpolicy_desc(ato_Keystore *ks);

/**
Add a credential from another keystore to a destination keystore - replaces ato_ks_copycredential.

If multi-threading, ensure that either keystores are not involved in any ato_ks_changepwd operation.

@param ctx the current thread local Context.
@param ks the keystore.
@param pwd the password of the keystore.
@param cr the credential to add to the keystore.
@param crpwd the password of the credential (source keystore); can be NULL if same as destination keystore.
@param isreplacable used if the alias already exists in the dstks. If TRUE then replace it,

@return ATO_ERR_OK(0) if successful, otherwise
- ATO_AKM_ERR_CREDNOTFOUND if the credential does not exist in the source store
- ATO_AKM_ERR_ALIASEXISTS if isreplacable is FALSE and credential alias already exists in the destination store
- ATO_AKM_ERR_BADPWD if the source credential password is wrong
- ATO_AKM_ERR_PWDDEST if the destination store password is wrong
- ATO_AKM_ERR_PWDPOLICY if this is a new password that doesn't meet strength requirements
- ATO_AKM_ERR_COPYCREDENTIAL if a general error occured - see message text for details
*/
ATO_AKM_EXPORT int ato_ks_addcredential(ato_Ctx *ctx, ato_Keystore *ks, const char *pwd, ato_Credential *cr, const char *crpwd, bool isreplacable);

#ifdef __cplusplus
}
#endif

/*! @} */

#endif /*__ATO_AKM_KS_H__*/
