#ifndef __ATO_AKM_LIB_H__
#define __ATO_AKM_LIB_H__

/** @addtogroup atoakm_api_lib

@{
*/

#ifdef __cplusplus
extern "C" {
#endif

/**
Return the version of the keystore Manager API.
@return version in the form major.minor.patch.tweak (not all levels may be present)
*/
ATO_AKM_EXPORT const char *ato_akm_version(void);

/**
Initialise the SDK. This must be called at application startup prior to any threads being started.

ato_base_init() must be called first.

@param ctx the current thread local Context.
@param flag used to control the initialisation process - use ATO_AKM_INIT_ALL unless advised otherwise.

@return ATO_ERR_OK
*/
ATO_AKM_EXPORT int ato_akm_init(ato_Ctx *ctx, unsigned short flag);
/**
Deinitialise the SDK. This must be called at application shutdown after threads are finished and
any allocated objects freed.

ato_base_deinit() should be called after all other deinitialisation routines are called.
*/
ATO_AKM_EXPORT void ato_akm_deinit(void);

/**
The default directory/folder for the keystore. This is the user's home, platform/OS specific, folder.
@return the default home keystore folder.
*/
ATO_AKM_EXPORT const char *ato_akm_default_dir(void);

/**
The default keystore filename.
@return the default keystore filename.
*/
ATO_AKM_EXPORT const char *ato_akm_default_ksfile(void);

/**
The full path for the default keystore including the default filename.
@return the default keystore path.
*/
ATO_AKM_EXPORT const char *ato_akm_default_kspath(void);

/**
Convenience method to load a keystore from the give path "dirname/filename" into a buffer - caller must free.

If dirname and/or ato_akm_default_ksfile() are NULL then try to discover the default values from the
config file else use the hard coded values provided by ato_akm_default_dir()/ato_akm_default_ksfile().

@param ctx the current thread local Context.
@param buffer the address of the buffer that is allocated if successful.
@param dirname the directory to load from
@param filename the keystore filename to load

@return ATO_ERR_OK if successful or error.
*/
ATO_AKM_EXPORT int ato_akm_loadks(ato_Ctx *ctx, char **buffer, const char *dirname, const char *filename);

/**
Convenience method to save a keystore buffer to the given path "dirname/filename".

If dirname and/or ato_akm_default_ksfile() are NULL then try to discover the default values from the
config file else use the hard coded values provided by ato_akm_default_dir()/ato_akm_default_ksfile().

@param ctx the current thread local Context.
@param buffer the buffer containing the raw keystore XML.
@param dirname the directory to save to
@param filename the keystore filename to save

@return ATO_ERR_OK if successful or error.
*/
ATO_AKM_EXPORT int ato_akm_saveks(ato_Ctx *ctx, const char *buffer, const char *dirname, const char *filename);

#ifdef __cplusplus
}
#endif

/*! @} */

#endif /*__ATO_AKM_LIB_H__*/
