#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>
#include <string.h>

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"
#include "atolib.h"
#include "atoksprop.h"

#include "atorenew.h"
#include "atocredential.h"

static const char *_library = ATO_AKM_LIBRARY;
static const char *_module = ATO_AKM_MODULE_CRR;
static unsigned long _moduleid = ATO_AKM_MODULEID_CRR;

/*********************************************************************************/

struct _ato_Crr {
    ato_ksProperties *ksprop;
    char *requestid;

    char *pbemode;
    char *fingerprint;
    char *legalName;
    char *creationDate;

    ato_Crypto *crypto;
    bool isCryptoExternallyOwned; // too inefficient to be cloning crypto
    const char *abn;
    const char *name1;
    const char *name2;
    const char *pid;
    ato_String *p10;
    ato_String *p7;
    ato_String *p8;

    ato_String *salt;
    ato_String *requestData;
    char *responseData;
    const char *type;

    // Current credential values
    ato_Crypto *cr_crypto;
};

/*********************************************************************************/
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

//////////////////////////// _GenerateSendData ////////////////////////////
static const char *quote = "\"";
static void _AddName(ato_String *data, const char *str, const char *postfix) {
    ato_str_add(data, quote); ato_str_add(data, str); ato_str_add(data, quote); ato_str_add(data, ":"); ato_str_add(data, postfix);
}
static void _AddValue(ato_String *data, const char *str, const char *postfix) {
    ato_str_add(data, quote); ato_str_add(data, str); ato_str_add(data, quote); ato_str_add(data, postfix);
}
static void _AddField(ato_String *data, const char *name, const char *value, const char *postfix) {
    _AddName(data, name, ""); _AddValue(data, value, postfix);
}

static void _AddXmlAppDetails(ato_String *data) {
    if (data == NULL) { return; }
    _AddName(data, "softwareinfo", "{");
    _AddName(data,   "application", "{");
    _AddField(data,    "organisation", ato_si_organisation(), ",");
    _AddField(data,    "product", ato_si_product(), ",");
    _AddField(data,    "version", ato_si_version(), ",");
    _AddField(data,    "timestamp", ato_si_timestamp(), "},");
    _AddName(data,   "sdk", "{");
    _AddField(data,    "name", ATO_AKM_LIBRARY, ",");
    _AddField(data,    "abiVersion", ato_akm_version(), ",");
    _AddField(data,    "fileVersion", ato_akm_version(), "},");
}

static ato_String *_GenerateSendData(ato_Crr *crr, ato_String *p10) {
    const char *data_hdr = "{";
    ato_String *data = NULL;
    ato_str_createconst(&data, data_hdr, strlen(data_hdr), TRUE);
    _AddXmlAppDetails(NULL); //data);
    _AddField(data, "clientRequestId", crr->requestid, ",");
    _AddField(data, "cmsP10", ato_str_value(p10), "}");
    ato_str_add_complete(data);
    return data;
}

static void _clear_data(ato_Crr *crr)
{
    crr->fingerprint = ato_free(crr->fingerprint);
    crr->legalName = ato_free(crr->legalName);
    crr->creationDate = ato_free(crr->creationDate);

    if (!crr->isCryptoExternallyOwned) { crr->crypto = ato_crypto_free(crr->crypto); crr->isCryptoExternallyOwned = FALSE; }
    crr->p10 = ato_str_free(crr->p10);
    crr->p7 = ato_str_free(crr->p7);
    crr->p8 = ato_str_free(crr->p8);
    crr->salt = ato_str_free(crr->salt);
    crr->requestData = ato_str_free(crr->requestData);
    crr->responseData = ato_free(crr->responseData);
    crr->pbemode = ato_free(crr->pbemode);
}

//////////////////////////// _check_for_errors ////////////////////////////
//Checks for a response containing an error. NOT network errors, so assumes we have a valid response.
static void _check_for_errors(ato_Ctx *ctx, const char *response) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_check_for_errors";
    int errcode = ATO_ERR_OK;
    char *id, *msg;

    id = ato_substrmatch(response, "<errorId>", "</errorId>");
    if (id == NULL) { return; }
    msg = ato_substrmatch(response, "<errorMessage>", "</errorMessage>");
    errcode = ATO_CTX_VNEWERR(ctx, ATO_AKM_ERR_RENEW, ato_strlen(id) + ato_strlen(msg), "Credential Renewal error. ID = %s, message = %s", id, ato_value(msg, ""));
    ato_free(id); ato_free(msg);
    Throw errcode;
}

//http://stackoverflow.com/questions/18267803/how-to-correctly-convert-a-hex-string-to-byte-array-in-c
static ato_String *_hex2data(const char *hstr)
{
    size_t hlen = ato_strlen(hstr), i = 0, blen = 0;
    char *bytes = NULL, *endptr = NULL;
    ato_String *sb = NULL;

    if (hlen == 0 || (strlen(hstr) % 2) != 0) {
        return NULL;
    }
    bytes = ato_alloc(hlen/2+1);
    for(i = 0; i < hlen; i += 2, hstr += 2) {
        char buf[5] = {'0', 'x', 0, 0, 0};
        buf[2] =  hstr[0]; buf[3] =  hstr[1];
        bytes[blen++] = (char)strtol(buf, &endptr, 0);
        if (endptr[0] != '\0') {
            return ato_free(bytes);
        }
    }

    return ato_str_create(&sb, bytes, blen, FALSE);
}

static char *_hex2b64(ato_Ctx *ctx, const char *fpsrc) {
    char *fp = NULL;
    ato_String *data = _hex2data(fpsrc);
    if (data == NULL) {
        return NULL;
    } else {
        ato_String *b64 = NULL, *b64z = NULL;
        ato_base64encode(ctx, &b64, data);
        ato_str_free(data);
        ato_str_dup2(&b64z, b64, TRUE); // We need this as null terminated string
        ato_str_free(b64);
        fp = ato_str_valueasowner(b64z);
        ato_str_free(b64z);
    }
    return fp;
}

//////////////////////////// _process ////////////////////////////
/*
{
    "messages" : [ ],
    "credentialId" : "1164",
    "serialNumber" : "403489683511146307907680144499754261694800328681",
    "pkcs7" : "MIAGCSqGSI............VsFeay6RLEyiZJR3ir8AAAAAAAA=",
    "sha1FingerPrint" : "9814a3e7f7cd46768e585da1579988469bc2304a",
    "messageId" : "a47933b4-c5e1-4280-b6a4-dbc0947f321d",
    "clientRequestId" : "atoakm-99"
}
*/

static char *get_json_value(const char *response, const char *name) {
    const char *value = NULL;
    if (response == NULL) { return NULL; }
    value = strstr(response, name);
    if (value == NULL) { return NULL; }
    value += strlen(name);
    return ato_substrmatch(value, "\"", "\"");
}

static void _process(ato_Ctx *ctx, ato_Crr *crr, const char *response) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_process";
    int errcode = ATO_ERR_OK;
    const char *name = NULL;
    char *p7cB64 = NULL, *id = NULL;
    time_t now;
    char bnow[101];

    now = get_now();
    //time(&now);
    ato_strftime_utc(bnow, 100, &(now), FALSE);
    crr->creationDate = ato_strdup(bnow, 0);

    Try {
        name = "\"clientRequestId\"";
        id = get_json_value(response, name);
        if (id == NULL) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, "Cannot find response clientRequestId");
        }
        if (!ato_streq(id, crr->requestid)) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, "Response clientRequestId does not match request clientRequestId");
        }
        //printf("\n%s=%s\n", name, id);

        name = "\"sha1FingerPrint\"";
        crr->fingerprint = get_json_value(response, name);
        if (crr->fingerprint == NULL) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, "Cannot find sha1FingerPrint");
        }
        if (strlen(crr->fingerprint) == 40) {
            char *fp = crr->fingerprint;
            crr->fingerprint = _hex2b64(ctx, fp);
            ato_free(fp);
            if (crr->fingerprint == NULL) {
                Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, "Failed to convert hex fingerprint to base64");
            }
        }
        //printf("\n%s=%s\n", name, crr->fingerprint);

        //crr->legalName = ato_substrmatch(response, "<legalName>", "</legalName>");

        name = "\"pkcs7\"";
        p7cB64 = get_json_value(response, name);
        if (p7cB64 == NULL) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, "Cannot find pkcs7");
        }
        //printf("\n%s=%s\n", name, p7cB64);
        // server seems to return non-printable (binary) characters at end of p7, so trim them
        //ato_trim(p7cB64, 0);
        ato_str_create(&(crr->p7), p7cB64, strlen(p7cB64), TRUE);

    } Catch (errcode) {
        ato_free(p7cB64);
    }
    ato_free(id);

    if (errcode != ATO_ERR_OK) { Throw errcode; }
}

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level) {
    _loglevel = level;
}

/*********************************************************************************/
int ato__crr_init(void) {
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__crr_deinit(void)
{
}

/*********************************************************************************/
void ato__crr_create(ato_Ctx *ctx, ato_Crr **obj, ato_ksProperties *ksprop, const char *id, const char *type, ato_Crypto *crypto)
{
    ato_Crr *crr = NULL;
    ATO_ASSERT_ISNOTALLOCATED(obj);

    assert(ksprop != NULL);
    assert(id != NULL);
    assert(type != NULL);
    assert(crypto != NULL);

    *obj = crr = calloc(1, sizeof(ato_Crr));
    assert(crr != NULL);
    crr->ksprop = ksprop;
    crr->cr_crypto = crypto;

    crr->requestid = ato_strdup(id, 0);
    crr->type = type;

    ATO_IGNORE(ctx);
}

void ato__crr_createnew(ato_Ctx *ctx, ato_Crr **obj, ato_ksProperties *ksprop, const char *id, const char *type, const char *abn, const char *name1, const char *name2, const char *pid) {
    ato_Crr *crr = NULL;
    ATO_ASSERT_ISNOTALLOCATED(obj);

    assert(ksprop != NULL);
    assert(id != NULL);
    assert(type != NULL);
    assert(abn != NULL);
    assert(name1 != NULL);

    *obj = crr = calloc(1, sizeof(ato_Crr));
    assert(crr != NULL);
    crr->ksprop = ksprop;
    crr->abn = abn;
    crr->name1 = name1;
    crr->name2 = name2;
    crr->pid = pid;

    crr->requestid = ato_strdup(id, 0);
    crr->type = type;

    ATO_IGNORE(ctx);
}

void *ato__crr_free(ato_Crr *crr)
{
    if (crr == NULL) return NULL;

    crr->requestid = ato_free(crr->requestid);

    _clear_data(crr);

    free(crr);
    return NULL;
}

/*********************************************************************************/
const char *ato__crr_name1(ato_Crr *crr) { return crr ? crr->name1 : NULL; }
const char *ato__crr_name2(ato_Crr *crr) { return crr ? crr->name2 : NULL; }
const char *ato__crr_abn(ato_Crr *crr) { return crr ? crr->abn : NULL; }
const char *ato__crr_fingerprint(ato_Crr *crr) { return crr ? crr->fingerprint : NULL; }
const char *ato__crr_legalName(ato_Crr *crr) { return crr ? crr->legalName : NULL; }
const char *ato__crr_creationDate(ato_Crr *crr) { return crr ? crr->creationDate : NULL; }
const char *ato__crr_pbemode(ato_Crr *crr) { return crr ? crr->pbemode : NULL; }
ato_Crypto *ato__crr_crypto_asowner(ato_Crr *crr) {
    if (crr == NULL) { return NULL; }
    crr->isCryptoExternallyOwned = TRUE;
    return crr->crypto;
}

ato_String *ato__crr_requestdata(ato_Crr *crr) { return crr ? crr->requestData : NULL; }
ato_String *ato__crr_b64p10(ato_Crr *crr) { return crr ? crr->p10 : NULL; }

ato_String *ato__crr_salt(ato_Crr *crr) { return crr ? crr->salt : NULL; }
ato_String *ato__crr_p8(ato_Crr *crr) { return crr ? crr->p8 : NULL; }
const char *ato__crr_requestid(ato_Crr *crr) { return crr ? crr->requestid : NULL; }

//static bool ato_crypto_generateP10new(ato_Ctx *ctx, const char *type, const char *abn, const char *name1, const char *name2, const char *pid, ato_PwdKdf *pk, ato_String *salt, ato_String **p10, ato_String **p8) {
//    ATO_IGNORE(ctx);
//    ATO_IGNORE(type);
//    ATO_IGNORE(abn);
//    ATO_IGNORE(name1);
//    ATO_IGNORE(name2);
//    ATO_IGNORE(pid);
//    ATO_IGNORE(pk);
//    ATO_IGNORE(salt);
//    ATO_IGNORE(p10);
//    ATO_IGNORE(p8);
//    return FALSE;
//}

ato_String *ato__crr_renew_generate(ato_Ctx *ctx, ato_Crr *crr, ato_PwdKdf *pk) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato__crr_renew_generate";
    char ca_name[240];
    char *saltch = calloc(50, sizeof(char));
    const char *keylen_id = ato_ksprop_var(crr->ksprop, "keylen");
    int keylen = 2048;

    if (keylen_id != NULL) {
        if (keylen_id[0] == '3') {
            keylen = 3072;
        } else if (keylen_id[0] == '4') {
            keylen = 4096;
        }
    }

    *ca_name = '\0';
    _clear_data(crr);
    ato_new_salt(ctx, saltch);
    ato_str_create(&(crr->salt), saltch, strlen(saltch), TRUE); // now owns saltch

    if (crr->cr_crypto == NULL) {
        if (!ato_crypto_generateP10new(ctx, keylen, crr->type, crr->abn, crr->name1, crr->name2, crr->pid, pk, crr->salt, &(crr->p10), &(crr->p8))) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, "Failed to generate P10");
        }
    } else {
        //printf("\n\n%s\n\n", "ato_crypto_generateP10");
        if (!ato_crypto_generateP10(ctx, keylen, crr->cr_crypto, pk, crr->salt, &(crr->p10), &(crr->p8))) {
            Throw ATO_CTX_NEWERR(ctx, ATO_AKM_ERR_RENEW, "Failed to generate P10");
        }
    }

    crr->requestData = _GenerateSendData(crr, crr->p10);
    return crr->requestData;
}

void ato__crr_renew_process(ato_Ctx *ctx, ato_Crr *crr, const char *responseData) {
    if (responseData == NULL) { responseData = crr->responseData; }
    _check_for_errors(ctx, responseData);
    _process(ctx, crr, responseData);
    ato_crypto_createstr_ref(ctx, &(crr->crypto), crr->p7, crr->p8, crr->salt); // hand over ownership of salt, p7, p8 to crypto
    crr->p7 = NULL; crr->p8 = NULL; crr->salt = NULL;
}
/*
void ato__crr_renew(ato_Ctx *ctx, ato_Crr *crr, ato_PwdKdf *pk) {
    ato__crr_renew_generate(ctx, crr, pk);
    ato__crr_renew_send(ctx, crr);
    ato__crr_renew_process(ctx, crr, NULL);
}
*/
