#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>
#include <ctype.h>
#include <sys/stat.h>

#include <curl/curl.h>

#include "akversion.h"

#include "atobase/all.h"
#include "atoakm/all.h"

#include "../common/util.h"
#include "../common/log.h"

#include "sys.h"
#include "misc.h"

ato_Ctx *g_ctx = NULL;
ato_Log *g_log = NULL;
ato_eLoglevel arg_loglevel = ATO_LOG_ALL;

static const char *_DEFAULT_ksdir = ".";
static const char *_DEFAULT_kspwd = "Password1!";
static const char *_DEFAULT_ksalias = "ABRD:27809366375_BetaDevice";

static const char *_pwd2 = "Password2!";

static const char *arg_organisation = NULL;
static const char *arg_installprefix = NULL;
static bool arg_logfd = FALSE;
static const char *arg_logpath = NULL;
static const char *arg_netlogflag = NULL;
static const char *arg_proxyurl = NULL;
static const char *arg_proxyauthtype = NULL;
static const char *arg_proxybypass = NULL;
static const char *arg_proxyuser = NULL;
static const char *arg_proxypwd = NULL;
static const char *arg_proxypwdfile = NULL;
static const char *arg_sslversion = NULL;
static const char *arg_sslcainfo = NULL;

static const char *arg_logcr = NULL;
static const char *arg_ksdir = NULL;
static const char *arg_ksfile = NULL;
static const char *arg_kspwd = NULL;
static const char *arg_ksalias = NULL;
static const char *arg_chpwd = NULL;
static size_t arg_creds = 3;
static bool arg_crypto = FALSE;
static const char *arg_crexpiryperiod = NULL;
static const char *arg_akmtests = NULL;

static void _addargs(void)
{
        ato_arg_add("organisation", TRUE, NULL, NULL, "Name of the organisation or person running this");
        ato_arg_add("installprefix", FALSE, NULL, NULL, "Where to look for dependent library resources, if any");
        ato_arg_add("logfd", FALSE, NULL, NULL, "Some modules will dump data to files for debugging purposes");
        ato_arg_add("logpath", FALSE, "messages.log", NULL, "Set the log path");
        ato_arg_add("loglevel", FALSE, "d", "off|fatal|err|warn|info|debug|trace|all", "Set the loglevel - only need first letter");
        ato_arg_add("netlogflag", FALSE, NULL, NULL, "the network library has additional loglevel settings");
        ato_arg_add("proxyurl", FALSE, NULL, NULL, "proxy server url");
        ato_arg_add("proxyauthtype", FALSE, NULL, NULL, "proxy server authentication type: any (default); basic; digest; ntlm");
        ato_arg_add("proxybypass", FALSE, NULL, NULL, "proxy server bypass list");
        ato_arg_add("proxyuser", FALSE, NULL, NULL, "proxy server user name - if not supplied read the settings from the current config file context");
        ato_arg_add("proxypwd", FALSE, NULL, NULL, "proxy server user password");
        ato_arg_add("proxypwdfile", FALSE, NULL, NULL, "read proxyuser (1st line) and proxypwd (2nd line) from this file");
        ato_arg_add("sslversion", FALSE, NULL, NULL, "CURL options for SSL/TLS(DEFAULT,TLSv1,SSLv2,SSLv3,TLSv1_0,TLSv1_1,TLSv1_2)");
        ato_arg_add("sslcainfo", FALSE, NULL, NULL, "Path to CURL CA store or 'off' to diable SSL - see CURL option CURLOPT_CAINFO");

        ato_arg_add("logcr", FALSE, "d", "off|fatal|err|warn|info|debug|trace|all", "Set the loglevel for the credential module");
        ato_arg_add("ksdir", FALSE, ".", NULL, "Folder to look for the kestore");
        ato_arg_add("ksfile", FALSE, ato_akm_default_ksfile(), NULL, "Filename of the keystore");
        ato_arg_add("kspwd", FALSE, "Password1!", NULL, "Password for the keystore");
        ato_arg_add("ksalias", FALSE, NULL, NULL, "Alias to load from the keystore");
        ato_arg_add("chpwd", FALSE, NULL, NULL, "keystore password to change to");
        ato_arg_add("creds", FALSE, "3", NULL, "Number of credentials to list");
        ato_arg_add("crypto", FALSE, NULL, NULL, "Dump private key details");
        ato_arg_add("crexpiryperiod", FALSE, "0", NULL, "Credential expiry period in months - minimum 14");
        ato_arg_add("akmtests", FALSE, NULL, NULL, "AKM tests to run: ',dumpks,loadsaveks,renew,changepwd,copycred,deleteallcrs,' or all if not supplied");
}

static void _initargs(void)
{
        arg_organisation = ato_arg_value("organisation");
        arg_installprefix = ato_arg_value("installprefix");;
        arg_logfd = ato_arg_value_bool("logfd");
        arg_logpath = ato_arg_value("logpath");
        arg_loglevel = ato_log_name2level(ato_arg_value("loglevel"));
        arg_netlogflag = ato_arg_value("netlogflag");
        arg_proxyurl = ato_arg_value("proxyurl");
        arg_proxyauthtype = ato_arg_value("proxyauthtype");
        arg_proxybypass = ato_arg_value("proxybypass");
        arg_proxyuser = ato_arg_value("proxyuser");
        arg_proxypwd = ato_arg_value("proxypwd");
        arg_proxypwdfile = ato_arg_value("proxypwdfile");
        arg_sslversion = ato_arg_value("sslversion");
        arg_sslcainfo = ato_arg_value("sslcainfo");

        arg_logcr = ato_arg_value("logcr");
        arg_ksdir = ato_arg_value("ksdir");
        arg_ksfile = ato_arg_value("ksfile");
        arg_kspwd = ato_arg_value("kspwd");
        arg_ksalias = ato_arg_value("ksalias");
        arg_chpwd = ato_arg_value("chpwd");
        arg_creds = (size_t) ato_arg_value_int("creds");
        arg_crypto = ato_arg_value_bool("crypto");
        arg_crexpiryperiod = ato_arg_value("crexpiryperiod");
        arg_akmtests = ato_arg_value("akmtests");
}

static bool _streqi(const char *src, const char *src2)
{
        if (src == NULL && src2 == NULL)
                return TRUE;
        if ((src == NULL && src2 != NULL) || (src != NULL && src2 == NULL))
                return FALSE;
        if (strlen(src) != strlen(src2))
                return FALSE;
        while (*src2 != '\0') {
                if (tolower((int)*src) != tolower((int)*src2))
                        return FALSE;
                src++; src2++;
        }
        return TRUE;
}

static void _dump_error(ato_Ctx *ctx)
{
        ato_Err *err = ato_ctx_err(ctx);
        if (!err || ato_err_code(err) == ATO_ERR_OK)
                return;
        printf("\n");
        printf("-----------------------------------------\n");

        for (; err; err = ato_err_inner(err)) {
                printf("%d: %s\n", ato_err_code(err), ato_err_msg(err));
                printf("-----------------------------------------\n");
        }
        printf("\n");
}

static bool _test_dumpks(void)
{
        bool ok = FALSE;
        size_t i = 0, count = 0, fcount = 0;
        ato_Keystore *ks = NULL;
        ato_Credential **crs = NULL;
        const char *ver = NULL;
        const char *ksdir = NULL, *kspwd = NULL;
    bool pwdok = FALSE;

    printf("\n");
    printf("Default keystore location=%s\n", ato_akm_default_dir());
        printf("================= dumpks =======================\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
        kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);

        if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
                goto finally;

        ver = ato_ks_schemaversion(ks);
        count = ato_ks_count(ks, ATO_FILTER_ALL);
    pwdok = (ato_ks_checkcorrectpwd(g_ctx, ks, kspwd) == ATO_ERR_OK);

        printf("ks count = %" PRINTF_SIZET "u, pwd valid = %d\n", ato_ks_count(ks, ATO_FILTER_ALL), pwdok);

        fcount = ato_ks_credentials(ks, &crs, ATO_FILTER_ALL);
        for (i = 0; i < fcount && i < arg_creds; i++) {
                dump_cred(g_ctx, crs[i], kspwd, arg_crypto);
        }

        printf("Version keystore=%s\n", ver);
        printf("Number of credentials = %" PRINTF_SIZET "u\n", count);
        printf("Number of filtered credentials = %" PRINTF_SIZET "u\n", fcount);

        printf("\n");
        printf("keystore backed up to %s\n", "keystore_out_backup.xml");
        if (saveks(g_ctx, ks, ksdir, "keystore_out_backup.xml") != ATO_ERR_OK)
                goto finally;

        ok = TRUE;
finally:
    if (crs != NULL) { crs = ato_free(crs); }
        ato_ks_free(ks);
        return ok;
}

static bool _test_loadsaveks(void)
{
        bool ok = FALSE;
        size_t fcount = 0;
        ato_Keystore *ks = NULL;
        ato_Credential **crs = NULL;
        const char *ksdir = NULL, *ksalias = NULL;

        printf("\n");
        printf("================= loadsaveks =======================\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
        ksalias = (arg_ksalias != NULL ? arg_ksalias : _DEFAULT_ksalias);

        printf("keystore at %s/%s\n",ksdir, arg_ksfile);
        fflush(stdout);
        if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
                goto finally;

        if (saveks(g_ctx, ks, ksdir, "keystore_out_backup.xml") != ATO_ERR_OK)
                goto finally;

        fcount = ato_ks_credentials(ks, &crs, ATO_FILTER_DEVICE);
        if (fcount == 0) {
            printf("No device credentials found\n");
            goto finally;
        }

        assert(ksalias != NULL);
        printf("\n");
        printf("Locating credential for alias=%s\n", ksalias);
        if (ksalias != NULL) {
                dump_alias_if_found(ks, ksalias, TRUE);
                printf("\n");

                printf("Deleting credential for alias=%s: expect %s\n", ksalias, "'failed to find'");
                if (ato_ks_delete(ks, ksalias) == 0)
                        goto finally;
                dump_alias_if_found(ks, ksalias, FALSE);
                printf("\n");

                printf("Reloading without save: expect %s\n", "'found'");
                ato_ks_free(ks); ks = NULL;
                if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
                        goto finally;
                dump_alias_if_found(ks, ksalias, TRUE);
                printf("\n");

                printf("Deleting credential for alias=%s: expect %s\n", ksalias, "'failed to find'");
                if (ato_ks_delete(ks, ksalias) == 0)
                        goto finally;
                dump_alias_if_found(ks, ksalias, FALSE);
                printf("\n");

                printf("Saving: expect %s\n", "'failed to find'");
                if (saveks(g_ctx, ks, ksdir, "keystore_out_update.xml") != ATO_ERR_OK)
                        goto finally;
                dump_alias_if_found(ks, ksalias, FALSE);
                printf("\n");
        }

        ok = TRUE;
finally:
    if (crs != NULL) { crs = ato_free(crs); }
        ato_ks_free(ks);
        return ok;
}

static bool _pwd_check(ato_Keystore *ks, ato_Credential *cr, const char *pwd, int expected) {
    bool valid = FALSE, resultok = FALSE;
    const char *msg = expected == ATO_ERR_OK ? "good" : "bad";

    printf("check cr pwd %s [%s]\n", msg, pwd);
    if (ato_cr_checkcorrectpwd(g_ctx, cr, pwd) != expected) { return FALSE; }
    printf("is cr pwd %s [%s]\n", msg, pwd);
    if (ato_cr_iscorrectpwd(g_ctx, cr, &valid, pwd) != expected) { return FALSE; }
    printf("is cr pwd %s (valid) [%s]\n", msg, pwd);
    resultok = expected == ATO_ERR_OK ? valid : !valid;
    if (!resultok) { return FALSE; }

    printf("check ks pwd %s [%s]\n", msg, pwd);
    if (ato_ks_checkcorrectpwd(g_ctx, ks, pwd) != expected) { return FALSE; }
    printf("is ks pwd %s [%s]\n", msg, pwd);
    if (ato_ks_iscorrectpwd(g_ctx, ks, &valid, pwd) != expected) { return FALSE; }
    printf("is ks pwd %s (valid) [%s]\n", msg, pwd);
    resultok = expected == ATO_ERR_OK ? valid : !valid;
    if (!resultok) { return FALSE; }
    return TRUE;
}

static bool _test_sign(void)
{
    bool ok = FALSE;
    ato_Keystore *ks = NULL;
    ato_Credential *cr = NULL;
    const char *ksdir = NULL, *kspwd = NULL, *ksalias = NULL;
    const char *data = "some data", *badpwd = "ascmceybsaxsab";
    ato_String *sdata = NULL;

    printf("\n");
    printf("================= sign =======================\n");

    ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
    kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
    ksalias = (arg_ksalias != NULL ? arg_ksalias : _DEFAULT_ksalias);

    if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
        goto finally;

    if ((cr = ato_ks_credential(ks, ksalias)) == NULL)
        goto finally;

    if (!_pwd_check(ks, cr, kspwd, ATO_ERR_OK)) { goto finally; }
    if (!_pwd_check(ks, cr, badpwd, ATO_AKM_ERR_BADPWD)) { goto finally; }

    printf("test sign with good pwd [%s]\n", kspwd);
    if (ato_cr_sign(g_ctx, cr, &sdata, kspwd, data, strlen(data)) != ATO_ERR_OK) { goto finally; }
    sdata = ato_str_free(sdata);

    printf("test sign with bad pwd [%s]\n", badpwd);
    if (ato_cr_sign(g_ctx, cr, &sdata, badpwd, data, strlen(data)) == ATO_ERR_OK) { goto finally; }

    ok = TRUE;
finally:
    ato_ks_free(ks);
    sdata = ato_str_free(sdata);
    return ok;
}

struct curl_response {
    char *data;
    size_t len;
};
static void curl_response_init(struct curl_response *r) {
    r->len = 0;
    r->data = calloc(r->len + 1, sizeof(char));
    if (r->data == NULL) {
        fprintf(stderr, "calloc failed\n");
        exit(1);
    }
}
static void curl_response_deinit(struct curl_response *r) {
    if (r == NULL) { return; }
    if (r->data) { free(r->data); r->data = NULL; }
    r->len = 0;
}
static size_t curl_response_data(void *data, size_t size, size_t nmemb, struct curl_response *r) {
    size_t len = r->len + (size * nmemb);
    r->data = realloc(r->data, len + 1);

    if (r->data == NULL) {
        fprintf(stderr, "realloc failed\n");
        exit(1);
    }

    memcpy(r->data + r->len, data, size * nmemb);
    r->data[len] = '\0';
    r->len = len;

    return size * nmemb;
}

static bool _send(const char *payload, struct curl_response *response) {
    bool ok = FALSE;
    CURLcode ret = CURLE_OK;
    struct curl_slist *headers = NULL;
    long resp_code = 0;
    CURL *curl = curl_easy_init();
    headers = curl_slist_append(headers, "Content-Type: application/json");
    curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
    curl_easy_setopt(curl, CURLOPT_URL, "https://softwareauthorisations.acc.ato.gov.au/api/v1/credentials");
    curl_easy_setopt(curl, CURLOPT_POST, 1L);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDS, payload);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, response);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, (void*) curl_response_data);

    if (arg_sslcainfo && strcmp(arg_sslcainfo, "off") == 0) {
        // Test only if problems with SSL
        printf("\n%s\n", "Disabling SSL validation");
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
    } else {
        const char *ca_bundle = getenv("CURL_CA_BUNDLE");
        if (ca_bundle != NULL) {
            ret = curl_easy_setopt(curl, CURLOPT_CAINFO, ca_bundle);
            printf("\n%s %s\n", "Setting CURLOPT_CAINFO to ", ca_bundle);
        }
     }

    //curl_easy_setopt(net->curl, CURLOPT_TIMEOUT, 30);
    //curl_easy_setopt(net->curl, CURLOPT_CONNECTTIMEOUT, 10);
    ret = curl_easy_perform(curl);

    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &resp_code);

    if (ret != CURLE_OK || resp_code != 200) {
        goto finally;
    }

    ok = TRUE;
finally:
    curl_easy_cleanup(curl);
    curl_slist_free_all(headers);
    return ok;
}

static bool _test_renew(void)
{
    bool ok = FALSE;
    ato_ksProperties *prop = NULL;
    ato_Keystore *ks = NULL;
    ato_Credential **crs = NULL;
    ato_Credential *cr = NULL;
    size_t crcount = 0, idx = 0;
    const char *ksdir = NULL, *kspwd = NULL, *ksalias = NULL;
    struct curl_response response;
    const char *ksnew = "keystore_out_renew.xml";
    int errcode = ATO_ERR_OK;
    ato_String *request = NULL;
    //ato_String *p10 = NULL;

    curl_response_init(&response);

    printf("\n");
    printf("================= renew =======================\n");

    ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
    kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
    ksalias = (arg_ksalias != NULL ? arg_ksalias : _DEFAULT_ksalias);

    if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
        goto finally;
    prop = ato_ks_properties(ks);

    ato_ksprop_var_set(prop, "renew.expiry.months", arg_crexpiryperiod);

    //Let's pick a random alias to renew instead
    crcount = ato_ks_credentials(ks, &crs, ATO_FILTER_ALL);
    srand((unsigned)time(NULL));
    idx = rand() % crcount;
    cr = crs[idx];
    ksalias = ato_cr_alias(cr);

    // no need - but get cr again
    if ((cr = ato_ks_credential(ks, ksalias)) == NULL)
        goto finally;

    if (!_pwd_check(ks, cr, kspwd, ATO_ERR_OK)) { goto finally; }

    if (!ato_cr_isrenewable(g_ctx, cr)) {
        printf("\n\nCredential '%s' is not renewable in keystore %s/%s\n\n", ksalias, ksdir, arg_ksfile);
        goto finally;
    }

    if (ato_cr_request_generate(g_ctx, cr, kspwd, "atoakm-99") != ATO_ERR_OK) {
        printf("*** WARNING: Generating renewal for Credential '%s' failed ***\n\n", ato_cr_alias(cr));
        goto finally;
    }
    //p10 = ato_cr_b64p10(cr);
    request = ato_cr_request_data(cr);
    printf("*** Sending renewal request to '%s' ***\n\n", "https://softwareauthorisations.acc.ato.gov.au/api/v1/credentials");

    if (!_send(ato_str_value(request), &response)) {
        //printf("\n\ncurl -v -H 'Content-Type: application/json' -X POST -d '%s' https://softwareauthorisations.acc.ato.gov.au/api/v1/credentials\n\n", request);
        printf("*** ERROR: send failed for %s\n\n", ksalias);
        goto finally;
    }
    //printf("\n\n%s\n\n", response.data);

    errcode = ato_cr_request_process(g_ctx, cr, response.data);
    if (errcode != ATO_ERR_OK) {
        printf("*** ERROR (%d): Failed to process response: %s\n\n", errcode, response.data);
        _dump_error(g_ctx);
        goto finally;
    }

    errcode = saveks(g_ctx, ks, ksdir, ksnew);
    if (errcode != ATO_ERR_OK) {
        printf("*** ERROR (%d): Failed to save to keystore: %s/%s\n\n", errcode, ksdir, ksnew);
        goto finally;
    }

    printf("\n\nCredential '%s' renewed in keystore %s/%s\n\n", ksalias, ksdir, ksnew);

    ks = ato_ks_free(ks); // Note: ksalias invalid after this point
    printf("\n\n");
    errcode = loadks(g_ctx, &ks, ksdir, ksnew);
    if (errcode != ATO_ERR_OK) {
        printf("*** ERROR (%d): Failed to reload keystore: %s/%s\n\n", errcode, ksdir, ksnew);
    }

    ok = TRUE;
finally:
    if (crs != NULL) { crs = ato_free(crs); }
    ato_ks_free(ks);
    curl_response_deinit(&response);
    return ok;
}

static bool _test_changepwd_save(void)
{
        bool ok = FALSE;
        ato_Keystore *ks = NULL;
        const char *ksdir = NULL, *kspwd = NULL, *pwd2=NULL;
        int res = ATO_ERR_OK;
    int errcode = ATO_ERR_OK;

        printf("\n");
        printf("================= changepwd_save =======================\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
        kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
        pwd2 = (arg_chpwd == NULL) ? _pwd2 : arg_chpwd;

        printf("\n");
        printf("%s\n", "_test_changepwd_save: loading keystore.xml");
        if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
                goto finally;

        assert(kspwd != NULL);
    errcode = ato_ks_checkcorrectpwd(g_ctx, ks, kspwd);
    if (errcode == ATO_ERR_OK) {
        printf("\nPassword is ok\n");
    } else {
        printf("\nPassword is NOT ok\n");
        goto finally;
    }

        printf("%s %s to %s\n", "_test_changepwd_save: changing pwd from ", kspwd, pwd2);
    res = ato_ks_changepwd(g_ctx, ks, kspwd, pwd2);
        if (res != ATO_ERR_OK){
                printf("res = %d\n", res);
                ato_errh_handler(g_ctx, res);
                goto finally;
        }

    errcode = ato_ks_checkcorrectpwd(g_ctx, ks, kspwd);
    if (errcode != ATO_ERR_OK) {
        printf("\nSuccess: Old Password fails\n");
    } else {
        printf("\nFAILED: Old Password works\n");
        goto finally;
    }

    errcode = ato_ks_checkcorrectpwd(g_ctx, ks, pwd2);
    if (errcode == ATO_ERR_OK) {
        printf("\nSuccess: New Password works\n");
    } else {
        printf("\nFAILED: New Password fails\n");
        goto finally;
    }

        printf("%s\n", "_test_changepwd_save: saving keystore_out_newpwd.xml");
        if (saveks(g_ctx, ks, ksdir, "keystore_out_newpwd.xml") != ATO_ERR_OK)
                goto finally;

        printf("\n");
        printf("keystore with new pwd saved to file '%s'.", "keystore_out_newpwd.xml");

        ok = TRUE;
finally:
        ato_ks_free(ks);
        return ok;
}

static bool _test_changepwd_reload(void)
{
        bool ok = FALSE;
        ato_Keystore *ks = NULL;
        const char *ksdir = NULL, *kspwd = NULL, *pwd2 = NULL;
        int errcode = ATO_ERR_OK;

        printf("\n");
        printf("================= changepwd_reload =======================\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
        kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
    pwd2 = (arg_chpwd == NULL) ? _pwd2 : arg_chpwd;

        printf("\n");
        printf("%s\n", "_test_changepwd_reload: loading keystore_out_newpwd.xml");
        if (loadks(g_ctx, &ks, ksdir, "keystore_out_newpwd.xml") != ATO_ERR_OK)
                goto finally;

    errcode = ato_ks_checkcorrectpwd(g_ctx, ks, kspwd);
    if (errcode != ATO_ERR_OK) {
                printf("%s\n", "Success: _test_changepwd_reload: old pwd fails");
    } else {
                printf("%s\n", "FAILED: _test_changepwd_reload: old pwd still works");
                goto finally;
    }

    errcode = ato_ks_checkcorrectpwd(g_ctx, ks, pwd2);
    if (errcode == ATO_ERR_OK) {
                printf("%s\n", "Success: _test_changepwd_reload: new pwd works");
    } else {
                printf("%s\n", "FAILED: _test_changepwd_reload: new pwd fails");
                goto finally;
    }

        ok = TRUE;
finally:
        ato_ks_free(ks);
        return ok;
}

static bool _test_changepwd(void)
{
        printf("\n");
        printf("================= changepwd =======================\n");
        return _test_changepwd_save() && _test_changepwd_reload();
}

static bool _test_copycred_reload(void)
{
        bool ok = FALSE;
        ato_Keystore *ks = NULL;
        const char *ksdir = NULL, *kspwd = NULL, *pwd2 = NULL;
        int errcode = ATO_ERR_OK;

        printf("\n");
        printf("================= copycred_reload =======================\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
        kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
        pwd2 = (arg_chpwd == NULL) ? _pwd2 : arg_chpwd;

        printf("\n");
        printf("%s\n", "loading keystore_out_new.xml");
        if (loadks(g_ctx, &ks, ksdir, "keystore_out_new.xml") != ATO_ERR_OK)
                goto finally;

    errcode = ato_ks_checkcorrectpwd(g_ctx, ks, kspwd);
    if (errcode != ATO_ERR_OK) {
                printf("%s\n", "Success: _test_changepwd_reload: old pwd fails");
    } else {
                printf("%s\n", "FAILED: _test_changepwd_reload: old pwd still works");
        goto finally;
    }

    errcode = ato_ks_checkcorrectpwd(g_ctx, ks, pwd2);
    if (errcode == ATO_ERR_OK) {
                printf("%s\n", "Success: _test_changepwd_reload: new pwd works");
    } else {
                printf("%s\n", "FAILED: _test_changepwd_reload: new pwd fails");
        goto finally;
    }

        ok = TRUE;
finally:
        ato_ks_free(ks);
        return ok;
}
static bool _test_copycred_save(void)
{
        bool ok = FALSE;
        int errcode = 0;
        size_t count = 0;
        ato_Keystore *ks = NULL, *toks = NULL;
        ato_Credential **crs = NULL;
        ato_Credential *cr = NULL;
        const char *ksdir = NULL, *kspwd = NULL, *pwd2 = NULL;
        //const char *ksalias = NULL;

        printf("\n");
        printf("================= copycred_save =======================\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
        kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
        //ksalias = (arg_ksalias != NULL ? arg_ksalias : _DEFAULT_ksalias);
        pwd2 = (arg_chpwd == NULL) ? _pwd2 : arg_chpwd;

        if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
                goto finally;

        count = ato_ks_credentials(ks, &crs, ATO_FILTER_DEVICE);
        if (count == 0) {
            printf("No device credentials found\n");
            goto finally;
        }
    cr = crs[0];
        //ksalias = ato_cr_alias(crs[0]);

        if ((ato_ks_create(g_ctx, &toks, NULL, NULL)) != ATO_ERR_OK)
                goto finally;

        printf("\n");
        if (ato_ks_checkcorrectpwd(g_ctx, toks, pwd2) != ATO_ERR_OK)
                goto finally;
        printf("Before copy: toks count = %" PRINTF_SIZET "u\n", ato_ks_count(toks, ATO_FILTER_ALL));

        //if (ato_ks_copycredential(g_ctx, ks, ksalias, kspwd, toks, pwd2, TRUE) != ATO_ERR_OK)
        if (ato_ks_addcredential(g_ctx, toks, pwd2, cr, kspwd, TRUE) != ATO_ERR_OK)
                goto finally;

        if (ato_ks_checkcorrectpwd(g_ctx, toks, pwd2) != ATO_ERR_OK)
                goto finally;
        printf("After copy: toks count = %" PRINTF_SIZET "u\n", ato_ks_count(toks, ATO_FILTER_ALL));
        if (saveks(g_ctx, toks, ksdir, "keystore_out_new.xml") != ATO_ERR_OK)
                goto finally;
        printf("new keystore saved to file '%s' with pwd '%s'.", "keystore_out_new.xml", pwd2);
        printf("\n");

        //errcode = ato_ks_copycredential(g_ctx, ks, ksalias, kspwd, toks, pwd2, FALSE);
        errcode = ato_ks_addcredential(g_ctx, toks, pwd2, cr, kspwd, FALSE);
        if (errcode == ATO_ERR_OK) {
                printf("ERROR: Copy credential a second time didn't fail\n");
                goto finally;
        } else if (errcode != ATO_AKM_ERR_ALIASEXISTS) {
                goto finally;
        }
        //errcode = ato_ks_copycredential(g_ctx, ks, ksalias, kspwd, toks, pwd2, TRUE);
        errcode = ato_ks_addcredential(g_ctx, toks, pwd2, cr, kspwd, TRUE);
        if (errcode != ATO_ERR_OK)
                goto finally;

        printf("\n");
        if (ato_ks_checkcorrectpwd(g_ctx, toks, pwd2) != ATO_ERR_OK)
                goto finally;
        printf("After replace: toks count = %" PRINTF_SIZET "u\n", ato_ks_count(toks, ATO_FILTER_ALL));
        if (saveks(g_ctx, toks, ksdir, "keystore_out_new2.xml") != ATO_ERR_OK)
                goto finally;
        printf("new keystore saved to file '%s' with pwd '%s'.", "keystore_out_new2.xml", pwd2);
        printf("\n");

        ok = TRUE;
finally:
        if (crs != NULL) { crs = ato_free(crs); }
        ato_ks_free(toks);
        ato_ks_free(ks);
        return ok;
}

static bool _test_deleteallcrs(void)
{
        bool ok = FALSE;
        size_t count = 0, delcount = 0;
        ato_Keystore *ks = NULL;
        const char *ksdir = NULL;

        printf("\n");
        printf("================= deleteallcrs =======================\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);

        if (loadks(g_ctx, &ks, ksdir, arg_ksfile) != ATO_ERR_OK)
                goto finally;

        printf("\n");
        count = ato_ks_count(ks, ATO_FILTER_ALL);
        printf("Count before delete all = %" PRINTF_SIZET "u\n", count);

        delcount = ato_ks_delete(ks, NULL);
        if (delcount == 0)
                goto finally;
        printf("Delete count = %" PRINTF_SIZET "u\n", delcount);

        count = ato_ks_count(ks, ATO_FILTER_ALL);
        printf("Count after delete all = %" PRINTF_SIZET "u\n", count);

        if (saveks(g_ctx, ks, ksdir, "keystore_out_empty.xml") != ATO_ERR_OK)
                goto finally;
        printf("empty keystore saved to file '%s'.", "keystore_out_empty.xml");
        printf("\n");

        ok = TRUE;
finally:
        ato_ks_free(ks);
        return ok;
}

static bool _test_copycred(void)
{
        printf("\n");
        printf("================= copycred =======================\n");
        return _test_copycred_save() && _test_copycred_reload();
}

static bool _init_tests(void)
{
        int errcode = ATO_ERR_OK;

        if (arg_organisation == NULL || arg_organisation[0] == '\0') { errcode = -1; goto finally; }
        ato_si_set(arg_organisation, _EXE_DESCRIPTION, _EXE_VERSION, _EXE_TIMESTAMP, _EXE_SRC, _EXE_HASH);
        ato_ctx_create(&g_ctx, NULL);
        errcode = init(g_ctx);

        if (errcode != ATO_ERR_OK) {
                ato_Err *err = ato_ctx_err(g_ctx);
                printf("Error: %d (%s) %s\n", ato_err_code(err), ato_err_codename(err), ato_err_msg(err));
                goto finally;
        }

        ato_log_setlogfn(customlogger); //ato_log_setlogfn(verysimplelogger);
        g_log = ato_ctx_log_set(g_ctx, "", "2", NULL, "w");

        // Set log level for all modules
        ato_setloglevel(NULL, 0, arg_loglevel);
        // Override CR as it can generate too much if there are many credentials
        {
                ato_eLoglevel logcr = ATO_LOG_WARN;
                if (arg_logcr != NULL)
                        logcr = ato_log_name2level(arg_logcr);
                ato_setloglevel(ATO_AKM_LIBRARY, ATO_AKM_MODULEID_CR, logcr);
        }
        // Some modules dump extra files useful for debugging
        ato_setlogfiledump(ATO_BASE_LIBRARY, 0, arg_logfd);
        // Extra log settings for application network traffic
        //if (arg_netlogflag) ato_net_setdebugflag(ato_net_names2flag(arg_netlogflag));

finally:
        return errcode == ATO_ERR_OK;
}

static void _deinit_tests(void)
{
        ato_ctx_free(g_ctx);
      deinit();
}

static bool _run_test(char *test)
{
    bool rc = FALSE;
    bool testrun = FALSE;
    if (test == NULL || *test == '\0') { return TRUE; }

    if (_streqi(test, "dumpks")) { rc = _test_dumpks(); testrun = TRUE; }
    else if (_streqi(test, "sign")) { rc = _test_sign(); testrun = TRUE; }
    else if (_streqi(test, "renew")) { rc = _test_renew(); testrun = TRUE; }
    else if (_streqi(test, "loadsaveks")) { rc = _test_loadsaveks(); testrun = TRUE; }
    else if (_streqi(test, "changepwd")) { rc = _test_changepwd(); testrun = TRUE; }
    else if (_streqi(test, "copycred")) { rc = _test_copycred(); testrun = TRUE; }
    else if (_streqi(test, "deleteallcrs")) { rc = _test_deleteallcrs(); testrun = TRUE; }

    if (testrun) { printf("\n\n"); }
    else { printf("Unknown Test %s\n", test); }
    return rc;
}

static bool _run_tests(void)
{
        bool ok = FALSE;
        bool inited = FALSE;
    char testargbuf[1000], *test = testargbuf;

        if ((inited = _init_tests()) == FALSE)
                goto finally;

        ato_log_start(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", ato_si_product());

        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "BASE Version code=%s", ato_base_version());
        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "AKM Version code=%s", ato_akm_version());

        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "",
                "SoftwareInfo organisation='%s', product='%s', version='%s', timestamp='%s'",
                ato_si_organisation(), ato_si_product(), ato_si_version(), ato_si_timestamp());

    if (arg_akmtests != NULL) { strcpy(testargbuf, arg_akmtests); }
    for (test = strtok(test, ","); test != NULL; test = strtok(NULL, ",")) {
                if (!_run_test(test)) goto finally;
    }
    printf("\nTests OK\n");

        ok = TRUE;
finally:
        if (inited) ato_log_end(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", FALSE, ato_si_product());
        _deinit_tests();
        return ok;
}

static void _forceemptyline(void)
{
        printf("\n"); printf("\n"); printf(" \n");
}

static bool _setargs(int argc, const char *argv[])
{
        bool argres = FALSE;
        _addargs();
        if (argc > 1) {
                argres = ato_arg_process(argc, argv);
                if (argres) {
                        _initargs();
                        _forceemptyline();
                        argres = ato_arg_check();
                        _forceemptyline();
                }
                argres = (ato_arg_value_bool("help") == FALSE);
        }
        return argres;
}

int main(int argc, const char *argv[])
{
        bool ok = FALSE;
        setbuf(stdout, NULL);
        if (_setargs(argc, argv)) {
                ato_arg_dump();
                _forceemptyline();
                ok = _run_tests();
                if (!ok) {
                        _forceemptyline();
                        printf("An error occurred - please check logs if any\n");
                }
        } else {
                ato_arg_usage(_EXE_NAME);
        }
        ato_arg_free();
        _forceemptyline();
        return ok == TRUE ? 0: -1;
}
