#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/all.h"
#include "atoakm/all.h"

#include "misc.h"

static void _writebinary2file(char **filepath, const char *prefix, const char *id, const char *postfix, ato_String *data)
{
    char *offset = NULL;
    FILE *fp = NULL;
    assert(data != NULL);
    *filepath = calloc(100, sizeof(char));
    strcpy(*filepath, prefix);
    if (id != NULL && strlen(id) > 0) {
        strcat(*filepath, "-");
        strcat(*filepath, id);
        if ((offset = strchr(*filepath, ':')) != NULL)
            *offset = '_';
    }
    strcat(*filepath, postfix);
    fp = fopen(*filepath, "wb");
    fwrite(ato_str_value(data), sizeof(char), ato_str_len(data), fp);
    fclose(fp);
}

void dump_alias_if_found(ato_Keystore *ks, const char *alias, bool isexpectfound)
{
    ato_Credential *cr = ato_ks_credential(ks, alias);
    if (cr != NULL)
        printf("%s: Found alias %s (serialnr=%s)\n", isexpectfound ? "Success" : "Failure" , ato_cr_alias(cr), ato_cr_serialnr(cr));
    else
        printf("%s: Failed to find alias %s\n", !isexpectfound ? "Success" : "Failure", alias);
}

static void _dump_cred_cryptostr(const char *name, const char *value, bool fullvalue)
{
    size_t vlen = 0;
    if (value == NULL) value = "";
    vlen = strlen(value);
    if (vlen < 100 || fullvalue) {
        printf("%s=%s\n", name, value);
    } else {
        printf("%s=%.70s...\n", name, value);
    }
}

void dump_cred(ato_Ctx *ctx, ato_Credential *cr, const char *pwd, bool displaycrypto)
{
    const char *id = NULL;
    ato_eCredtype type;
    assert(cr != NULL);

    id = ato_cr_alias(cr);
    type = ato_cr_type(cr);
    printf("%s\n", "");
    if (type == ATO_CREDUSER)
        printf("%s\n", "=== USER ================================");
    else
        printf("%s\n", "=== DEVICE ==============================");
    printf("alias=%s\n", id);
    printf("abn=%s\n", ato_cr_abn(cr));
    printf("legalname=%s\n", ato_cr_legalname(cr));
    printf("serialnr=%s\n", ato_cr_serialnr(cr));
    printf("creationdate=%s\n", ato_cr_creationdate(cr));
    printf("notbefore=%s\n", ato_cr_notbefore(cr));
    printf("notafter=%s\n", ato_cr_notafter(cr));
    printf("sha1fingerprint=%s\n", ato_cr_sha1fingerprint(cr));
    printf("sha1fingerprint_hex=%s\n", ato_cr_sha1fingerprint_hex(ctx, cr));
    if (type == ATO_CREDUSER) {
        printf("personid=%s\n", ato_cr_personid(cr));
        printf("givenames=%s\n", ato_cr_givenames(cr));
        printf("familyname=%s\n", ato_cr_familyname(cr));
        assert(ato_cr_devicename(cr) == NULL);
    } else {
        printf("devicename=%s\n", ato_cr_devicename(cr));
        assert(ato_cr_personid(cr) == NULL);
        assert(ato_cr_givenames(cr) == NULL);
        assert(ato_cr_familyname(cr) == NULL);
    }
    printf("isvalid=%d\n", ato_cr_iscurrent(cr));
    _dump_cred_cryptostr("integrityvalue", ato_str_value(ato_cr_integrityvalue(cr)), displaycrypto);
    _dump_cred_cryptostr("salt", ato_str_value(ato_cr_b64salt(cr)), displaycrypto);
    _dump_cred_cryptostr("p7c", ato_str_value(ato_cr_b64p7(cr)), displaycrypto);
    _dump_cred_cryptostr("p8", ato_str_value(ato_cr_b64p8(cr)), displaycrypto);
    if (displaycrypto) {
        printf("%s\n", "");
        dump_cert(ctx, id, cr);
        printf("%s\n", "");
        dump_privatekey(ctx, id, cr, pwd);
    }
    printf("%s\n", "");
}

void dump_cert(ato_Ctx *ctx, const char *id, ato_Credential *cr)
{
    char *filepath = NULL;
    ato_String *data = NULL;
    int errcode = 0;
    assert(cr != NULL);

    errcode = ato_cr_certificate(ctx, cr, &data);
    if (errcode == 0) {
        _writebinary2file(&filepath, ato_cr_serialnr(cr), id, ".cert", data);
        printf("cert has been written to %s\n", filepath);
        free(filepath);
    } else {
        printf("%s\n", "failed to extract cert");
    }
    ato_str_free(data);
}

void dump_chain(ato_Ctx *ctx, const char *id, ato_Credential *cr)
{
    char *filepath = NULL;
    ato_String *data = NULL;
    int errcode = 0;
    assert(cr != NULL);

    errcode = ato_cr_chain(ctx, cr, &data);
    if (errcode == 0) {
        _writebinary2file(&filepath, ato_cr_serialnr(cr), id, ".certs", data);
        printf("chain has been written to %s\n", filepath);
        free(filepath);
    } else {
        printf("%s\n", "failed to extract chain");
    }
    ato_str_free(data);
}

void dump_privatekey(ato_Ctx *ctx, const char *id, ato_Credential *cr, const char *pwd)
{
    char *filepath = NULL;
    ato_String *data = NULL;
    int errcode = 0;
    assert(cr != NULL);
    assert(pwd != NULL);

    if (ato_cr_isrenewable(ctx, cr)) {
        //if (ato_cr_renew(ctx, cr, pwd) == ATO_ERR_OK)
        //  printf("*** Credential '%s' renewed ***\n\n", id);
        //else
        //  printf("*** WARNING: renewal for Credential '%s' failed ***\n\n", id);
        printf("*** WARNING: Skipping renewal for Credential '%s' (for test purposes) ***\n\n", id);
    }

    errcode = ato_cr_privatekey(ctx, cr, &data, pwd);
    if (errcode == 0) {
        _writebinary2file(&filepath, ato_cr_serialnr(cr), id, ".privatekey", data);
        printf("privatekey has been written to %s\n", filepath);
        free(filepath);
    } else {
        printf("%s\n", "failed to decrypt private key");
    }
    ato_str_free(data);
}

void dump_p12(ato_Ctx *ctx, const char *id, ato_Credential *cr, const char *pwd)
{
    char *filepath = NULL;
    ato_String *data = NULL;
    int errcode = 0;
    assert(cr != NULL);
    assert(pwd != NULL);

    errcode = ato_cr_p12(ctx, cr, &data, pwd);
    if (errcode == 0) {
        _writebinary2file(&filepath, ato_cr_serialnr(cr), id, ".p12", data);
        printf("p12 has been written to %s\n", filepath);
        free(filepath);
    } else {
        printf("%s\n", "failed to generate p12");
    }
    ato_str_free(data);
}
