#ifndef __ATO_CTX_H__
#define __ATO_CTX_H__

#include <stddef.h>

/** @addtogroup atobase_api_public_ctx

//typedef struct _ato_Ctx ato_Ctx;

@{
*/

#ifdef __cplusplus
extern "C" {
#endif

/**
Create a context object for thread local storage - use ato_ctx_free() to free.

At least one object should be created per thread.
@verbatim
    ato_ctx_create(&global_ctx);
@endverbatim

@param ctx the address of the context object to create.
@param id if NULL create an internal guid to distinguish instances.
*/
ATO_EXPORT void ato_ctx_create(ato_Ctx **ctx, const char *id);


/**
Free the context object. This should be done AFTER freeing all other objects bug PRIOR to freeing the
configuration manager.

@param ctx the context object
*/
ATO_EXPORT void ato_ctx_free(ato_Ctx *ctx);

/**
Get the logger associated with the current context.

@param ctx the context.

@return the logger or NULL if none has been initialised.
*/
ATO_EXPORT ato_Log *ato_ctx_log(ato_Ctx *ctx);

/**
Get the error object associated with the current context, if any.

@param ctx the context.

@return the error object or NULL if no error.
*/
ATO_EXPORT ato_Err *ato_ctx_err(ato_Ctx *ctx);

/**
Get the unique id associated with this instance.

@param ctx the context.

@return the context "id".
*/
ATO_EXPORT const char *ato_ctx_id(ato_Ctx *ctx);

/**
Set logging properties to control the behaviour of logging.

@param ctx the context.
@param id a logging id to use - if NULL defaults to the context id
@param logindent amount to indent nested calls - NULL or empty - use ato_arg_value("logindent") else use "2"
@param logpath null or empty - use ato_arg_value("logpath") else use "messages.log"; "none" - don't log; "stdout"; "stderr"; else use as logfile (ato_log_outputtype())
@param mode passed to fopen

@return the logging object associated with the context.
*/
ATO_EXPORT ato_Log *ato_ctx_log_set(ato_Ctx *ctx, const char *id, const char *logindent, const char *logpath, const char *mode);

/**
Get the value object associated with the given key in the given context.

@param ctx the context.
@param key the key name. 

@return the associate value or NULL.
*/
ATO_EXPORT void *ato_ctx_cache(ato_Ctx *ctx, const char *key);
/**
Set the value object associated with the given key in the given context.

@param ctx the context.
@param key the key name - an internal copy is made.
@param value - this is referenced - do not free 

@return the associate value or NULL.
*/
ATO_EXPORT void *ato_ctx_cache_set(ato_Ctx *ctx, const char *key, void *value);
/**
Get the value object associated with the given key in the given context.

@param ctx the context.
@param key the key name. 

@return the associate value or NULL.
*/
ATO_EXPORT const void *ato_ctx_cachec(ato_Ctx *ctx, const char *key);
/**
Set the value object associated with the given key in the given context.

@param ctx the context.
@param key the key name - an internal copy is made.
@param value - this is referenced - do not free 

@return the associate value or NULL.
*/
ATO_EXPORT const void *ato_ctx_cachec_set(ato_Ctx *ctx, const char *key, const void *value);

#ifdef __cplusplus
}
#endif

/*! @} */

#endif /*__ATO_CTX_H__*/
