#ifndef __ATO_ERR_H__
#define __ATO_ERR_H__

/** @addtogroup atobase_api_public_errgen

Error objects are created and attached to the current Context object, if an error occurs.
Errors can be nested - see ato_err_inner().

For a general description see @ref atobase_error_overview.

@{
*/

/**
The Error type.
*/
typedef struct _ato_Err ato_Err;

/**
The severity of the error
*/
typedef enum {
    ATO_ESEVERITY_FATAL, /**< the application may halt immediately, if not it should exit ASAP */
    ATO_ESEVERITY_ERR, /**< the application should go into recovery mode (e.g. restoring connections) */
    ATO_ESEVERITY_WARN /**< business or user level error such as invalid password, so try again */
} ato_eErrSeverity;

#ifdef __cplusplus
extern "C" {
#endif

/**
Convenience method - FALSE if ATO_ERR_OK.
@param err error object
@return TRUE or FALSE
*/
ATO_EXPORT bool ato_err_iserror(ato_Err *err);

/**
Get the error code.
@param err error object
@return the error code
*/
ATO_EXPORT int ato_err_code(ato_Err *err);

/**
Get the error code name as as string.
@param err error object
@return the error code name e.g. "ATO_ERR_OK" is the name for code 0.
*/
ATO_EXPORT const char *ato_err_codename(ato_Err *err);

/**
Get the stock message defined for the given error code.
@param err error object
@return the error code message
*/
ATO_EXPORT const char *ato_err_codemsg(ato_Err *err);

/**
Get the severity.
@param err error object
@return the severity
*/
ATO_EXPORT ato_eErrSeverity ato_err_severity(ato_Err *err);

/**
Get the library (string) where the error was created.
@param err error object
@return library name
*/
ATO_EXPORT const char *ato_err_library(ato_Err *err);

/**
Get the module (file) where the error was created.
@param err error object
@return module name
*/
ATO_EXPORT const char *ato_err_module(ato_Err *err);

/**
Get the function where the error was created.
@param err error object
@return function name
*/
ATO_EXPORT const char *ato_err_function(ato_Err *err);

/**
Get the line where the error was created.
@param err error object
@return line number
*/
ATO_EXPORT int ato_err_line(ato_Err *err);

/**
Get the message attached to the error.
@param err error object
@return message
*/
ATO_EXPORT const char *ato_err_msg(ato_Err *err);

/**
Get the nested/inner error object if any.
@param err error object
@return inner error
*/
ATO_EXPORT ato_Err *ato_err_inner(ato_Err *err);

/**
Get the custom object if any.
@param err error object
@return custom object
*/
ATO_EXPORT void *ato_err_custom(ato_Err *err);

#ifdef __cplusplus
}
#endif


/*! @} */

#endif /*__ATO_ERR_H__*/
