#ifndef __ATO_LOG_H__
#define __ATO_LOG_H__

/* @addtogroup atobase_api_public_log

@{
*/

/** @defgroup atobase_api_public_log_defines Defines
@ingroup atobase_api_public_log

*/

/** @defgroup atobase_api_public_log_defines_name Name Defines
@ingroup atobase_api_public_log_defines

Names of loglevels as strings. The first character is unique and can be used to distinguish between them.
This is useful, for example, if passing in on the command line.

@{
*/
#define ATO_LOG_OFF_NAME "off" /**< "off" Log level as a string */
#define ATO_LOG_FATAL_NAME "fatal" /**< "fatal" Log level as a string */
#define ATO_LOG_ERR_NAME "err" /**< "err" Log level as a string */
#define ATO_LOG_WARN_NAME "warn" /**< "warn" Log level as a string */
#define ATO_LOG_INFO_NAME "info" /**< "info" Log level as a string */
#define ATO_LOG_DEBUG_NAME "debug" /**< "debug" Log level as a string */
#define ATO_LOG_TRACE_NAME "trace" /**< "trace" Log level as a string */
#define ATO_LOG_ALL_NAME "all" /**< "all" Log level as a string */

/*! @} */

/** @defgroup atobase_api_public_log_defines_output Output Type
@ingroup atobase_api_public_log_defines

Character identifying where logging is sent to.

@{
*/

#define ATO_LOG_OUTPUTTYPE_NONE 'n' /**< Configuration entry 'n'one */
#define ATO_LOG_OUTPUTTYPE_STDOUT 'o' /**< Configuration entry 'o'ut (stdout) */
#define ATO_LOG_OUTPUTTYPE_STDERR 'e' /**< Configuration entry 'e'err (stderr) */
#define ATO_LOG_OUTPUTTYPE_PATH 'p' /**< Configuration entry 'p'ath */

/*! @} */

/** @defgroup atobase_api_public_log_types Types
@ingroup atobase_api_public_log

@{
*/

/**
The log object.
*/
typedef struct _ato_Log ato_Log;

/**
Indicates whether logging to stdout, stderr, a file, or nothing.
*/
typedef enum {
    ATO_LOGTO_NULL = 0, /**< Don't log */
    ATO_LOGTO_STDOUT, /**< Log to stdout */
    ATO_LOGTO_STDERR, /**< Log to stderr */
    ATO_LOGTO_FILE /**< Log to a file */
} ato_eLogMode;

/**
Controls the state/type of logging - e.g. header, trailer, or content.
This controls level of indentation if used.
*/
typedef enum {
    ATO_LOGSTATE_MSG = 0, /**< Just log - usually called within a method */
    ATO_LOGSTATE_OPEN, /**< Log a "header" entry and setup to indent - usually called at method start */
    ATO_LOGSTATE_CLOSE /**< Log a "trailer" entry  and reduce indent - usually called at method end */
} ato_eLogState;

/**
The signature for the logging function - see ato_log_setlogfn()
*/
typedef bool (ato_logfunction)(ato_Log *log, ato_eLoglevel loglevel, ato_eLogState logstate, const char *library, const char *module, const char *function, const char *format, va_list args);

/*! @} */

#ifdef __cplusplus
extern "C" {
#endif

/** @defgroup atobase_api_public_log_methods Methods
@ingroup atobase_api_public_log

@{
*/

/**
Override the default logging function.
@param logfunction the custom application log function to use
*/
ATO_EXPORT void ato_log_setlogfn(ato_logfunction logfunction);

/**
Convert the loglevel name to the enum equivalent.
@param c the log level as a string - only first character is significant
@return the loglevel enum
*/
ATO_EXPORT ato_eLoglevel ato_log_name2level(const char *c);

/**
Convert the loglevel enum to the string equivalent.
@param level the log level enum
@return the loglevel string name
*/
ATO_EXPORT const char *ato_log_level2name(ato_eLoglevel level);

/**
The unique id of the logger - typically a guid.
Used to uniquify the logfile name per thread.

@param log the log object
@return logger id
*/
ATO_EXPORT const char *ato_log_id(ato_Log *log);

/**
The output type - 'n'one, 'e'rr (stderr), 'out' (stdout), 'p'ath (dir/file)
@param log the log object
@return output type
*/
ATO_EXPORT char ato_log_outputtype(ato_Log *log);

/**
The full path of the file being logged to if ato_log_outputtype() is path.
@param log the log object
@return path
*/
ATO_EXPORT const char *ato_log_path(ato_Log *log);

/**
The full 'unique' path, incorporating ato_log_id(), of the file being logged to if ato_log_outputtype() is path.
This will prefix the id before the file extension, so that logfile.log becomes logfile-id.log.
@param log the log object
@return path incorporating the id
*/
ATO_EXPORT const char *ato_log_upath(ato_Log *log);

/**
Amount to indent each level - defaults to 2
@param log the log object
@return Indent size
*/
ATO_EXPORT size_t ato_log_indentsize(ato_Log *log);

/**
Indent level as determined by nested ato_log_start()/ato_log_end() calls.
@param log the log object
@return Indent level
*/
ATO_EXPORT size_t ato_log_indentlevel(ato_Log *log);

/**
Target loglevel of any log calls within a library/module.
@param library name of library
@param module name of module within library
@return Log level
*/
ATO_EXPORT ato_eLoglevel ato_loglevel(const char *library, const char *module);

/**
Set the loglevel for individual registered libraries/modules.
@code
e.g.
// Set all registered libraries and modules
ato_setloglevel(NULL, 0, ATO_LOG_WARN);
// Turn off logging in crypto library
ato_setloglevel(ATO_BASE_LIBRARY, ATO_BASE_MODULEID_CRYPTO, ATO_LOG_OFF);
@endcode

@param library the library name - NULL indicates all.
Current known libraries using this SDK are BASE, AKM, STM, CSR. See the relevant \#defines e.g. ATO_BASE_LIBRARY
@param moduleidmask the modules each library registers - 0 indicates all. These allow bitwise or-ing
@param level the loglevel to use
*/
ATO_EXPORT void ato_setloglevel(const char *library, unsigned long moduleidmask, ato_eLoglevel level);

/**
For modules that dump content to files (to aid debugging), check if file dumping is on or off.
@param library name of library
@param module name of module within library
@return TRUE if file dumping is on.
*/
ATO_EXPORT bool ato_logfiledump(const char *library, const char *module);

/**
For modules that dump content to files (to aid debugging), turn file dumping on or off.
This has no affect on modules that do not dump files.
@param library name of library
@param moduleidmask the modules each library registers - 0 indicates all. These allow bitwise or-ing
@param filedump TRUE to turn dumping on.
*/
ATO_EXPORT void ato_setlogfiledump(const char *library, unsigned long moduleidmask, bool filedump);

/**
Log message content (ATO_LOGSTATE_MSG) if current >= target.
@param log the log object
@param target the level the module is set at
@param current the level of the call
@param library call came from
@param module call came from
@param function call came from
@param format printf() format characters
@param ... parameters
@return TRUE if logging occurred.
*/
ATO_EXPORT bool ato_log_msg(ato_Log *log, ato_eLoglevel target, ato_eLoglevel current, const char *library, const char *module, const char *function, const char *format, ...);

/**
Log message header (ATO_LOGSTATE_OPEN) if current >= target.
@param log the log object
@param target the level the module is set at
@param current the level of the call
@param library call came from
@param module call came from
@param function call came from
@param format printf() format characters
@param ... parameters
@return TRUE if logging occurred.
*/
ATO_EXPORT bool ato_log_start(ato_Log *log, ato_eLoglevel target, ato_eLoglevel current, const char *library, const char *module, const char *function, const char *format, ...);

/**
Log message trailer (ATO_LOGSTATE_CLOSE) if current >= target.
@param log the log object
@param target the level the module is set at
@param current the level of the call
@param library call came from
@param module call came from
@param function call came from
@param doindentonly don't actually log anything - just adjust the indent level
@param format printf() format characters
@param ... parameters
@return TRUE if logging occurred.
*/
ATO_EXPORT bool ato_log_end(ato_Log *log, ato_eLoglevel target, ato_eLoglevel current, const char *library, const char *module, const char *function, bool doindentonly, const char *format, ...);

/*! @} */

#ifdef __cplusplus
}
#endif

#endif /*__ATO_LOG_H__*/

