#ifndef __ATO_CTX_PRIVATE_H__
#define __ATO_CTX_PRIVATE_H__

#include "atobase/ctx.h"

/* ! @defgroup atobase_api_pctx PContext
@ingroup atobase_api_private

@{
*/

#ifdef __cplusplus
extern "C" {
#endif

ATO_EXPORT struct exception_context *ato__ctx_ec(ato_Ctx *ctx);

/**@internal
Clear any errors associated with the current context.
This is done at the start of any method which can generate errors.
@param ctx the context or NULL.
@return ATO_ERR_OK if successful.
*/
ATO_EXPORT int ato_ctx_err_clear(ato_Ctx *ctx);

/**@internal
Create an error object associated with the current context.
This will wrap any existing error object as an "inner" error.
@param ctx the context or NULL.
@param library the source library generating the error
@param module the library module
@param function the function
@param line __LINE__
@param code the error code, not being ATO_ERR_OK
@param severity the severity of the error - most are ATO_ESEVERITY_WARN
@param msg specific message for the error.
@param customobj a custom object to attach to the error. This is interpreted by the source creating the error.
@return ATO_ERR_OK if successful.
*/
ATO_EXPORT int ato_ctx_err_new(ato_Ctx *ctx, const char *library, const char *module, const char *function, int line, int code, ato_eErrSeverity severity, void *customobj, const char *msg);

ATO_EXPORT int ato_ctx_err_vnew(ato_Ctx *ctx, const char *library, const char *module, const char *function, int line, int code, ato_eErrSeverity severity, void *customobj, size_t size, const char *format, ...);

#ifdef __cplusplus
}
#endif

/**
Creates an error object, using the given severity, in the context and wraps any existing error.
*/
#define ATO_CTX_NEWERR_SEV(ctx, code, severity, msg) \
    ato_ctx_err_new(ctx, _library, _module, function, __LINE__, (code), (severity), NULL, (msg))

/**
Creates an error object in the context and wraps any existing error. Assigns severity level ATO_ESEVERITY_WARN.
*/
#define ATO_CTX_NEWERR(ctx, code, msg) \
    ato_ctx_err_new(ctx, _library, _module, function, __LINE__, (code), ATO_ESEVERITY_WARN, NULL, (msg))

/**
Creates an error object in the context and wraps any existing error. Assigns severity level ATO_ESEVERITY_WARN.
*/
#define ATO_CTX_VNEWERR(ctx, code, argsize, format, ...) \
    ato_ctx_err_vnew(ctx, _library, _module, function, __LINE__, (code), ATO_ESEVERITY_WARN, NULL, argsize, format, __VA_ARGS__)

#define ATO_CTX_FN_START(ctx) { \
    ATO_LOG_TRACESTART(ato_ctx_log((ctx))); \
    ato_ctx_err_clear((ctx)); \
    }

#define ATO_CTX_FN_END(ctx, errcode) { \
    ato_errh_handler((ctx), (errcode)); \
    ATO_LOG_TRACEEND(ato_ctx_log((ctx)), (errcode)); \
    }

/* ! @} */

#endif /*__ATO_CTX_PRIVATE_H__*/

