#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atoutilcore.h"

#include "atoarg.h"

typedef enum {
    T_ATOT_ARGSECTION_FOUND,
    T_ATOT_ARGSECTION_NOSECTIONS,
    T_ATOT_ARGSECTION_NOTFOUND,
    T_ATOT_ARGSECTION_BADSECTION,
    T_ATOT_ARGSECTION_MISSING
} t_atot_eArgSectionRetCode;

typedef enum {
    T_ATOT_ARG_BAD,
    T_ATOT_ARG_STD,
    T_ATOT_ARG_SECTION,
    T_ATOT_ARG_COMMENT
} t_atot_eArgType;

#define ATOT_ARG_MAX 1000
typedef struct _ato_Arg ato_Arg;
struct _ato_Arg {
    bool required;
    char *name;
    char *defvalue;
    char *allowedvalues;
    char *help;
    char *value;
};
static ato_Arg *_argv[ATOT_ARG_MAX + 1];
static size_t _argc = 0;
static bool _ignoreArgFile = 0;

///////////////////////////////////////////////////////////////////////////
static char *_trim(char *str, size_t len) {
    if (str == NULL) { return NULL; }
    if (len == 0) { len = strlen(str); }
    while (len > 0 && isspace((int) str[len - 1])) {
        str[len - 1] = '\0';
        len -= 1;
    }
    return str;
}

static char *_arg_strdup(const char *src) {
    char *dst = NULL;
    if (src == NULL)
        return NULL;
    dst = calloc(strlen(src) + 1, sizeof(char));
    assert(dst != NULL);
    return strcpy(dst, src);
}
static char *_arg_strndup(const char *src, size_t len) {
    return ato_strndup(src, 1, len);
}

///////////////////////////////////////////////////////////////////////////
static size_t _vartok_count(const char *src, const char *delim) {
    size_t count = 0;
    if (src == NULL) { return 0; }
    while ((src = strstr(src, delim)) != NULL) {
        count++; src++;
    }
    return count ? count * 2 + 1 : 0;
}
static void *_ato_vartokfree(char *dst[]) {
    size_t i = 0;
    if (dst == NULL)
        return NULL;
    for (i = 0; dst[i] != NULL; i++)
        free(dst[i]);
    free(dst);
    return NULL;
}
static char *_ato_vartoksub(char *var) {
    const char *val = getenv(var);
    if (val == NULL) { val = ato_arg_value(var); }
    if (val == NULL) val = "";
    return _arg_strdup(val);
}
// If new str allocated return, else return NULL (no change from src)
static char * _ato_vartok(int *recurse, const char *src) {
    char varname[101];
    size_t count = _vartok_count(src, "${");
    int i = 0;
    char *dst = NULL;
    char **da = NULL;
    size_t totallen = 0;
    const char *s = src;

    if (src == NULL || count == 0 || (*recurse)++ > 9) { return dst; }

    da = calloc(count + 1, sizeof(char *));

    while (i < (int) count) {
        int len = 0;
        const char *e = strstr(s, "${");
        if (e == NULL) {
            len = strlen(s);
            if (len > 0) {
                da[i] = _arg_strndup(s, len);
                totallen += len+1;
                i++;
            }
            break;
        }
        len = e - s;
        if (len > 0) {
            da[i] = _arg_strndup(s, len);
            totallen += len+1;
            i++;
        }

        s = e + 2;
        e = strchr(s, '}');
        if (e == NULL) { i = -1; break; }
        len = e - s;
        if (len == 0) { i = -1; break; }
        if (len > 100) { i = -1; break; }
        strncpy(varname, s, len)[len] = '\0';
        {
            char *value = _ato_vartoksub(varname);
            char *nvalue = _ato_vartok(recurse, value);
            da[i] = ato_striff(nvalue, value);
            if (nvalue != NULL) { ato_free(value); }
        }
        totallen += strlen(da[i])+1;
        i++;

        s = e + 1;
    }

    if (i != -1) {
        dst = calloc(totallen+1, sizeof(char));
        for (i = 0; i < (int) count && da[i] != NULL; i++) {
            strcat(dst, da[i]);
        }
    }

    da = _ato_vartokfree(da);
    return dst;
}


///////////////////////////////////////////////////////////////////////////

static char *_arg_free(char *str) {
    if (str)
        free(str);
    return NULL;
}

static const char *_arg_strorblank(const char *str) {
    if (str)
        return str;
    return "";
}

static ato_Arg *_findArg(const char *name) {
    size_t i = 0;
    assert(name != NULL && *name != '\0');
    for (i = 0; i < _argc; i++) {
        ato_Arg *arg = _argv[i];
        if (strcmp(arg->name, name) == 0)
            return arg;
    }
    return NULL;
}

static void s_arg_dump(ato_Arg *arg, bool full) {
    if (arg == NULL) { return; }
    if (full && arg->help) {
        printf("-%s%s = (%s) [(%s)(%s) %s%s]\n", arg->name, arg->required ? "*" : "", arg->value?arg->value:"(null)", _arg_strorblank(arg->defvalue), _arg_strorblank(arg->allowedvalues), _arg_strorblank(arg->help), "");
    } else if (!ato_strstartswith(arg->name, "a_")) {
        printf("%s=%s\n", arg->name, arg->value?arg->value:"(null)");
    }
}

void ato_arg_dump(void) {
    size_t i = 0;
    for (i = 0; i < _argc; i++) {
        s_arg_dump(_argv[i], FALSE);
    }
}

void ato_arg_usage(const char *appname) {
    printf("\n");
    printf("Usage: %s -?|-h|-help\n", appname);
    printf("Usage: %s @argumentfile", appname);
    printf("\n");
    printf("%s\n", "@argumentfile can be mixed in with '-' args - last occurence has precedence.");
    printf("\n");
    printf("Description of argument list below:\n");
    printf("  -argument* = (value) [(defaultvalue)(allowedvalues) helptext]\n");
    printf("  where defaultvalue, allowedvalues may not be present if not applicable\n");
    printf("  and arguments marked with * are required\n");
    printf("\n");

    {
        size_t i = 0;
        for (i = 0; i < _argc; i++) {
            s_arg_dump(_argv[i], TRUE);
        }
    }
}

bool ato_arg_check(void) {
    bool argres = TRUE;
    size_t i = 0;
    for (i = 0; i < _argc; i++) {
        ato_Arg *arg = _argv[i];
        if (arg->required && (arg->value == NULL || (arg->value)[0] == '\0')) {
            argres = FALSE;
            printf("Please supply a value for %s\n", arg->name);
        }
    }
    return argres;
}

static char *s_cutcomment(char *str) {
    if (str == NULL) { return NULL; }
    char *commentch = strchr(str, '#');
    if (commentch && commentch[1] != '#') { *commentch = '\0'; }
    return _trim(str, 0);
}

static const char *s_arg_reset(ato_Arg *arg, const char *value) {
    if (!ato_streq(arg->value, value)) {
        _arg_free(arg->value);
        arg->value = s_cutcomment(ato_strdup(value, 0));
    }
    return arg->value?arg->value:arg->defvalue;
}

static ato_Arg *s_arg_set(ato_Arg *arg, bool required, const char *defvalue, const char *allowedvalues, const char *help) {
    arg->defvalue = _arg_free(arg->defvalue);
    arg->allowedvalues = _arg_free(arg->allowedvalues);
    arg->help = _arg_free(arg->help);
    arg->required = required;
    arg->defvalue = _arg_strdup(defvalue);
    arg->allowedvalues = _arg_strdup(allowedvalues);
    arg->help = _arg_strdup(help);
    return arg;
}

static ato_Arg *s_arg_new(const char *name, bool required, const char *defvalue, const char *allowedvalues, const char *help) {
    ato_Arg *arg = calloc(1, sizeof(ato_Arg));
    _argv[_argc++] = arg;
    arg->name = _arg_strdup(name);
    return s_arg_set(arg, required, defvalue, allowedvalues, help);
}

void ato_arg_add(const char *name, bool required, const char *defvalue, const char *allowedvalues, const char *help) {
    ato_Arg *arg = _findArg(name);
    if (arg == NULL) {
        s_arg_new(name, required, defvalue, allowedvalues, help);
    }
}

const char *ato_arg_value_set(const char *name, const char *value) {
    ato_Arg *arg = _findArg(name);
    if (arg == NULL) { return NULL; }
    return s_arg_reset(arg, value);
}

const char *ato_arg_value(const char *name) {
    const char *value = NULL;
    ato_Arg *arg = _findArg(name);
    if (arg) {
        value = arg->value;
        if (!value)
            value = arg->defvalue;
    }
    return value;
}

#define UTL_ISTRUE(value) (value && strchr("yYtT1", (int)(value[0])) != NULL)
bool ato_arg_value_bool(const char *name) {
    const char *value = ato_arg_value(name);
    return UTL_ISTRUE(value);
}

int ato_arg_value_int(const char *name) {
    const char *value = ato_arg_value(name);
    return value ? (int) strtol(value, NULL, 10) : 0;
}

static void s_process_tokens(void) {
    size_t i = 0;
    for (i = 0; i < _argc; i++) {
        int recurse = 0;
        ato_Arg *arg = _argv[i];
        //char *value = _ato_vartok(&recurse, NULL);
        char *value = _ato_vartok(&recurse, arg->value);
        if (value != NULL) {
            _arg_free(arg->value);
            arg->value = s_cutcomment(value);
        }
        //s_arg_dump(arg);
    }

}

static t_atot_eArgType _setArgValue(const char *name, const char *value) {
    t_atot_eArgType argtype = T_ATOT_ARG_STD;
    ato_Arg *arg = _findArg(name);
    if (arg == NULL) {
        //printf("Unknown argument supplied (ignored): %s\n", name);
        arg = s_arg_new(name, FALSE, NULL, NULL, NULL);
    }
    s_arg_reset(arg, value);
    return argtype;
}

static t_atot_eArgType _processArgFile(const char *fname);

static t_atot_eArgType _processArg(const char *const_arg) {
    t_atot_eArgType argtype = T_ATOT_ARG_STD;
    char *argbase = NULL, *arg = NULL;
    arg = argbase = _trim(_arg_strdup(const_arg), 0);
    if (arg == NULL || *arg == '#' || *arg == '\0') {
        argtype = T_ATOT_ARG_COMMENT;
    } else if (*arg == '[') {
        argtype = T_ATOT_ARG_SECTION;
    } else if (*arg == '@') {
        if (!_ignoreArgFile) {
            argtype = _processArgFile(arg + 1);
        }
    } else if (*arg == '-') {
        ++arg;
        if (*arg != '\0') {
            char *argname = strtok(arg, "=");
            argtype = _setArgValue(argname, strtok(NULL, "="));
        } else {
            printf("Empty argument supplied\n");
            argtype = T_ATOT_ARG_BAD;
        }
    } else {
        printf("Invalid argument supplied: %s\n", arg);
        argtype = T_ATOT_ARG_BAD;
    }
    if (argbase != NULL)
        free(argbase);
    return argtype;
}

static FILE *_findandopen(const char *fname, const char *mode) {
    char _fnameprefix[FILENAME_MAX+1];
    char _fname[FILENAME_MAX+1];
    FILE *fp = NULL;
    int count = 0;
    strcpy(_fnameprefix, "");
    strcpy(_fname, fname);
    while ((fp = fopen(_fname, mode)) == NULL && count++ < 10) {
        strcat(_fnameprefix, "../");
        strcpy(_fname, _fnameprefix);
        strcat(_fname, fname);
    }
    return fp;
}

static char _line[1001];
// If there are no sections at all, then section will always match the start of the file
static t_atot_eArgSectionRetCode _seekArgFileSection(FILE * fp, const char *fname, const char *section) {
    bool anysectionfound = 0;
    char *arg = NULL;
    bool report_missing_ok = (*section == '-');
    //printf("SECTION=%s (%s)\n", section, fname);
    if (*section == '-' || *section == '+') { ++section; }
    fseek(fp, 0L, SEEK_SET);
    while ((arg = fgets(_line, sizeof(_line) - 1, fp)) != NULL) {
        if (*arg == '[') {
            char *arg_end = strchr(arg, ']');
            anysectionfound = 1;
            if (arg_end == NULL) {
                printf("Missing closing ']' at '%s'\n", _line);
                assert(arg_end != NULL);
                return T_ATOT_ARGSECTION_BADSECTION;
            }
            *arg_end = '\0';
            if (strcmp(section, arg + 1) == 0) {
                return T_ATOT_ARGSECTION_FOUND;
            }
        }
    }
    if (!report_missing_ok) {
        fseek(fp, 0L, SEEK_SET);
        printf("Failed to find section '%s' in file '%s'\n", section, fname);
        return T_ATOT_ARGSECTION_MISSING;
    }
    else if (!anysectionfound) {
        fseek(fp, 0L, SEEK_SET);
        return T_ATOT_ARGSECTION_NOSECTIONS;
    }
    return T_ATOT_ARGSECTION_NOTFOUND;
}

static t_atot_eArgType _processArgFileSection(FILE * fp, bool stopOnSection) {
    t_atot_eArgType argtype = T_ATOT_ARG_STD;
    char *arg = NULL;
    while ((arg = fgets(_line, sizeof(_line) - 1, fp)) != NULL) {
        argtype = _processArg(arg);
        if (argtype == T_ATOT_ARG_BAD)
            break;
        else if (stopOnSection && argtype == T_ATOT_ARG_SECTION)
            break;
    }
    return argtype;
}

#define SECTION_PRELOAD "-preload"
#define SECTION_POSTLOAD "-postload"

static bool _skip_section(const char *section) {
    return section == NULL || *section == '\0' || (strcmp(section, SECTION_PRELOAD) == 0) || (strcmp(section, SECTION_POSTLOAD) == 0);
}

static t_atot_eArgType _processArgFileSections(FILE * fp, const char *fname, const char *sections) {
    t_atot_eArgType argtype = T_ATOT_ARG_STD;
    t_atot_eArgSectionRetCode secrc = T_ATOT_ARGSECTION_NOTFOUND;

    if (sections == NULL || *sections == '\0' || (strcmp(sections, ",") == 0)) {
        sections = NULL;
    }

    secrc = _seekArgFileSection(fp, fname, SECTION_PRELOAD);
    if (secrc == T_ATOT_ARGSECTION_BADSECTION || secrc == T_ATOT_ARGSECTION_MISSING) {
        return T_ATOT_ARG_BAD;
    }

    if (secrc == T_ATOT_ARGSECTION_FOUND || secrc == T_ATOT_ARGSECTION_NOSECTIONS) {
        argtype = _processArgFileSection(fp, 1);
        if (argtype == T_ATOT_ARG_BAD)
            return argtype;
    }

    if (secrc == T_ATOT_ARGSECTION_NOSECTIONS) { // No sections in file, so we're done
        return argtype;
    }

    if (sections != NULL) {
        char *tokstr = ato_strdup(sections, 0), *tokctx = NULL, *tok = NULL, *tokstart = tokstr;
        for (; (tok = ato_strtok(tokstart, ",", &tokctx)) != NULL; tokstart = NULL) {
            if (_skip_section(tok)) { continue; }
            secrc = _seekArgFileSection(fp, fname, tok);
            if (secrc == T_ATOT_ARGSECTION_FOUND) {
                argtype = _processArgFileSection(fp, 1);
            }
            if (argtype == T_ATOT_ARG_BAD || secrc == T_ATOT_ARGSECTION_BADSECTION || secrc == T_ATOT_ARGSECTION_MISSING) {
                break;
            }
        }
        free(tokstr);
        if (secrc == T_ATOT_ARGSECTION_BADSECTION || secrc == T_ATOT_ARGSECTION_MISSING) {
            return T_ATOT_ARG_BAD;
        }
    }

    secrc = _seekArgFileSection(fp, fname, SECTION_POSTLOAD);
    if (secrc == T_ATOT_ARGSECTION_BADSECTION || secrc == T_ATOT_ARGSECTION_MISSING) {
        return T_ATOT_ARG_BAD;
    }

    if (secrc == T_ATOT_ARGSECTION_FOUND) {
        argtype = _processArgFileSection(fp, 1);
    }

    return argtype;
}

static t_atot_eArgType _processArgFile(const char *filename) {
    t_atot_eArgType argtype = T_ATOT_ARG_STD;
    FILE *fp = NULL;
    const char *fname = filename;
    char *tmpfname = NULL;
    char *sections = NULL;

    sections = strchr(filename, ':');
    if (sections != NULL) {
        fname = tmpfname = _arg_strdup(filename);
        sections = strchr(tmpfname, ':');
        *sections++ = '\0';
    }

    fp = _findandopen(fname, "r");
    if (fp == NULL) {
        printf("Failed to open argument file '%s'\n", fname);
        argtype = T_ATOT_ARG_BAD;
    } else {
        argtype = _processArgFileSections(fp, fname, sections);
    }

    if (tmpfname != NULL)
        free(tmpfname);
    if (fp)
        fclose(fp);
    return argtype;
}

void ato_arg_option_ignoreFile(bool ignore) {
    _ignoreArgFile = ignore;
}

bool ato_arg_process(int argc, const char *argv[]) {
    t_atot_eArgType argtype = T_ATOT_ARG_STD;
    int i = 0;
    ato_arg_add("help", FALSE, "0", "yYtT1 ", "Usage option");
    for (i = 1; i < argc; i++) {
        const char *arg = NULL;
        arg = argv[i];
        argtype = _processArg(arg);
        if (argtype == T_ATOT_ARG_BAD) { return FALSE; }
    }
    s_process_tokens();
    return TRUE;
}

void ato_arg_free(void) {
    size_t i = 0;
    for (i = 0; i < _argc; i++) {
        ato_Arg *arg = _argv[i];
        arg->name = _arg_free(arg->name);
        arg->defvalue = _arg_free(arg->defvalue);
        arg->allowedvalues = _arg_free(arg->allowedvalues);
        arg->help = _arg_free(arg->help);
        arg->value = _arg_free(arg->value);
        free(arg);
    }
}

//int ato_arg_init(void) { return 0; }
