#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>
#if !defined(WIN32)
#include <stdint.h>
#else
#include <Windows.h>
#include <WinCrypt.h> /* X509_NAME defined here. It is explicitly undefined in openssl/x509.h for WIN32. */
#endif

#define OPENSSL_THREAD_DEFINES
#include "openssl/opensslconf.h"
//#if !defined(__FreeBSD__)
//  #if defined(OPENSSL_THREADS)
//      #pragma message("OpenSSL thread support enabled.")
//  #else
//      #pragma message("OpenSSL thread support disabled.")
//  #endif
//#endif

/*#define OPENSSL_NO_SHA512*/
#include "openssl/err.h"
#include "openssl/x509.h"
#include "openssl/x509v3.h"
#include "openssl/conf.h"
#include "openssl/bio.h"
#include "openssl/ssl.h"
#include "openssl/err.h"
#include "openssl/pem.h"
#include "openssl/evp.h"
#include "openssl/pkcs12.h"
#include "openssl/rand.h"
#include "openssl/hmac.h"
#include "openssl/cms.h"
#include "openssl/crypto.h"

#include "atointernal.h"
#include "atotypes.h"
#include "atoexcept.h"
#include "atostr.h"
#include "atoerr.h"
#include "atolog.h"
#include "atoutil.h"
#include "atoctx.h"
#include "atopbe.h"
#include "atopk.h"
#include "atobcshim.h"

#include "atobcrypto.h"

//#define MGID_FORCE_OPENSSL10 1

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_CRYPTO;
static unsigned long _moduleid = ATO_BASE_MODULEID_CRYPTO;

/*********************************************************************************/
typedef struct _ato_crypto_obj ato_crypto_obj;
struct _ato_crypto_obj {
    ato_String *b64;
    ato_String *data;
};

#define _ATO_CERT_USER_NR 0
#define _ATO_CERT_INTR_NR 1
#define _ATO_CERT_ROOT_NR 2

struct _ato_Crypto {
    ato_crypto_obj *salt;
    ato_crypto_obj *p7;
    ato_crypto_obj *p8;
    PKCS7 *pkcs7;
    STACK_OF(X509) *certchain;
    X509 *usercert;
    X509 *intrcert;
    X509 *rootcert;
    X509_SIG *pksig;
};


/*********************************************************************************/
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
static void _setcrypto(ato_Ctx *ctx, ato_Crypto *crypto, bool isforced);
static void _extractp7certs(ato_Ctx *ctx, ato_Crypto *crypto);
static ato_String *_p12(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk);
static EVP_PKEY *_getpkey(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk);
static ato_String *_decrypt(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk);
static ato_String *_extractbiodata(ato_Ctx *ctx, BIO *bo);
//static ato_String *_extractbiodatan(ato_Ctx *ctx, BIO *bo);
static ato_String *_cert2der(ato_Ctx *ctx, X509 *cert);

static bool _reencrypt(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk, ato_PwdKdf *newkdf);
static ato_String *_getchain(ato_Ctx *ctx, ato_Crypto *crypto);
static void _checkb64(ato_Ctx *ctx, ato_String *b64, ato_String *data);

static ato_String *_base64decode(ato_Ctx *ctx, ato_String *b64);
static ato_String *_base64encode(ato_Ctx *ctx, ato_String *data);

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

#define ATO_RAND_BUF_SIZE 4096
static unsigned char *_init_rand(void) {
    unsigned char *buf = calloc((ATO_RAND_BUF_SIZE+1), sizeof(char));
    //if (!RAND_status()) {
        RAND_seed(buf, ATO_RAND_BUF_SIZE);
        if (!RAND_status()) {
            buf = ato_free(buf);
            //printf("\n%s\n", "rand bad");
        } else {
            //printf("\n%s\n", "rand good");
        }
    //}
    return buf;
}

/*********************************************************************************/

int ato__crypto_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    bcshim_init();

    //ato_free(_init_rand());

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__crypto_deinit(void)
{
    static bool invoked = FALSE;
    if (invoked) return;
    invoked = TRUE;

    bcshim_deinit();
}

/*********************************************************************************/
ato_String *ato_crypto_b64salt(ato_Crypto *crypto) { assert(crypto != NULL); assert(crypto->salt != NULL); return crypto->salt->b64; }
ato_String *ato_crypto_b64p7(ato_Crypto *crypto) { assert(crypto != NULL); assert(crypto->p7 != NULL); return crypto->p7->b64; }
ato_String *ato_crypto_b64p8(ato_Crypto *crypto) { assert(crypto != NULL); assert(crypto->p8 != NULL); return crypto->p8->b64; }

static ato_Crypto *_create(ato_Ctx *ctx, ato_Crypto **obj) {
    ato_Crypto *crypto = NULL;
    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = crypto = calloc(1, sizeof(ato_Crypto));
    assert(crypto != NULL);

    crypto->p7 = calloc(1, sizeof(ato_crypto_obj));
    assert(crypto->p7 != NULL);
    crypto->p8 = calloc(1, sizeof(ato_crypto_obj));
    assert(crypto->p8 != NULL);
    crypto->salt = calloc(1, sizeof(ato_crypto_obj));
    assert(crypto->salt != NULL);

    ATO_IGNORE(ctx);
    return crypto;
}
void ato_crypto_create(ato_Ctx *ctx, ato_Crypto **obj, const char *b64p7c, const char *b64p8, const char *b64salt) {
    ato_Crypto *crypto = _create(ctx, obj);
    assert(crypto != NULL);
    assert(b64p7c != NULL); assert(b64p8 != NULL); assert(b64salt != NULL);
    ato_str_create(&(crypto->p7->b64), ato_strdup(b64p7c, 0), strlen(b64p7c), TRUE);
    ato_str_create(&(crypto->p8->b64), ato_strdup(b64p8, 0), strlen(b64p8), TRUE);
    ato_str_create(&(crypto->salt->b64), ato_strdup(b64salt, 0), strlen(b64salt), TRUE);
}
void ato_crypto_create_ref(ato_Ctx *ctx, ato_Crypto **obj, char *b64p7c, char *b64p8, char *b64salt) {
    ato_Crypto *crypto = _create(ctx, obj);
    assert(crypto != NULL);
    assert(b64p7c != NULL); assert(b64p8 != NULL); assert(b64salt != NULL);
    ato_str_create(&(crypto->p7->b64), b64p7c, strlen(b64p7c), TRUE);
    ato_str_create(&(crypto->p8->b64), b64p8, strlen(b64p8), TRUE);
    ato_str_create(&(crypto->salt->b64), b64salt, strlen(b64salt), TRUE);
}
void ato_crypto_createstr_ref(ato_Ctx *ctx, ato_Crypto **obj, ato_String *b64p7c, ato_String *b64p8, ato_String *b64salt) {
    ato_Crypto *crypto = _create(ctx, obj);
    assert(crypto != NULL);
    assert(b64p7c != NULL); assert(b64p8 != NULL); assert(b64salt != NULL);
    crypto->p7->b64 = b64p7c;
    crypto->p8->b64 = b64p8;
    crypto->salt->b64 = b64salt;
}

static void *_free_cryptoobj(ato_crypto_obj *co)
{
    if (!co) return NULL;
    co->b64 = ato_str_free(co->b64);
    co->data = ato_str_free(co->data);
    free(co);
    return NULL;
}

void *ato_crypto_free(ato_Crypto *crypto)
{
    if (crypto == NULL) return NULL;
    crypto->p7 = _free_cryptoobj(crypto->p7);
    crypto->p8 = _free_cryptoobj(crypto->p8);
    crypto->salt = _free_cryptoobj(crypto->salt);

    if (crypto->pkcs7 != NULL) { PKCS7_free(crypto->pkcs7); crypto->pkcs7 = NULL; }
    if (crypto->pksig != NULL) { X509_SIG_free(crypto->pksig); crypto->pksig = NULL; }
    //if (crypto->certchain != NULL) { sk_X509_pop_free(crypto->certchain, X509_free); crypto->certchain = NULL; }
    //if (crypto->certchain != NULL) { sk_X509_free(crypto->certchain); crypto->certchain = NULL; }
    crypto->usercert = crypto->intrcert = crypto->rootcert = NULL;

    free(crypto);
    return NULL;
}

/*********************************************************************************/

static void _fix_time(ASN1_TIME *asn1time, char **timeStr);

typedef enum {ROOT_CERT, USER_CERT} cert_type;

static char *_crypto_cert_value(ato_Ctx *ctx, ato_Crypto *crypto, char *buf, size_t len, cert_type cert_t, int nid) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    X509_NAME *name = NULL;
    if (crypto == NULL) { return NULL; }
    Try {
        _setcrypto(ctx, crypto, FALSE);
        switch (cert_t) {
        case ROOT_CERT:
            name = X509_get_subject_name(crypto->rootcert);
            break;
        case USER_CERT:
            name = X509_get_subject_name(crypto->usercert);
            break;
        }
        X509_NAME_get_text_by_NID(name, nid, buf, len);
    } Catch(errcode) {
        *buf = '\0';
    }
    return errcode == ATO_ERR_OK  ? buf : NULL;
}

char *ato_crypto_rootcert_name(ato_Ctx *ctx, ato_Crypto *crypto, char *buf, size_t len) {
    return _crypto_cert_value(ctx, crypto, buf, len, ROOT_CERT, NID_commonName);
}

char *ato_crypto_usercert_name(ato_Ctx *ctx, ato_Crypto *crypto, char *buf, size_t len) {
    return _crypto_cert_value(ctx, crypto, buf, len, USER_CERT, NID_commonName);
}

char *ato_crypto_usercert_abn(ato_Ctx *ctx, ato_Crypto *crypto, char *buf, size_t len) {
    return _crypto_cert_value(ctx, crypto, buf, len, USER_CERT, NID_organizationName);
}

static bool _crypto_isalgnid(ato_Crypto *crypto, int nid) {
    assert(crypto != NULL);
    assert(crypto->pksig != NULL);
    return bcshim_X509_SIG_get_nid(crypto->pksig) == nid;
}

bool ato_crypto_ispbe2(ato_Crypto *crypto) {
    return _crypto_isalgnid(crypto, NID_pbes2);
}

void ato_crypto_iscorrectpwd(ato_Ctx *ctx, ato_Crypto *crypto, bool *correct, ato_PwdKdf *pk)
{
    EVP_PKEY *pkey = NULL;
    assert(crypto != NULL);
    _setcrypto(ctx, crypto, FALSE);
    pkey = _getpkey(ctx, crypto, pk);
    *correct = (pkey != NULL);
    if (pkey != NULL) EVP_PKEY_free(pkey);
}

bool ato_crypto_changepwd(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk, ato_PwdKdf *newkdf)
{
    assert(pk != NULL); assert(newkdf != NULL);
    _setcrypto(ctx, crypto, FALSE);
    return _reencrypt(ctx, crypto, pk, newkdf);
}

void ato_crypto_certificate(ato_Ctx *ctx, ato_Crypto *crypto, ato_String **certificate)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_crypto_certificate";

    ATO_ASSERT_ISNOTALLOCATED(certificate);

    _setcrypto(ctx, crypto, FALSE);

    if ((*certificate = _cert2der(ctx, crypto->usercert)) == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_DER, "Failed to DER encode certificate");
}

void ato_crypto_chain(ato_Ctx *ctx, ato_Crypto *crypto,  ato_String **certificates)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_crypto_chain";
    assert(crypto != NULL);

    ATO_ASSERT_ISNOTALLOCATED(certificates);

    _setcrypto(ctx, crypto, FALSE);

    if ((*certificates = _getchain(ctx, crypto)) == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_DER, "Failed to DER encode certificate chain");
}

bool ato_crypto_privatekey(ato_Ctx *ctx, ato_Crypto *crypto, ato_String **privatekey, ato_PwdKdf *pk)
{
    assert(crypto != NULL);

    ATO_ASSERT_ISNOTALLOCATED(privatekey);
    assert(pk != NULL);

    _setcrypto(ctx, crypto, FALSE);

    if ((*privatekey = _decrypt(ctx, crypto, pk)) == NULL)
        return FALSE;

    return TRUE;
}

bool ato_crypto_p12(ato_Ctx *ctx, ato_Crypto *crypto, ato_String **p12, ato_PwdKdf *pk)
{
    assert(crypto != NULL);

    ATO_ASSERT_ISNOTALLOCATED(p12);
    assert(pk != NULL);

    _setcrypto(ctx, crypto, FALSE);

    if ((*p12 = _p12(ctx, crypto, pk)) == NULL)
        return FALSE;

    return TRUE;
}

ato_String *ato_crypto_serialnr(ato_Ctx *ctx, ato_Crypto *crypto, ato_String **serialNr)
{
    ASN1_INTEGER *serialnr = NULL;
    BIGNUM *bnser = NULL;
    char *serialNumber = NULL;

    ATO_ASSERT_ISNOTALLOCATED(serialNr);
    assert(crypto != NULL);
    _setcrypto(ctx, crypto, FALSE);

    serialnr = X509_get_serialNumber(crypto->usercert); // DO NOT FREE
    if (serialnr == NULL){
        //printf("ato_crypto_serialnr  error getting serial number.\n");
        goto cleanup;
    }
    bnser = ASN1_INTEGER_to_BN(serialnr, NULL);
    serialNumber = BN_bn2dec(bnser);

    ato_str_create(serialNr, serialNumber, strlen(serialNumber), TRUE);

    cleanup:
    if (bnser != NULL) { BN_free(bnser); }

    return *serialNr;
}

//ato_String *ato_crypto_sha1fingerprint(ato_Ctx *ctx, ato_Crypto *crypto)
//{
//    ato_String *buffer = NULL;
//    //BIO *bo = NULL;
//    assert(crypto != NULL);
//    _setcrypto(ctx, crypto, FALSE);
//
//    {
//        unsigned int len = 0;
//        char md[EVP_MAX_MD_SIZE+1];
//        if (X509_digest(crypto->usercert, EVP_sha1(), (unsigned char *)md, &len)) {
//            ato_str_create(&buffer, md, (size_t)len, FALSE);
//            ato_str_setformat(buffer, ATO_STR_FORMAT_BINARY);
//        }
//    }
//    return buffer;
//}

ato_String *ato_crypto_notbefore(ato_Ctx *ctx, ato_Crypto *crypto, ato_String **notBefore)
{
    char *data = NULL;
    ATO_ASSERT_ISNOTALLOCATED(notBefore);
    assert(crypto != NULL);

    _setcrypto(ctx, crypto, FALSE);

    if (crypto->usercert != NULL) {
        ASN1_TIME *asn1time = X509_get_notBefore(crypto->usercert);
        _fix_time(asn1time, &data);
    }
    return ato_str_create(notBefore, (char*)data, strlen(data), TRUE);
}

ato_String *ato_crypto_notafter(ato_Ctx *ctx, ato_Crypto *crypto, ato_String **notAfter)
{
    char *data = NULL;
    ATO_ASSERT_ISNOTALLOCATED(notAfter);
    assert(crypto != NULL);

    _setcrypto(ctx, crypto, FALSE);

    _fix_time( X509_get_notAfter(crypto->usercert), &data);

    return ato_str_create(notAfter, data, strlen(data), TRUE);
}

/*********************************************************************************/

//format time string
static void _fix_time(ASN1_TIME *asn1time, char **timeStr)
{
    const char *str = NULL;
    size_t i = 0;
    char *timeptr = NULL;

    assert(asn1time != NULL);
    str = (const char*) asn1time->data;
//  printf("_fix_time input:%s.\n", str);

    *timeStr = calloc(40, sizeof(char));
    timeptr = *timeStr;
    if (asn1time->type == V_ASN1_UTCTIME) /* two digit year */
    {
        //Assume 21st century only !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        *timeptr++ = '2';
        *timeptr++ = '0';
    }

    //year
    *timeptr++ = str[i++];
    *timeptr++ = str[i++];
    *timeptr++ = '-';

    //month
        *timeptr++ = str[i++];
    *timeptr++ = str[i++];
    *timeptr++ = '-';
    //day
    *timeptr++ = str[i++];
    *timeptr++ = str[i++];
    //time
    *timeptr++ = 'T';
    *timeptr++ = str[i++]; *timeptr++ = str[i++];
    *timeptr++ = ':';
    *timeptr++ = str[i++]; *timeptr++ = str[i++];
    *timeptr++ = ':';
    *timeptr++ = str[i++]; *timeptr++ = str[i++];
    //*timeptr++ = '.'; *timeptr++ = '0'; *timeptr++ = '0';*timeptr++ = '0'; //dummy milliseconds

    //time zone - TODO daylight saving? Assuming AUSt EST.
    *timeptr++ = '+';
    *timeptr++ = '1';
    *timeptr++ = '0';
    *timeptr++ = ':';
    *timeptr++ = '0';
    *timeptr++ = '0';
    *timeptr = 0; //terminate

//  printf("_fix_time produced:%s.\n", *timeStr);
    return;
}
/*
isforced FALSE will not redo the crypto operations if already done. If TRUE, then the
fields are cleared and the crypto redone.
*/
static void _setcrypto(ato_Ctx *ctx, ato_Crypto *crypto, bool isforced)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_setcrypto";
    assert(ctx != NULL);
    assert(crypto != NULL);
    if (!isforced && crypto->certchain != NULL){
        return;
        }

    if (crypto->salt->data == NULL)
        crypto->salt->data = _base64decode(ctx, crypto->salt->b64);
    if (crypto->salt->data == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_B64, "Failed to decode salt");
    if (crypto->p7->data == NULL)
        crypto->p7->data = _base64decode(ctx, crypto->p7->b64);
    if (crypto->p7->data == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_B64, "Failed to decode p7");
    if (crypto->p8->data == NULL)
        crypto->p8->data = _base64decode(ctx, crypto->p8->b64);
    if (crypto->p8->data == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_B64, "Failed to decode p8");
    _checkb64(ctx, crypto->p8->b64, crypto->p8->data);
    if (crypto->certchain == NULL)
        _extractp7certs(ctx, crypto);
    if (crypto->certchain == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_P7, "Failed to extract cert chain from p7");
    if (crypto->pksig == NULL) {
        BIO *bi = NULL;
        bi = BIO_new_mem_buf(ATO_REMOVE_CONST(ato_str_value(crypto->p8->data), char *), ato_str_len(crypto->p8->data));
        crypto->pksig = d2i_PKCS8_bio(bi, NULL);
        BIO_free_all(bi);
    }
    if (crypto->pksig == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_P8, "Failed to extract p8 (privatekey)");
}

// if > 1, then just return ok.
// if <= 0, then check if we should retry, and if not, it's an error, else do it again.
static bool _flush(BIO *b)
{
    int count = 0;
    int i = 0;

    do {

        i = BIO_flush(b);
        if (i > 0)
            return TRUE; // success
        else if (!BIO_should_retry(b))
            return FALSE; // fail

        if (count++ > 50)
            return FALSE; // failsafe in case of 'infinite' looping

    } while (i <= 0);

    return FALSE;
}

static void _freebufmem(void *data) { if (data) BUF_MEM_free((BUF_MEM *)data); }

static ato_String *_extractbiodata(ato_Ctx *ctx, BIO *bo) {
    static bool use_BIO_NOCLOSE = FALSE; // FALSE = openssl bug https://bugzilla.redhat.com/show_bug.cgi?id=1691853
    BUF_MEM *bptr = NULL;
    ato_String *str = NULL;

    if (_flush(bo)) {
        BIO_get_mem_ptr(bo, &bptr);
        if (bptr && bptr->data) {
            size_t bptrlength= (size_t)(int)bptr->length;
            assert(bptrlength > 0);
            if (use_BIO_NOCLOSE) {
                (void)BIO_set_close(bo, BIO_NOCLOSE);
                ato_str_create(&str, bptr->data, bptrlength, FALSE);
                ato_str_setfreefn(str, _freebufmem, bptr);
            } else {
                ato_str_create(&str, ato_memdup(bptr->data, bptrlength, 0), bptrlength, FALSE);
                ato_str_setfreefn(str, _freebufmem, NULL);
            }
        }
    }

    ATO_IGNORE(ctx);
    return str;
}

static ato_String *_cert2der(ato_Ctx *ctx, X509 *cert)
{
    BIO *bo = NULL;
    ato_String *certificate = NULL;

    bo = BIO_new(BIO_s_mem());
    if (bo == NULL) goto finally;
    i2d_X509_bio(bo, cert);
    certificate = _extractbiodata(ctx, bo);

finally:
    if (bo != NULL) BIO_free_all(bo);
    return certificate;
}

static void _checkb64(ato_Ctx *ctx, ato_String *b64, ato_String *data)
{
#if defined(DEBUG) || defined(_DEBUG)
    int r = 0;
    ato_String *b = NULL;
    b = _base64encode(ctx, data);
    r = memcmp(ato_str_value(b), ato_str_value(b64), ato_str_len(b64));
    ato_str_free(b);
    assert(r == 0);
#else
    if (b64) {;}
    if (data) {;}
#endif
    ATO_IGNORE(ctx);
}

#define B64_BUF_SIZE 8192
static ato_String *_base64decode(ato_Ctx *ctx, ato_String *b64)
{
    BIO *mem = NULL, *bi = NULL, *bi64 = NULL, *bo = NULL;
    char inbuf[B64_BUF_SIZE];
    int inlen = 0;
    ato_String *data = NULL;

    if (b64 == NULL)
        return 0;

    mem = BIO_new_mem_buf(ATO_REMOVE_CONST(ato_str_value(b64), char *), ato_str_len(b64));
    assert(mem != NULL);
    BIO_set_mem_eof_return(mem, 0);
    bi64 = BIO_new(BIO_f_base64());
    BIO_set_flags(bi64, BIO_FLAGS_BASE64_NO_NL);
    bi = BIO_push(bi64, mem);
    bo = BIO_new(BIO_s_mem());

    while ((inlen = BIO_read(bi, inbuf, B64_BUF_SIZE)) > 0) {
        BIO_write(bo, inbuf, inlen);
    }

    data = _extractbiodata(ctx, bo);

    BIO_free_all(bi);
    BIO_free_all(bo);
    return data;
}

static ato_String *_base64encode(ato_Ctx *ctx, ato_String *data)
{
    BIO *mem = NULL, *bi = NULL, *bo64 = NULL, *bo = NULL;
    char inbuf[B64_BUF_SIZE];
    int inlen = 0;
    ato_String *b64 = NULL;

    bi = BIO_new_mem_buf(ATO_REMOVE_CONST(ato_str_value(data), char *), ato_str_len(data));
    assert(bi != NULL);
    BIO_set_mem_eof_return(bi, 0);
    mem = BIO_new(BIO_s_mem());
    assert(mem != NULL);
    bo64 = BIO_new(BIO_f_base64());
    assert(bo64 != NULL);
    BIO_set_flags(bo64, BIO_FLAGS_BASE64_NO_NL);
    bo = BIO_push(bo64, mem);

    while ((inlen = BIO_read(bi, inbuf, B64_BUF_SIZE)) > 0) {
        BIO_write(bo, inbuf, inlen);
    }

    b64 = _extractbiodata(ctx, bo);
    ////////if (len > 0) (*b64)[len] = '\0';

    BIO_free_all(bi);
    BIO_free_all(bo);
    return b64;
}

static void _extractp7certs(ato_Ctx *ctx, ato_Crypto *crypto)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_extractp7certs";
    int errcode = ATO_ERR_OK;
    int i = 0;
    BIO *bi = NULL;

    assert(crypto->p7 != NULL);
    assert(crypto->p7->b64 != NULL);
    assert(crypto->certchain == NULL);
    assert(crypto->pkcs7 == NULL);
    bi = BIO_new_mem_buf(ATO_REMOVE_CONST(ato_str_value(crypto->p7->data), char *), ato_str_len(crypto->p7->data));
    if (d2i_PKCS7_bio(bi, &(crypto->pkcs7)) == NULL) {
        errcode = ATO_CTX_NEWERR(ctx, ATO_ERR_UNKNOWN, "Failed to convert to P7 certificate");
        goto finally;
    }
    if (bi) { BIO_free_all(bi); bi = NULL; }

    i = OBJ_obj2nid(crypto->pkcs7->type);
    switch (i) {
        case NID_pkcs7_signed:
            crypto->certchain=crypto->pkcs7->d.sign->cert;
            break;
        case NID_pkcs7_signedAndEnveloped:
            crypto->certchain=crypto->pkcs7->d.signed_and_enveloped->cert;
            break;
        default:
            break;
    }

            crypto->usercert = sk_X509_value(crypto->certchain, _ATO_CERT_USER_NR);
            crypto->intrcert = sk_X509_value(crypto->certchain, _ATO_CERT_INTR_NR);
            crypto->rootcert = sk_X509_value(crypto->certchain, _ATO_CERT_ROOT_NR);


    finally:
    if (bi) { BIO_free_all(bi); bi = NULL; }
    ATO_RETHROW_ONERR(errcode);
}

/*
static void _extractp7certs(ato_Ctx *ctx, ato_Crypto *crypto)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_extractp7certs";
    int errcode = ATO_ERR_OK;
    int i = 0;
    STACK_OF(X509) *certs=NULL;
    BIO *bi = NULL;

    crypto->usercert = NULL;
    crypto->certchain = NULL;

    assert(crypto->p7->b64 != NULL);
    bi = BIO_new_mem_buf(ATO_REMOVE_CONST(ato_str_value(crypto->p7->data), char *), ato_str_len(crypto->p7->data));
    if (d2i_PKCS7_bio(bi, &(crypto->pkcs7)) == NULL) {
    //if ((crypto->pkcs7 = d2i_PKCS7_bio(bi, NULL)) == NULL) {
    //bi = BIO_new_mem_buf(ATO_REMOVE_CONST(ato_str_value(crypto->p7->b64), char *), ato_str_len(crypto->p7->b64));
    //if (PEM_read_bio_PKCS7(bi, &(crypto->pkcs7), NULL, NULL) == NULL) {
        errcode = ATO_CTX_NEWERR(ctx, ATO_ERR_UNKNOWN, "Failed to convert to P7 certificate");
        goto finally;
    }
    if (bi) { BIO_free_all(bi); bi = NULL; }

    i = OBJ_obj2nid(crypto->pkcs7->type);
    switch (i) {
        case NID_pkcs7_signed:
            certs=crypto->pkcs7->d.sign->cert;
            break;
        case NID_pkcs7_signedAndEnveloped:
            certs=crypto->pkcs7->d.signed_and_enveloped->cert;
            break;
        default:
            break;
    }

    crypto->certchain = sk_X509_new_null();
    if (crypto->certchain != NULL && certs != NULL) {
        X509 *x = NULL;
        int count = sk_X509_num(certs);
        if (count > 0) {
            for (i = 0; i < count; i++) {
                x = sk_X509_value(certs, i);
                if (!sk_X509_push(crypto->certchain, x)) {
                    errcode = ATO_CTX_NEWERR(ctx, ATO_ERR_UNKNOWN, "Failed to extract P7 certificate");
                    goto finally;
                }
            }
            crypto->usercert = sk_X509_value(crypto->certchain, _ATO_CERT_USER_NR);
            crypto->intrcert = sk_X509_value(crypto->certchain, _ATO_CERT_INTR_NR);
            crypto->rootcert = sk_X509_value(crypto->certchain, _ATO_CERT_ROOT_NR);
            //X509_keyid_set1(crypto->usercert, NULL, 0);
            //X509_alias_set1(crypto->usercert, NULL, 0);
        }
    }

    finally:
    if (bi) { BIO_free_all(bi); bi = NULL; }
    ATO_RETHROW_ONERR(errcode);
}
*/

static ato_String *_getchain(ato_Ctx *ctx, ato_Crypto *crypto)
{
    BIO *bo = NULL;
    ato_String *certificates = NULL;

    bo = BIO_new(BIO_s_mem());
    if (bo == NULL) goto finally;
    // if (!_write_pkcs7certs(bo, crypto->pkcs7)) goto finally;  //cert PEM

    // PEM_write_bio_PKCS7(bo, crypto->pkcs7); //p7 PEM
    if (!i2d_PKCS7_bio(bo, crypto->pkcs7)) goto finally; //DER

    certificates = _extractbiodata(ctx, bo);

finally:
    if (bo != NULL) BIO_free_all(bo);

    ATO_IGNORE(ctx);
    return certificates;
}

static int _pbe_get_nid(const ato_Pbe *pbe) {
    int nid = 0;
    switch (ato_pbe_hmac(pbe)) {
        case 0: nid = NID_pbe_WithSHA1And3_Key_TripleDES_CBC; break;
        case 1: nid = NID_hmacWithSHA1; break;
        case 224: nid = NID_hmacWithSHA224; break;
        case 256: nid = NID_hmacWithSHA256; break;
        case 384: nid = NID_hmacWithSHA384; break;
        case 512: nid = NID_hmacWithSHA512; break;
    }
    return nid;
}
static const EVP_CIPHER *_pbe_get_cipher(const ato_Pbe *pbe) {
    const EVP_CIPHER *cipher = NULL;
    switch (ato_pbe_enc(pbe)) {
        case 0: cipher = NULL; break;
        case 3: cipher = EVP_des_ede3_cbc(); break;
        case 128: cipher = EVP_aes_128_cbc(); break;
        case 192: cipher = EVP_aes_192_cbc(); break;
        case 256: cipher = EVP_aes_256_cbc(); break;
    }
    return cipher;
}

//Extracted underlying _encrypt working to remove assumption that have a single crypto object.
static bool _encrypt_pkey(ato_Ctx *ctx, ato_String* salt, EVP_PKEY *pkey, ato_PwdKdf *pk, X509_SIG **xp8, ato_String **xp8data, ato_String **b64xp8)
{
    bool ok = FALSE;
    BIO *bo = NULL;
    const EVP_CIPHER *cipher = NULL;
    PKCS8_PRIV_KEY_INFO *p8inf = NULL;
    const ato_Pbe *pbe = ato_pk_pbe(pk);
    int pbe_prf_nid, pbe_iter;
    assert(pbe != NULL);
    if (*xp8 != NULL) { X509_SIG_free(*xp8); *xp8 = NULL; }

    pbe_prf_nid = _pbe_get_nid(pbe);
    if (pbe_prf_nid == 0) { goto finally; }
    pbe_iter = ato_pbe_iter(pbe);
    cipher = _pbe_get_cipher(pbe);

    p8inf = EVP_PKEY2PKCS8(pkey);
    if (p8inf == NULL) { goto finally; }

    if (!ato_pbe_isPbe2(pbe)) {
        *xp8 = PKCS8_encrypt(pbe_prf_nid, cipher, ato_pk_pwd(ctx, pk), (int)strlen(ato_pk_pwd(ctx, pk)), ATO_REMOVE_CONST(ato_str_value(salt), unsigned char *), ato_str_len(salt), pbe_iter, p8inf);
    } else {
        if (cipher == NULL) { goto finally; }
        *xp8 = bcshim_PKCS8_encrypt_pbe2(pbe_prf_nid, cipher, ato_pk_pwd(ctx, pk), (int)strlen(ato_pk_pwd(ctx, pk)), ATO_REMOVE_CONST(ato_str_value(salt), unsigned char *), ato_str_len(salt), pbe_iter, p8inf);
    }
    if (*xp8 == NULL) { goto finally; }

    bo = BIO_new(BIO_s_mem());
    i2d_PKCS8_bio(bo, *xp8);
    *xp8data = ato_str_free(*xp8data);
    *xp8data = _extractbiodata(ctx, bo);
    if (*xp8data) {
        ato_String *b64 = _base64encode(ctx, *xp8data);
        *b64xp8 = ato_str_free(*b64xp8);
        ato_str_dup2(b64xp8, b64, TRUE); // We need this as null terminated string
        ato_str_free(b64);
        ok = TRUE;
    }
    else {
        ok = FALSE;
    }

finally:
    if (p8inf != NULL) PKCS8_PRIV_KEY_INFO_free(p8inf);
    if (bo != NULL) BIO_free_all(bo);

    return ok;
}

static bool _encrypt_pkey_b64salt(ato_Ctx *ctx, ato_String* b64salt, EVP_PKEY *pkey, ato_PwdKdf *pk, X509_SIG **xp8, ato_String **xp8data, ato_String **b64xp8) {
    bool ok = FALSE;
    ato_String *salt = NULL;
    int errcode = ato_base64decode(ctx, &salt, b64salt);
    if (errcode == ATO_ERR_OK) {
        ok = _encrypt_pkey(ctx, salt, pkey, pk, xp8, xp8data, b64xp8);
    }
    ato_str_free(salt);
    return ok;
}


static bool _encrypt(ato_Ctx *ctx, ato_Crypto *crypto, EVP_PKEY *pkey, ato_PwdKdf *pk)
{
    bool ok = FALSE;

    ok = _encrypt_pkey(ctx, crypto->salt->data, pkey, pk, &crypto->pksig, &crypto->p8->data, &crypto->p8->b64);

    return ok;
}

static bool _reencrypt(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk, ato_PwdKdf *newkdf)
{
    bool ok = FALSE;
    EVP_PKEY *pkey = NULL;

    if ((pkey = _getpkey(ctx, crypto, pk)) == NULL)
        goto finally;
        //Throw ATO_CTX_NEWERR(ctx, ATO_ERR_DECRYPT, "Internal error - failed to decrypt the private key");

    ok = _encrypt(ctx, crypto, pkey, newkdf);
    //if (!ok)
        //Throw ATO_CTX_NEWERR(ctx, ATO_ERR_ENCRYPT, "Internal error - failed to reencrypt the private key");
finally:
    if (pkey != NULL) EVP_PKEY_free(pkey);
    return ok;
}

static ato_String *_decrypt(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk)
{
    BIO *bo = NULL;
    EVP_PKEY *pkey=NULL;
    ato_String *privatekey = NULL;

    if ((pkey = _getpkey(ctx, crypto, pk)) == NULL)
        goto finally;

    bo = BIO_new(BIO_s_mem());
    if (!i2d_PrivateKey_bio(bo, pkey)) goto finally;

    privatekey = _extractbiodata(ctx, bo);

finally:
    if (pkey != NULL) EVP_PKEY_free(pkey);
    if (bo != NULL) BIO_free_all(bo);

    return privatekey;
}

static EVP_PKEY *_getpkey(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk)
{
    PKCS8_PRIV_KEY_INFO *p8inf = NULL;
    EVP_PKEY *pkey = NULL;
    const char *pwd = NULL;


    if (_crypto_isalgnid(crypto, NID_pbe_WithSHA1And3_Key_TripleDES_CBC)) {
        pwd = ato_pk_rpwd(pk);
    } else {
        pwd = ato_pk_dpwd(ctx, pk);
    }
    p8inf = PKCS8_decrypt(crypto->pksig, pwd, strlen(pwd));
    if (p8inf == NULL)
        goto finally;
    pkey = EVP_PKCS82PKEY(p8inf);

finally:
    if (p8inf != NULL) PKCS8_PRIV_KEY_INFO_free(p8inf);
    ATO_IGNORE(ctx);
    return pkey;
}

static ato_String *_p12(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk)
{
    int i = 0, count = 0;
    BIO *bo = NULL;
    EVP_PKEY *pkey = NULL;
    STACK_OF(X509) *certs = NULL;
    X509 *ucert = NULL;
    PKCS12 *p12 = NULL;
    int keytype = 0;
    //char *lpwd = NULL;
    ato_String *p12array = NULL;

    pkey = _getpkey(ctx, crypto, pk);
    if (pkey == NULL)
        goto finally;

    // https://wiki.nikhef.nl/grid/How_to_handle_OpenSSL_and_not_get_hurt_and_what_does_that_library_call_really_do%3F#Proper_memory_liberation_of_a_STACK_OF_.28X509.29_.2A
    // https://www.gsp.com/cgi-bin/man.cgi?section=3&topic=sk_X509_value
    certs = sk_X509_dup(crypto->certchain);
    /* Find cert for matching private key  */
    count = sk_X509_num(certs);
    for(i = 0; i < count; i++)  {
        X509 *x = sk_X509_value(certs, i);
        if(X509_check_private_key(x, pkey)) {
            ucert = x;
            X509_keyid_set1(ucert, NULL, 0);
            X509_alias_set1(ucert, NULL, 0);
            (void)sk_X509_delete(certs, i);
            break;
        }
    }
    if (ucert == NULL)
        goto finally;

    p12 = PKCS12_create(ATO_REMOVE_CONST(ato_pk_pwd(ctx, pk), char *), NULL, pkey, ucert, certs,
        NID_pbe_WithSHA1And3_Key_TripleDES_CBC, NID_pbe_WithSHA1And40BitRC2_CBC, PKCS12_DEFAULT_ITER, -1, keytype);
    PKCS12_set_mac(p12, ato_pk_pwd(ctx, pk), -1, NULL, 0, PKCS12_DEFAULT_ITER, NULL);

    bo = BIO_new(BIO_s_mem());
    i2d_PKCS12_bio(bo, p12);

    p12array = _extractbiodata(ctx, bo);

finally:
    if (pkey != NULL) EVP_PKEY_free(pkey);
    if (bo != NULL) BIO_free_all(bo);
    if (p12 != NULL) PKCS12_free(p12);
    if (certs != NULL) sk_X509_free(certs); // DO NOT USE sk_X509_pop_free - the certs still in the crypto->certchain
    //if (lpwd != NULL) free(lpwd);
    return p12array;
}

/*********************************************************************************/
char *ato_md5(ato_Ctx *ctx, char *data, size_t len, bool hex)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_md5";
    const EVP_MD *md_type = EVP_get_digestbyname("md5");
    char *md_value = NULL;
    char *value = NULL;
    unsigned int md_len = 0;
    ATO_IGNORE(ctx);
    value = md_value = calloc(EVP_MAX_MD_SIZE+1, sizeof(char));
    assert(value);
    //if (!EVP_Digest(data, len, (unsigned char *)md_value, (unsigned int *)&md_len, md_type, NULL)) {
    if (!EVP_Digest(data, len, (unsigned char *)md_value, &md_len, md_type, NULL)) {
        value = ato_free(value);
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_HASH, "Failed to generate MD5 hash");
    }
    if (hex) {
        unsigned int i = 0;
        value = calloc((md_len+5)*2, sizeof(char));
        assert(value);
        for (i = 0; i < md_len; i++) {
            sprintf(value + i*2, "%02X", (unsigned char)(md_value[i]));
        }
        md_value = ato_free(md_value);
    }
    return value;
}

int ato_generatehmac(ato_Ctx *ctx, ato_String **output, ato_String *value, ato_String *salt)
{
    unsigned char md[EVP_MAX_MD_SIZE+1];
    unsigned int mdlen = 0;
    bool ok = TRUE;
    unsigned char key[20];
    int keylen = 20;
    int iter = 1024;
    const char pwdarray[] = { 0 };
    ato_String *hash = NULL;

    ATO_ASSERT_ISNOTALLOCATED(output);
    assert(value != NULL); assert(salt != NULL);

    ato_str_createconst(&hash, pwdarray, 1, FALSE);

    ok = PKCS12_key_gen(
        ato_str_value(hash), ato_str_len(hash),
        ATO_REMOVE_CONST(ato_str_value(salt), unsigned char *), ato_str_len(salt),
        PKCS12_MAC_ID, iter, keylen, key, EVP_sha1());

    if (ok)
        ok = HMAC(EVP_sha1(), key, keylen, (const unsigned char *)ato_str_value(value), ato_str_len(value), md, &mdlen) != NULL;

    if (ok)
        ato_str_create(output, ato_memdup(md, mdlen, 0), mdlen, FALSE);

/* useful if troubleshooting during code development
    if (errcode == ATO_ERR_OK) {
        ato_String *str = NULL;
        ato_filewrite(pwd, "c:\\tmp\\c-ivpwd.txt");
        ato_base64encode(&str, salt);
        ato_filewrite(str, "c:\\tmp\\c-ivsalt.txt");
        ato_str_free(str); str = NULL;
        ato_base64encode(&str, value);
        ato_filewrite(str, "c:\\tmp\\c-ivinput.txt");
        ato_str_free(str); str = NULL;
        ato_base64encode(&str, *output);
        ato_filewrite(str, "c:\\tmp\\c-iv.txt");
        ato_str_free(str); str = NULL;
    }
*/

    ato_str_free(hash);

    ATO_IGNORE(ctx);
    return ok ? ATO_ERR_OK : ATO_ERR_UNKNOWN;
}

int ato_base64decode(ato_Ctx *ctx, ato_String **data, ato_String *b64)
{
    ATO_ASSERT_ISNOTALLOCATED(data);
    *data = _base64decode(ctx, b64);
    if (!*data)
        return ATO_ERR_UNKNOWN;
    return ATO_ERR_OK;
}

int ato_base64encode(ato_Ctx *ctx, ato_String **b64, ato_String *data)
{
    ATO_ASSERT_ISNOTALLOCATED(b64);
    *b64 = _base64encode(ctx, data);
    if (!*b64)
        return ATO_ERR_UNKNOWN;
    return ATO_ERR_OK;
}

int ato_base64encode2(ato_Ctx *ctx, ato_String **b64, ato_String *data, int flags)
{
    ATO_ASSERT_ISNOTALLOCATED(b64);
    *b64 = _base64encode(ctx, data);
    if (!*b64)
        return ATO_ERR_UNKNOWN;
    if (flags == 1) {
        ato_String *b64x = NULL;
        ato_str_dup2(&b64x, *b64, TRUE); // We need this as null terminated string
        ato_str_free(*b64);
        *b64 = b64x;
    }
    return ATO_ERR_OK;
}

int ato_base64decode_array(ato_Ctx *ctx, ato_String **data, const char *b64, size_t len)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    ato_String *str = NULL;
    ato_str_createconst(&str, b64, len, FALSE);
    Try {
        errcode = ato_base64decode(ctx, data, str);
    } Catch(errcode) {
    }
    ato_str_free(str);
    return errcode;
}

int ato_base64encode_array(ato_Ctx *ctx, ato_String **b64, const char *data, size_t len)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    ato_String *str = NULL;
    ato_str_createconst(&str, data, len, FALSE);
    Try {
        errcode = ato_base64encode(ctx, b64, str);
    } Catch(errcode) {
    }
    ato_str_free(str);
    return errcode;
}

#define ATO_RANDFMT_GUID 1
#define ATO_RANDFMT_SALT 2

static char *_pseudo_rand(ato_Ctx *ctx, char *dstbuffer, size_t num, int format)
{
    ato_String *buffer = NULL, *value = NULL;
    size_t ndx = 0;
    BIO *bo = NULL;
    size_t i = 0, chunk = 0;
    unsigned char *buf = _init_rand();

    assert(dstbuffer != NULL);
    *dstbuffer = '\0';

    if (buf == NULL) { goto finally; }

    bo = BIO_new(BIO_s_mem());
    if (bo == NULL) goto finally;

    while (num > 0) {
        chunk = num;
        if (chunk > ATO_RAND_BUF_SIZE)
            chunk = ATO_RAND_BUF_SIZE;
        if (RAND_bytes(buf, (int)chunk) <= 0)
            goto finally;
        for (i = 0; i < chunk; i++) {
            if (format == ATO_RANDFMT_GUID) {
                if (ndx == 4 || ndx == 6 || ndx == 8 || ndx == 10)
                    BIO_printf(bo, "-");
                BIO_printf(bo, "%02x", buf[i]);
            } else if (format == ATO_RANDFMT_SALT) {
                BIO_printf(bo, "%c", buf[i]);
            } else {
                BIO_printf(bo, "%c", buf[i]);
            }
            ndx++;
        }
        num -= chunk;
    }

    buffer = _extractbiodata(ctx, bo);
    if (buffer == NULL)
        goto finally;
    value = buffer;

    if (format == ATO_RANDFMT_SALT) {
        value = _base64encode(ctx, buffer);
    }

    memcpy(dstbuffer, ato_str_value(value), ato_str_len(value));
    dstbuffer[ato_str_len(value)] = '\0';

finally:
    if (buf != NULL) free(buf);
    if (bo != NULL) BIO_free_all(bo);
    if (buffer != NULL) ato_str_free(buffer);
    if (value != NULL && value != buffer) ato_str_free(value);
    return *dstbuffer == '\0' ? NULL : dstbuffer;
}

char *ato_new_pseudo_guid(ato_Ctx *ctx, char *guidstr)
{
    return _pseudo_rand(ctx, guidstr, 16, ATO_RANDFMT_GUID);
}
char *ato_new_salt(ato_Ctx *ctx, char *salt)
{
    return _pseudo_rand(ctx, salt, 20, ATO_RANDFMT_SALT);
}

/*
static void dumpstr2file(const char *fname, ato_String *str) {
    const char *data = ato_str_value(str);
    size_t len = ato_str_len(str);
        FILE *fp = fopen(fname, "wb");
        fwrite(data, sizeof(char), len, fp);
        fclose(fp);
        printf("****************************************");
        printf("Written to %s", fname);
        printf("****************************************");
}*/


bool ato_crypto_generateP10new(ato_Ctx *ctx, int keylen, const char *type, const char *abn, const char *name1, const char *name2, const char *pid, ato_PwdKdf *pk, ato_String *b64salt, ato_String **p10, ato_String **p8)
{
    bool res = FALSE;
    X509_REQ *req = NULL;
    EVP_PKEY *pkey = NULL;
    RSA *rsa = NULL;
    X509_NAME *name = NULL;
    STACK_OF(X509_EXTENSION) *exts = NULL;
    X509_EXTENSION *ext = NULL;

    int len = 0;
    char *p10_buf = NULL;
    int abn_nid = 0;
    ASN1_OCTET_STRING *abn_octet = NULL;
    ASN1_IA5STRING *abn_ia5string = NULL;

    X509_SIG *xp8 = NULL;
    ato_String *xp8data = NULL;
    ato_String *p10data = NULL;

    const char *country = "AU";
    char *fullname = NULL;

    unsigned char *encoded_abn = NULL;
    unsigned char *bytes = NULL;

    BIGNUM *bn = NULL;
    bn = BN_new();
    BN_set_word(bn, RSA_F4);
    rsa = RSA_new();
    len = RSA_generate_key_ex(rsa, keylen, bn, NULL); // 1.1
    BN_clear_free(bn); // 1.1
    if (len == 0) goto end;

    // Generate key pair
    pkey=EVP_PKEY_new();
    if (pkey == NULL) goto end;

    if (!EVP_PKEY_assign_RSA(pkey, rsa)) goto end;

    req = X509_REQ_new();
    if(req == NULL) {
        //error or something...
        //printf("X509_REQ_new() failed\n");
        goto end;
    }
    //Set the new public key
    if(!X509_REQ_set_pubkey(req, pkey)) {
        //printf("X509_REQ_set_pubkey  failed\n");
        goto end;
    }

    name = X509_NAME_new();
    if (strcmp(type, "U") == 0) {
        fullname = calloc((strlen(name1) + strlen(name2) + 2), sizeof(char));
        strcpy(fullname, name1);
        strcat(fullname, " ");
        strcat(fullname, name2);
        X509_NAME_add_entry_by_txt(name, "CN", MBSTRING_ASC, (const unsigned char *)fullname, -1, -1, 0);
        X509_NAME_add_entry_by_txt(name, "dnQualifier", MBSTRING_ASC, (const unsigned char *)pid, -1, -1, 0);
        X509_NAME_add_entry_by_txt(name, "O", MBSTRING_ASC, (const unsigned char *)abn, -1, -1, 0);
        X509_NAME_add_entry_by_txt(name, "C", MBSTRING_ASC, (const unsigned char *)country, -1, -1, 0);
        free(fullname);
    } else if (strcmp(type, "D") == 0) {
        X509_NAME_add_entry_by_txt(name, "dnQualifier", MBSTRING_ASC, (const unsigned char *)"ABR", -1, -1, 0);
        X509_NAME_add_entry_by_txt(name, "CN", MBSTRING_ASC, (const unsigned char *)name1, -1, -1, 0);
        X509_NAME_add_entry_by_txt(name, "O", MBSTRING_ASC, (const unsigned char *)abn, -1, -1, 0);
        X509_NAME_add_entry_by_txt(name, "C", MBSTRING_ASC, (const unsigned char *)country, -1, -1, 0);
    }

    if(!X509_REQ_set_subject_name(req, name)) {
        //printf("set subject name failed!!\n");
        goto end;
    }

    // Add abn extension to asn1 structure
    abn_ia5string = ASN1_IA5STRING_new();
    ASN1_STRING_set(abn_ia5string, abn, -1);

    len = i2d_ASN1_IA5STRING(abn_ia5string, NULL); // 1.1
    bytes = calloc(len, sizeof(unsigned char));
    encoded_abn = bytes;
    i2d_ASN1_IA5STRING(abn_ia5string, &bytes); // 1.1 // TODO check for error here

    abn_octet = ASN1_OCTET_STRING_new();
    {
        unsigned char *encoded_abn2 = calloc((len + 1), sizeof(unsigned char));
        memcpy(encoded_abn2, encoded_abn, len);
        encoded_abn2[len] = '\0';
        ASN1_OCTET_STRING_set(abn_octet, encoded_abn2, -1);
        free(encoded_abn2);
    }

    abn_nid = OBJ_create("1.2.36.1.333.1", "abn", "abn Extension");
    ext = X509_EXTENSION_create_by_NID(NULL, abn_nid, FALSE, abn_octet);
    exts = sk_X509_EXTENSION_new_null();
    if (!sk_X509_EXTENSION_push(exts, ext)) {
        //printf("push may have failed");
    }
    if (!X509_REQ_add_extensions(req, exts)) {
        //printf("Error adding exts\n");
        goto end;
    }

    // sign with the new key.
    if (!X509_REQ_sign(req, pkey, EVP_sha256())) {
        //TODO: error handling
        //printf("sign req failed!\n");
        goto end;
    }

    //DER encode req. i2d_X509_REQ returns the length of p10
    len = i2d_X509_REQ(req, (unsigned char**)(&p10_buf));
    if (len < 1) {
        //printf("error DER encoding req\n");
        goto end;
    }

    ato_str_createconst(&p10data, p10_buf, len, FALSE);
    if (ato_str_isnullorempty(p10data)) {
        //printf("/n failed to create p10data. /n");
        goto end;
    } else {
        ato_String *b64 = _base64encode(ctx, p10data);
        *p10 = ato_str_free(*p10);
        ato_str_dup2(p10, b64, TRUE); // We need this as null terminated string
            //dumpstr2file("/home/bruce/tmp/p10.bin", p10data);
            //dumpstr2file("/home/bruce/tmp/p10.b64", b64);
            //dumpstr2file("/home/bruce/tmp/p10f.b64", *p10);
        ato_str_free(b64);
    }

    res = _encrypt_pkey_b64salt(ctx, b64salt, pkey, pk, &xp8, &xp8data, p8);

end:

    ato_str_free(xp8data); /* dummy - not needed here. */
    X509_SIG_free(xp8); /* ditto */

    sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
    X509_REQ_free(req);
    EVP_PKEY_free(pkey);
    OPENSSL_free(p10_buf);
    X509_NAME_free(name);
    ato_str_free(p10data);

    ASN1_OCTET_STRING_free(abn_octet);
    ASN1_IA5STRING_free(abn_ia5string);

    free(encoded_abn);

    return res;
}

ato_String *ato_crypto_sign(ato_Ctx *ctx, ato_Crypto *crypto, ato_PwdKdf *pk, const char *data, size_t len)
{
    STACK_OF(X509) *certs = NULL;
    X509 *int_cert = NULL;
    CMS_ContentInfo *cms = NULL;
    BIO *bio_data = NULL;
    BIO *bio_cms = NULL;
    char *cms_buf = NULL;
    char *err_buf = NULL;
    EVP_PKEY *pkey = NULL;

    ato_String *raw = NULL;
    ato_String *ret = NULL;

    int i = 0;

    err_buf = calloc(200, sizeof(char));
    //ato_String *p10data = NULL;

    bio_data = BIO_new_mem_buf(ATO_REMOVE_CONST(data, void *), len);
    certs = sk_X509_new_null();
    for (i = 1; i < sk_X509_num(crypto->certchain); i++) {
        int_cert = sk_X509_value(crypto->certchain, i);
        len = sk_X509_push(certs, int_cert);
        if (!len) {
            //printf("error stacking x509\n");
            goto end;
        }
    }

    //Get orig private key.
    pkey = _getpkey(ctx, crypto, pk);
    if (pkey == NULL) {
        //password error or something...
        goto end;
     }

    cms = CMS_sign(crypto->usercert, pkey, certs, bio_data, CMS_BINARY | CMS_STREAM | CMS_NOSMIMECAP);
    if (!cms) {
        //printf("error CMS_sign\n");
        ERR_error_string_n(ERR_get_error(), err_buf, 200);
        err_buf[199] = 0;
        //printf("Error:%s.\n", err_buf);
        goto end;
    }
    //res = CMS_final(cms, bio_p10, NULL, NULL); /* not needed if BIO streaming operation done. */
    bio_cms = BIO_new(BIO_s_mem());
    if (!bio_cms) {
        //printf("error constucting output bio\n");
        goto end;
    }
    if (!i2d_CMS_bio_stream(bio_cms, cms, bio_data, CMS_BINARY | CMS_STREAM | CMS_NOSMIMECAP)) {
        //printf("write CMS to buffer failed\n");
        goto end;
    }
    len = BIO_get_mem_data(bio_cms, &cms_buf);
    if(len < 1 ) {
        //printf("BIO_get_mem_data failed.\n");
        goto end;
    }

    ato_str_createconst(&raw, cms_buf, len, FALSE);
    ret = _base64encode(ctx, raw);

 end:

    EVP_PKEY_free(pkey);

    CMS_ContentInfo_free(cms);
    sk_X509_free(certs);
    ato_free(err_buf);
    BIO_vfree(bio_data);
    BIO_vfree(bio_cms);

    ato_str_free(raw);

    return ret;
}

// Generate p10 from an existing certificate, using its x509 representation.
bool ato_crypto_generateP10(ato_Ctx *ctx, int keylen, ato_Crypto *crypto, ato_PwdKdf *pk, ato_String *b64salt, ato_String **p10, ato_String **p8)
{
    bool res = FALSE;
    //BIO *bio_err;  TODO
    X509_REQ *req=NULL;
    EVP_PKEY *pkey=NULL;
    EVP_PKEY *pkey_orig =NULL;
    RSA *rsa = NULL;
    X509_NAME *name=NULL;
    STACK_OF(X509_EXTENSION) *exts = NULL;
    X509_NAME *name_orig=NULL;
    X509_EXTENSION *ext = NULL;
    X509_EXTENSION *ext_cpy = NULL;
    int origNB = 0; //number of extensions
    int i =0;
    int ABNnid=0;
    int len=0;
    char *p10_buf = NULL;

    STACK_OF(X509) *certs = NULL;
    X509 *int_cert = NULL;
    CMS_ContentInfo *cms = NULL;
    BIO *bio_p10 = NULL;
    BIO *bio_cms = NULL;
    char *cms_buf = NULL;
    char *err_buf = NULL;
    X509_SIG *xp8 = NULL;
    ato_String *xp8data = NULL;
    ato_String *p10data = NULL;

    BIGNUM *bn = NULL;
    bn = BN_new();
    BN_set_word(bn, RSA_F4);
    rsa = RSA_new();

    //TODO: entropy. need to seed random number generator
    err_buf = (char *)calloc(200, sizeof(char));

    if ((pkey=EVP_PKEY_new()) == NULL)
        goto end;

    len = RSA_generate_key_ex(rsa, keylen, bn, NULL); // 1.1
    if (len == 0) goto end;

    if (!EVP_PKEY_assign_RSA(pkey,rsa))
        goto end;

    //Get orig private key.
    pkey_orig = _getpkey(ctx, crypto, pk);
    if(pkey_orig == NULL){
        //password error or something...
        goto end;
     }

     req = X509_REQ_new();
     if(req == NULL){
        //error or something...
            //printf("X509_REQ_new() failed\n");
            goto end;
     }
     //Set the new public key
     if(! X509_REQ_set_pubkey(req, pkey)){
       //printf("X509_REQ_set_pubkey  failed\n");
       goto end;
     }
     //Set the data. Get original subject name and duplicate.
     name_orig= X509_get_subject_name(crypto->usercert);
     if(name_orig == NULL){
           //printf("error get orig subject name failed!\n");
           goto end;
     }
     name = X509_NAME_dup(name_orig);
     if(name == NULL){
               //printf("error dup subject name failed!\n");
               goto end;
          }
     if(! X509_REQ_set_subject_name(req, name)){
           //printf("set subject name failed!!\n");
           goto end;
     }

        origNB = X509_get_ext_by_NID(crypto->usercert, ABNnid, -1);
        if(origNB != -1) {
            //add the extensions. OK just ABN.
            ABNnid = OBJ_create("1.2.36.1.333.1", "abn", "abn Extension");
            // printf("ABNnid=%d.\n", ABNnid);
            exts = sk_X509_EXTENSION_new_null();
            //if(exts == NULL) printf("exts NULL\n");

            ext = X509_get_ext(crypto->usercert, origNB);
            if (ext == NULL) { goto end; }
            ext_cpy = X509_EXTENSION_dup(ext);
            if (ext_cpy == NULL) { goto end; }
                if(!sk_X509_EXTENSION_push(exts, ext_cpy) ){
                 //printf("push may have failed for i=%d\n", i);
            }

            if(!X509_REQ_add_extensions(req, exts)){
                goto end;
            }

        }

        // sign with the new key.
        if (!X509_REQ_sign(req, pkey, EVP_sha1())){
            //TODO: error handling
            //printf("sign req failed!\n");
            goto end;
        }

        //DER encode req. i2d_X509_REQ returns the length of p10
        len = i2d_X509_REQ(req, (unsigned char**)(&p10_buf));
        if(len < 1){
                //printf("error DER encoding req\n");
                goto end;
        }

      //create and sign CMS
        bio_p10 = BIO_new_mem_buf((void *)p10_buf, len);
        certs = sk_X509_new_null();
        for (i = 1; i < sk_X509_num(crypto->certchain); i++) {
            int_cert =sk_X509_value(crypto->certchain, i);
            len = sk_X509_push(certs, int_cert);
            if(!len){
            //printf("error stacking x509\n");
            goto end;
            }
        }
        //cms = CMS_encrypt(certs, bio_p10, EVP_rc4(),CMS_BINARY|CMS_STREAM);
        cms = CMS_sign(crypto->usercert, pkey_orig, certs, bio_p10, CMS_BINARY|CMS_STREAM|CMS_NOSMIMECAP);
        if (!cms){
            //printf("error CMS_sign\n");
            ERR_error_string_n(ERR_get_error(), err_buf, 200);
            err_buf[199]=0;
            //printf("Error:%s.\n", err_buf);
            goto end;
        }
        //res = CMS_final(cms, bio_p10, NULL, NULL); /* not needed if BIO streaming operation done. */
        bio_cms = BIO_new(BIO_s_mem());
        if (!bio_cms){
            //printf("error constucting output bio\n");
            goto end;
        }
        if (!i2d_CMS_bio_stream(bio_cms, cms, bio_p10, CMS_BINARY|CMS_STREAM|CMS_NOSMIMECAP)){
            //printf("write CMS to buffer failed\n");
            goto end;
        }
        len = BIO_get_mem_data(bio_cms, &cms_buf);
        if(len <1){
            //printf("BIO_get_mem_data failed.\n");
            goto end;
        }
        ato_str_createconst(&p10data, cms_buf, len, FALSE);
        if(ato_str_isnullorempty(p10data)){
            //printf("/n failed to create p10data. /n");
            goto end;
        } else {
            ato_String *b64 = _base64encode(ctx, p10data);
            *p10 = ato_str_free(*p10);
            ato_str_dup2(p10, b64, TRUE); // We need this as null terminated string
            ato_str_free(b64);
        }

        res = _encrypt_pkey_b64salt(ctx, b64salt, pkey, pk, &xp8, &xp8data, p8);

        ato_str_free(xp8data); /* dummy - not needed here. */
        X509_SIG_free(xp8); /* ditto */

        end:
    //clean up!!!!
    if (exts != NULL) {
        sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
    }
            X509_REQ_free(req);
            EVP_PKEY_free(pkey);
            EVP_PKEY_free(pkey_orig);
    CMS_ContentInfo_free(cms);
    sk_X509_free(certs);
    OPENSSL_free(p10_buf);
    ato_free(err_buf);
    BIO_vfree(bio_p10);
    BIO_vfree(bio_cms);
    X509_NAME_free(name);
    ato_str_free(p10data);

    BN_clear_free(bn); // 1.1

            //RSA_free(rsa);   //THIS seems to cause problems - TODO investigate. maybe data has been handed off in the key operations?
            // the sample request code just sets rsa to NULL (which seems bad)

    return res;
}

char *ato_crypto_kdf(ato_Ctx *ctx, char **dpwd, const char *pwd, const ato_String *bsalt, size_t iterations, size_t octets, int prfid) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_crypto_kdf";
    int res = 0, err = 0;
    const EVP_MD *digest = NULL;
    char *derivedpwd = NULL;
    ato_String *sdpwd = NULL;
    ATO_ASSERT_ISNOTALLOCATED(dpwd);
    switch (prfid) {
        case 1: digest = EVP_sha1(); break;
        case 224: digest = EVP_sha224(); break;
        case 256: digest = EVP_sha256(); break;
        case 384: digest = EVP_sha384(); break;
        case 512: digest = EVP_sha512(); break;
        default: {
            Throw ATO_CTX_NEWERR(ctx, ATO_ERR_CRYPTO, "Invalid prfid.");
        }
    }
    derivedpwd = calloc(octets + 1, sizeof(char));
    res = PKCS5_PBKDF2_HMAC(pwd, strlen(pwd), (const unsigned char *)ato_str_value(bsalt), ato_str_len(bsalt), iterations, digest, octets, (unsigned char *)derivedpwd);
    if (res == 0) {
        free(derivedpwd);
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_CRYPTO, "Failed to generate derived passphrase.");
    }
    err = ato_base64encode_array(ctx, &sdpwd, derivedpwd, octets);
    memset(derivedpwd, 0, octets);
    free(derivedpwd);
    if (err != 0) {
        ato_str_init(sdpwd, 0);
        ato_str_free(sdpwd);
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_CRYPTO, "Failed to base64 derived passphrase.");
    }
    *dpwd = ato_str_cstr(sdpwd);
    return *dpwd;
}
