/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
Note that this module does initialise the exception mechanism for new context objects.
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atoexcept.h"
#include "atostr.h"
#include "atoutil.h"
#include "atolst.h"
#include "atoerr.h"
#include "atolog.h"
#include "atoarg.h"

#include "atoctx.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_CTX;
static unsigned long _moduleid = ATO_BASE_MODULEID_CTX;

static int _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
struct _ato_Ctx {
        struct exception_context ec[1];
        char *id;
        ato_Log *log;
        ato_Err *err;
        ato_List *cache;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
        _loglevel = level;
}
/*********************************************************************************/
int ato__ctx_init(void)
{
        static bool invoked = FALSE;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        return ATO_ERR_OK;
}

void ato__ctx_deinit(void)
{
        static bool invoked = FALSE;
        if (invoked) return;
        invoked = TRUE;
}

/*********************************************************************************
static void _setlog(ato_Ctx *ctx) {
    const char *path = ato_arg_value("logpath");
    const char *indent = ato_arg_value("logindent");
    if (path == NULL) { path = "messages.log"; }
        ato__log_create(
                &(ctx->log),
                ctx->id,
                (size_t)ato_value_int(indent, 2),
                path
                );
        assert(ctx->log != NULL);
} */

/*********************************************************************************/
void ato_ctx_create(ato_Ctx **obj, const char *id) {
        ato_Ctx *ctx = NULL;
        ATO_ASSERT_ISNOTALLOCATED(obj);

        ctx = calloc(1, sizeof(ato_Ctx));
        assert(ctx != NULL);

        init_exception_context(ctx->ec);
        ato_lst_create(&(ctx->cache));

        if (id == NULL) {
            ctx->id = calloc(50, sizeof(char));
            ato_new_pseudo_guid(ctx, ctx->id);
        } else {
            ctx->id = ato_strdup(id, 0);
        }

        //_setlog(ctx);

        *obj = ctx;
}

static void _ctx_free(ato_Ctx *ctx)
{
        if (ctx == NULL) return;
        ato__log_free(ctx->log);
        ato_ctx_err_clear(ctx);
}

void ato_ctx_free(ato_Ctx *ctx)
{
        if (ctx == NULL) return;
        _ctx_free(ctx);
        ato_lst_free(ctx->cache);
        ctx->id = ato_free(ctx->id);
        free(ctx);
}

ato_Log *ato_ctx_log_set(ato_Ctx *ctx, const char *id, const char *logindent, const char *logpath, const char *mode) {
        assert(ctx != NULL);
        if (logpath == NULL) { logpath = ato_arg_value("logpath"); }
        if (logpath == NULL) { logpath = "messages.log"; }
        if (logindent == NULL) { logindent = ato_arg_value("logindent"); }
        if (logindent == NULL) { logindent = "2"; }
        ctx->log = ato__log_free(ctx->log);
        ato__log_create(&(ctx->log), id != NULL ? id : ctx->id, (size_t)ato_value_int(logindent, 2), logpath, mode);
        return ctx->log;
}

const char *ato_ctx_id(ato_Ctx *ctx)
{
        assert(ctx != NULL);
        return ctx->id;
}

ato_Log *ato_ctx_log(ato_Ctx *ctx)
{
        assert(ctx != NULL);
        return ctx->log;
}

ato_Err *ato_ctx_err(ato_Ctx *ctx)
{
        assert(ctx != NULL);
        return ctx->err;
}

int ato_ctx_err_clear(ato_Ctx *ctx)
{
        assert(ctx != NULL);
        ato__err_free(ctx->err);
        ctx->err = NULL;
        return ATO_ERR_OK;
}

struct exception_context *ato__ctx_ec(ato_Ctx *ctx)
{
        assert(ctx != NULL);
        return ctx->ec;
}

int ato_ctx_err_new(ato_Ctx *ctx, const char *library, const char *module, const char *function, int line, int code, ato_eErrSeverity severity, void *customobj, const char *msg)
{
        ato_Err *err = NULL;
        assert(ctx != NULL);
        ato__err_create(&err, library, module, function, line, code, severity, msg, customobj, ato_ctx_err(ctx));
        assert(err != NULL);
        ctx->err = err;
        return code;
}

int ato_ctx_err_vnew(ato_Ctx *ctx, const char *library, const char *module, const char *function, int line, int code, ato_eErrSeverity severity, void *customobj, size_t size, const char *format, ...)
{
        int errcode = 0;
        assert(format != NULL);
        if (size > 0) {
                char *msg = NULL;
                size_t nsize = size;
                va_list args;
                nsize += strlen(format);
                msg = calloc(nsize+1, sizeof(char));
                assert(msg != NULL);
                va_start(args, format);
                vsnprintf(msg, nsize, format, args);
                va_end(args);
                msg[nsize] = '\0';
                errcode = ato_ctx_err_new(ctx, library, module, function, line, code, severity, customobj, msg);
                free(msg);
        } else {
                errcode = ato_ctx_err_new(ctx, library, module, function, line, code, severity, customobj, format);
        }
        return errcode;
}
/*********************************************************************************/
void *ato_ctx_cache(ato_Ctx *ctx, const char *key) {
        void *value = NULL;
        ato_ListNode *node = NULL;
        assert(ctx != NULL);
        node = ato_lst_find(ctx->cache, key);
        if (node) {
                value = ato_lst_value(node);
        }
        return value;
}
const void *ato_ctx_cachec(ato_Ctx *ctx, const char *key) {
        return ato_lst_findvalueconst(ctx->cache, key);
}

void *ato_ctx_cache_set(ato_Ctx *ctx, const char *key, void *value) {
        ato_ListNode *node = NULL;
        assert(ctx != NULL);
        node = ato_lst_find(ctx->cache, key);
        if (value != NULL) {
                if (node) {
                        assert(ato_lst_value(node) == NULL);
                } else {
                        node = ato_lst_add(ctx->cache, key, value);
                }
        }
        if (node)
                ato_lst_setvalue(node, value);
        return value;
}
const void *ato_ctx_cachec_set(ato_Ctx *ctx, const char *key, const void *value) {
        ato_ListNode *node = NULL;
//                        printf("Adding %s=%s\n", key, (const char *)value);
        assert(ctx != NULL);
        node = ato_lst_find(ctx->cache, key);
        if (value != NULL) {
                if (node) {
                        assert(ato_lst_value(node) == NULL);
                } else {
//                        printf("Added %s=%s\n", key, (const char *)value);
                        node = ato_lst_addconst(ctx->cache, key, value);
                }
        }
        if (node)
                ato_lst_setvalueconst(node, value);
        return value;
}
