/*
This file is for documentation purposes only.
*/

//-----------------------------------------------------------

/*! @addtogroup atobase_overview_error

@section atobase_error_overview Overview

If a method raises an error (at any severity level), the method and containing methods, will return immediately back to the
application code.

API methods which can generate errors have the following characteristics:
- they return an int which represents an error code
- they accept a thread specific context object as the first argument, to which error details can be attached

All other methods:
- are property methods which return non-int values
- will always return some value or a default (such as NULL for char * methods)

Either of these types of methods can assert if invalid arguments are used or an out of memory condition occurs.

Where an error code is returned that is not ATO_ERR_OK, the @ref ato_Ctx object
has an @ref ato_Err object. This can be used to retrieve details on the
last error that occurred including any nested errors. There is a separate Error API to retrieve details
of the error.

The errorcode associated with an error object, and returned by many of the API methods, is an int.
Given the nature of C, it is not possible to guarantee non-overlapping codes, so each errorcode returned
should be interpreted in relation to the method called. For convenience, each self-contained API contains
a list of non-overlapping errorcodes as "#defines". The @ref ato_Err object does contain
the library and module that raised the error, which provides more information if required.

All errors have an associated severity level - see @ref ato_eErrSeverity.
While the catagorisation of some errors with a given severity level
is somewhat arbitrary, the table below gives a rational for the catagorisation.

Currently all errors are given a severity level of "ATO_ESEVERITY_WARN" (the default).


@section atobase_error_severity Error severity

@htmlonly
<table border>
    <tr><td><b>Error Severity</b></td><td><b>Implication</b></td><td><b>Action</b></td><td><b>Example causes</b></td></tr>
    <tr>
        <td>ATO_ESEVERITY_FATAL</td>
        <td>Panic</td>
        <td>Halt (sometimes assertion)</td>
        <td>Coding errors; environment errors: out of memory, disc corruption, missing mandatory configuration data</td>
    </tr>
    <tr>
        <td>ATO_ESEVERITY_ERR</td>
        <td>Affects all processing - cannot process any messages/transactions until fixed</td>
        <td>Warn user and go into RETRY until condition auto-corrects. If retry takes too long (or user cancels), escalate to FATAL</td>
        <td>Infrastructure errors: cannot contact server</td>
    </tr>
    <tr>
        <td>ATO_ESEVERITY_WARN</td>
        <td>Current msg/transaction cannot be processed. Others may be fine</td>
        <td>Warn user and go into RETRY until condition auto-corrects. If retry takes too long (or user cancels), escalate to FATAL</td>
        <td>Business rule violated for a given message or transaction: missing or bad data; incorrect user input such as bad password</td>
    </tr>
</table>
@endhtmlonly


@section atobase_error_iterating Iterating the error "stack"

It is recommended that dependent SDKs wrap errors returned by ATOBASE with their own SDK specific
error codes. This implies that ATOBASE errors are always inner/nested errors. In addition ATOBASE errors may
wrap other ATOBASE errors. For ease of use it is also recommended that dependant SKDs have non-overlapping errcodes.

ATOBASE errcodes are of the form ATO_ERR_... where there is an associated an associated error object attached to
he current context (contexts must be thread specific).

In addition, any error object can contain a custom error object which is interpreted according to an additional error
API that knows how to interpret the custom error object. Currently the only custom error object is related to reporting
network SOAP errors. This is the case if the errcode, of the containing general error object, is set to ATO_ERR_NET_SOAP.

While the default logger will automatically log errors (the application can override the default logger),
it may be appropriate to interpret the errors further. The generic way of doing this is to iterate "down" through
the inner errors and for each error check appropriate details and interpret any custom error.

@code
    ato_Err *err = ato_ctx_err(ctx);
    int errcode = ato_err_code(err);

    for (; err; err = ato_err_inner(err)) {
        errcode = ato_err_code(err);
        printf("%s\n", ato_err_msg(err));
        if (errcode == ATO_ERR_NET_SOAP) {
            ato_ErrSoap *soaperr = ato_err_custom(err);
            if (soaperr && ato_errsoap_issoaperr(soaperr)) {
                ato_ErrSoapCode *code = NULL;
                printf("  node=[%s]\n", ato_errsoap_node(soaperr));
                printf("  reason=[%s]\n", ato_errsoap_reason(soaperr));
                printf("  detail=[%s]\n", ato_errsoap_detail(soaperr));
                for (code = ato_errsoap_code_firstv(soaperr); code; code = ato_errsoap_code_nextv(soaperr)) {
                    printf("  code value=[%s] ns=[%s]\n", ato_errsoap_code_value(code), ato_errsoap_code_ns(code));
                }
            } else {
                printf("  SOAP error details not available\n");
            }
        }
    }
@endcode

*/
