/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atostr.h"
#include "atolst.h"
#include "atoutil.h"
#include "atolog.h"
#include "atoerr.h"
#include "atoctx.h"
#include "atoiter.h"
#include "atoerrsoap.h"

#include "atoerrh.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_ERRFN;
static unsigned long _moduleid = ATO_BASE_MODULEID_ERRFN;

static int _loglevel = ATO_LOG_WARN;

typedef struct _ato_errfndata _errfndata;
struct _ato_errfndata {
    ato_errfunction *errfunction;
};

ato_List *_errfnlst;

/*********************************************************************************/
static bool _default_errfn(ato_Ctx *ctx, ato_Err *err, ato_Log *log);
/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
int ato_errh_sethandler(ato_errfunction errfunction, const char *library)
{
    int errcode = ATO_ERR_OK;
    _errfndata *efd = NULL;
    efd = calloc(1, sizeof(_errfndata));
    efd->errfunction = errfunction;
    if (ato_lst_add(_errfnlst, library, efd) == NULL)
        errcode = ATO_ERR_DUPLICATE_ERRHANDLER;
    return errcode;
}

static bool _errh_handler_invoke(ato_Ctx *ctx, ato_Err *err)
{
    ato_Log *log = NULL;
    ato_ListNode *node = ato_lst_find(_errfnlst, ato_err_library(err));
    _errfndata *efd = ato_lst_value(node);
    ato_errfunction *errfunction = _default_errfn;
    if (ctx != NULL) log = ato_ctx_log(ctx);
    if (efd != NULL)
        errfunction = efd->errfunction;
    ato_err_setisprocessed(err, errfunction(ctx, err, log));
    return ato_err_isprocessed(err);
}

static bool _errh_handler(ato_Ctx *ctx, ato_Err *err)
{
    if (err != NULL && !ato_err_isprocessed(err)) {
        _errh_handler(ctx, ato_err_inner(err));
        return _errh_handler_invoke(ctx, err);
    }
    return FALSE;
}

bool ato_errh_handler(ato_Ctx *ctx, int errcode)
{
    bool processed = FALSE;

    ato_Err *err = ato_ctx_err(ctx);
    ATO_IGNORE(errcode);
    if (err == NULL)
        assert(err == NULL && errcode == 0);
    else if (err != NULL)
        assert(ato_err_code(err) == errcode);
    if (err != NULL)
        processed = _errh_handler(ctx, err);
    return processed;
}

void ato__errh_free(ato_Err *err)
{
    _errh_handler_invoke(NULL, err);
}

/*********************************************************************************/

static bool _default_errfn(ato_Ctx *ctx, ato_Err *err, ato_Log *log)
{
    const char *function = "_default_errfn";
    bool processed = TRUE;
    bool deleted = ato_err_isdeleted(err);
    int errcode = ato_err_code(err);
    if (deleted) {
        if (errcode == ATO_ERR_NET_SOAP) {
            ato_errsoap_free(ato_err_custom(err));
        }
    } else {
        const char *msg = ato_err_msg(err);
        if (msg == NULL)
            msg = "No error text";
        processed = ATO_LOG_MSGV(log, ATO_LOG_ERR, "%s.%s.%s: %d (%s: %s): %s",
            ato_err_library(err), ato_err_module(err),  ato_err_function(err),
            errcode, ato_err_codename(err), ato_err_codemsg(err), msg);
        if (errcode == ATO_ERR_NET_SOAP) {
            ato_ErrSoap *soaperr = ato_err_custom(err);
            if (soaperr) {
                ato_ErrSoapCode *code = NULL;
                const char *s = NULL;
                ATO_LOG_MSGV(log, ATO_LOG_ERR, "%s", "SOAP response error details");
                if ((s = ato_errsoap_reason(soaperr)) != NULL)
                    ATO_LOG_MSGV(log, ATO_LOG_ERR, "  reason=[%s]", s);
                if ((s = ato_errsoap_node(soaperr)) != NULL)
                    ATO_LOG_MSGV(log, ATO_LOG_ERR, "  node=[%s]", s);
                if ((s = ato_errsoap_detail(soaperr)) != NULL)
                    ATO_LOG_MSGV(log, ATO_LOG_ERR, "  detail=[%s]", s);
                for (code = ato_errsoap_code_firstv(soaperr); code; code = ato_errsoap_code_nextv(soaperr)) {
                    const char *c = NULL, *v = NULL;
                    if ((c = ato_errsoap_code_value(code)) == NULL)
                        c = "";
                    if ((v = ato_errsoap_code_ns(code)) == NULL)
                        v = "";
                    ATO_LOG_MSGV(log, ATO_LOG_ERR, "  code: value=[%s]; ns=[%s]", c, v);
                }
            }
        }
    }
    ATO_IGNORE(ctx);
    return processed;
}
/*********************************************************************************/

int ato__errh_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);

    ato_lst_create(&_errfnlst);
    return ato_errh_sethandler(_default_errfn, ATO_BASE_LIBRARY);
}
void ato__errh_deinit(void)
{
    ato_ListNode *curr = NULL;
    if (_errfnlst) {
        while ((curr = ato_lst_next(_errfnlst, curr)) != NULL) {
            free(ato_lst_value(curr));
        }
        ato_lst_free(_errfnlst);
        _errfnlst = NULL;
    }
}
/*********************************************************************************/
