/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>

#if _WIN32
#include <Windows.h>
#include <direct.h>
#else
#include <sys/stat.h>
#include <sys/types.h>
#endif

#include "atointernal.h"
#include "atotypes.h"
#include "atostr.h"
#include "atoutil.h"
//#include "atoarg.h"

#include "atolog.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_LOG;
static unsigned long _moduleid = ATO_BASE_MODULEID_LOG;

static int _loglevel = ATO_LOG_WARN;

static ato_logfunction *_logfunction;

/*********************************************************************************/
struct _ato_Log {
    char *id;
    char outputtype;
    char *filepath;
    char *ufilepath;
    //char *override_path;
    size_t indentsize;
    size_t indent;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
static void _setupath(ato_Log *log)
{
    char *ext = NULL;

    log->ufilepath = ato_strdup(log->filepath, strlen(log->id) + 5);
    ext = strrchr(log->ufilepath, (int)'.');
    if (ext != NULL) *ext = '\0';
    if (strlen(log->id) > 0) {
        strcat(strcat(log->ufilepath, "-"), log->id);
    }
    if (ext != NULL) {
        ext = strrchr(log->filepath, (int)'.');
        assert(ext);
        strcat(log->ufilepath, ext);
    }
}

static void make_path(const char *p) {
    char path[FILENAME_MAX] = {0};
    char *s = NULL;
    strcpy(path, p);

    while ((s = strchr(path, '\\')) != NULL) { *s = '/'; } // canonicalize
    s = path;
    while (*s != '\0' && *s == '/') { s++; } // skip leading /
    s = strchr(s, '/');
    while (s != NULL) {

        *s = '\0';

#if _WIN32
        _mkdir(path);
#else
        mkdir(path, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH); // TODO change from 777 perms
#endif
        *s = '/';
        s = strchr(s+1, '/');
    }
}

/*********************************************************************************/
void ato__log_create(ato_Log **obj, const char *id, size_t indentsize, const char *logpath, const char *mode)
{
    ato_Log *log = NULL;
    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = log = calloc(1, sizeof(ato_Log));
    assert(log != NULL);

    if (id == NULL) id = "";
    log->id = ato_strdup(id, 0);
    if (ato_isnullorempty(logpath)) {
        log->outputtype = ATO_LOG_OUTPUTTYPE_NONE;
    } else if (ato_streq(logpath, "none")) {
        log->outputtype = ATO_LOG_OUTPUTTYPE_NONE;
    } else if (ato_streq(logpath, "stdout")) {
        log->outputtype = ATO_LOG_OUTPUTTYPE_STDOUT;
    } else if (ato_streq(logpath, "stderr")) {
        log->outputtype = ATO_LOG_OUTPUTTYPE_STDERR;
    } else {
        log->outputtype = ATO_LOG_OUTPUTTYPE_PATH;
    }
    log->indentsize = indentsize;
    if (log->outputtype == ATO_LOG_OUTPUTTYPE_PATH) {
        log->filepath = ato_strdup(logpath, 0);
        _setupath(log);
        if (mode != NULL) {
            make_path(logpath);
            FILE *fp = fopen(log->ufilepath, mode);
            if (!fp) {
                *obj = log = ato__log_free(log);
            }
            fclose(fp);
        }
    }
}

void *ato__log_free(ato_Log *log)
{
    if (log == NULL) return NULL;

    log->id = ato_free(log->id);
    log->filepath = ato_free(log->filepath);
    log->ufilepath = ato_free(log->ufilepath);
    //log->override_path = ato_free(log->override_path);
    free(log);
    return NULL;
}

/*********************************************************************************/
void ato_log_setlogfn(ato_logfunction logfunction)
{
    _logfunction = logfunction;
}

/*
void ato_log_path_override_set(ato_Log *log, const char *path) {
    if (log == NULL) return;
    log->override_path = ato_free(log->override_path);
    log->override_path = ato_strdup(path, 0);
}
const char *ato_log_path_override(ato_Log *log) {
    return log->override_path;
}*/

char ato_log_outputtype(ato_Log *log) { assert(log != NULL); return log->outputtype; }
const char *ato_log_path(ato_Log *log) { assert(log != NULL); return log->filepath; }
const char *ato_log_upath(ato_Log *log) { assert(log != NULL); return log->ufilepath; }
size_t ato_log_indentsize(ato_Log *log) { assert(log != NULL); return log->indentsize; }
size_t ato_log_indentlevel(ato_Log *log) { assert(log != NULL); return log->indent; }

static bool _ato_log_vmsgstart(ato_Log *log, ato_eLoglevel loglevel, const char *library, const char *module, const char *function, const char *format, va_list args);
static bool _ato_log_vmsgend(ato_Log *log, ato_eLoglevel loglevel, const char *library, const char *module, const char *function, bool doindentonly, const char *format, va_list args);
static bool _ato_log_vmsg(ato_Log *log, ato_eLoglevel loglevel, const char *library, const char *module, const char *function, const char *format, va_list args);

bool ato_log_msg(ato_Log *log, ato_eLoglevel target, ato_eLoglevel current, const char *library, const char *module, const char *function, const char *format, ...)
{
    bool processed = FALSE;
    if (log && current >= target) {
        va_list args;
        va_start(args, format);
        processed = _ato_log_vmsg(log, target, library, module, function, format, args);
        va_end(args);
    }
    return processed;
}
bool ato_log_start(ato_Log *log, ato_eLoglevel target, ato_eLoglevel current, const char *library, const char *module, const char *function, const char *format, ...)
{
    bool processed = FALSE;
    if (log && current >= target) {
        va_list args;
        va_start(args, format);
        processed = _ato_log_vmsgstart(log, target, library, module, function, format, args);
        va_end(args);
    }
    return processed;
}
bool ato_log_end(ato_Log *log, ato_eLoglevel target, ato_eLoglevel current, const char *library, const char *module, const char *function, bool doindentonly, const char *format, ...)
{
    bool processed = FALSE;
    if (log && current >= target) {
        va_list args;
        va_start(args, format);
        processed = _ato_log_vmsgend(log, target, library, module, function, doindentonly, format, args);
        va_end(args);
    }
    return processed;
}

static bool _ato_log_vmsgstart(ato_Log *log, ato_eLoglevel loglevel, const char *library, const char *module, const char *function, const char *format, va_list args)
{
    bool processed = FALSE;
    assert(format != NULL);
    assert(log != NULL);

    processed = _logfunction(log, loglevel, ATO_LOGSTATE_OPEN, library, module, function, format, args);
    log->indent++;
    return processed;
}
static bool _ato_log_vmsgend(ato_Log *log, ato_eLoglevel loglevel, const char *library, const char *module, const char *function, bool doindentonly, const char *format, va_list args)
{
    assert(log != NULL);
    if (log->indent > 0) log->indent--;
    if (!doindentonly)
        return _logfunction(log, loglevel, ATO_LOGSTATE_CLOSE, library, module, function, format, args);
    return FALSE;
}
static bool _ato_log_vmsg(ato_Log *log, ato_eLoglevel loglevel, const char *library, const char *module, const char *function, const char *format, va_list args)
{
    return _logfunction(log, loglevel, ATO_LOGSTATE_MSG, library, module, function, format, args);
}

ato_eLoglevel ato_log_name2level(const char *c)
{
    ato_eLoglevel def = ATO_LOG_WARN;
    if (c == NULL || c[0] == '\0')
        return def;
    if (c[0] == ATO_LOG_OFF_NAME[0]) return ATO_LOG_OFF;
    if (c[0] == ATO_LOG_FATAL_NAME[0]) return ATO_LOG_FATAL;
    if (c[0] == ATO_LOG_ERR_NAME[0]) return ATO_LOG_ERR;
    if (c[0] == ATO_LOG_WARN_NAME[0]) return ATO_LOG_WARN;
    if (c[0] == ATO_LOG_INFO_NAME[0]) return ATO_LOG_INFO;
    if (c[0] == ATO_LOG_DEBUG_NAME[0]) return ATO_LOG_DEBUG;
    if (c[0] == ATO_LOG_TRACE_NAME[0]) return ATO_LOG_TRACE;
    if (c[0] == ATO_LOG_ALL_NAME[0]) return ATO_LOG_ALL;
    return def;
}

const char *ato_log_level2name(ato_eLoglevel level)
{
    switch (level) {
        case ATO_LOG_OFF:   return ATO_LOG_OFF_NAME;
        case ATO_LOG_FATAL: return ATO_LOG_FATAL_NAME;
        case ATO_LOG_ERR:   return ATO_LOG_ERR_NAME;
        case ATO_LOG_WARN:  return ATO_LOG_WARN_NAME;
        case ATO_LOG_INFO:  return ATO_LOG_INFO_NAME;
        case ATO_LOG_DEBUG: return ATO_LOG_DEBUG_NAME;
        case ATO_LOG_TRACE: return ATO_LOG_TRACE_NAME;
        case ATO_LOG_ALL:   return ATO_LOG_ALL_NAME;
    }
    return ATO_LOG_WARN_NAME;
}

/*********************************************************************************/

#define _TIMESTAMP_LEN 50
static const char *_timestamp(char *buffer)
{
    struct tm *time_now;
    time_t secs_now;
    time(&secs_now);
    time_now = localtime(&secs_now);
    strftime(buffer, _TIMESTAMP_LEN, "%Y%m%d-%H%M%S", time_now);
    return buffer;
}

static bool _print(ato_Log *log, ato_eLoglevel loglevel, ato_eLogState logstate, const char *library, const char *module, const char *function, const char *format, va_list args)
{
    FILE *fp = NULL;

    if (log == NULL)
        return FALSE;

    switch (ato_log_outputtype(log)) {
        case ATO_LOG_OUTPUTTYPE_STDOUT:
            fp = stdout;
            break;
        case ATO_LOG_OUTPUTTYPE_STDERR:
            fp = stderr;
            break;
        case ATO_LOG_OUTPUTTYPE_PATH:
            {
            //const char *filepath = log->override_path ? log->override_path : ato_log_upath(log);
            const char *filepath = ato_log_upath(log);
            if (filepath != NULL && (filepath)[0] != '\0')
                fp = fopen(filepath, "at");
            }
            break;
        case ATO_LOG_OUTPUTTYPE_NONE:
        default:
            return FALSE;
    }

    if (fp == NULL) { return FALSE; }

    {
    char buffer[_TIMESTAMP_LEN+1];
    fprintf(fp, "%s: ", _timestamp(buffer));
    }

    {
    size_t len = 0;
    size_t indentlen = ato_log_indentsize(log) * ato_log_indentlevel(log);
    for (len = 0; len < indentlen; len++)
        fprintf(fp, " ");
    }

    if (logstate == ATO_LOGSTATE_OPEN)
        fprintf(fp, "+");
    else if (logstate == ATO_LOGSTATE_CLOSE)
        fprintf(fp, "-");
    else
        fprintf(fp, ".");

    fprintf(fp, "%c ", (char)toupper((int)(ato_log_level2name(loglevel)[0])));

    if (logstate == ATO_LOGSTATE_OPEN || logstate == ATO_LOGSTATE_CLOSE)
        fprintf(fp, "%s.%s.%s ", library, module, function);

    fprintf(fp, "[");
    vfprintf(fp, format, args);
    fprintf(fp, "]");

    fprintf(fp, "\n");

    if (ato_log_outputtype(log) == ATO_LOG_OUTPUTTYPE_PATH && fp != NULL)
        fclose(fp);

    return TRUE;
}

/*********************************************************************************/

int ato__log_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    _logfunction = _print;
    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}
void ato__log_deinit(void)
{
}
/*********************************************************************************/
