#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atostr.h"
#include "atoutil.h"

#include "atopbe.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_PBE;
static unsigned long _moduleid = ATO_BASE_MODULEID_PBE;

/*********************************************************************************/
struct _ato_Pbe {
    char *str; // e.g. For can be either sha256/3des or 256/3 (canonical)
    int hmac;
    int enc;
    size_t iterations;
};

/*********************************************************************************/
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

/*********************************************************************************/

static const int _ATO_PBE_ITERATIONS_LEGACY = 1024;
static const int _ATO_PBE_ITERATIONS = 1024;

static bool _isPbe2(const ato_Pbe *pbe) {
    return (pbe->str != NULL);
}

static const char *_calculate_str(ato_Pbe *pbe) {
    char *enc = NULL;
    pbe->str = ato_free(pbe->str);
    if (pbe->hmac > 0) {
        pbe->str = calloc(30, sizeof(char));
        sprintf(pbe->str, "%d", pbe->hmac);
        if (pbe->enc > 0) {
            enc = pbe->str + strlen(pbe->str);
            sprintf(enc, "/%d", pbe->enc);
        }
    }
    return pbe->str;
}

static void _hmac_set(ato_Pbe *pbe, const char *shmac) {
    int hmac = 0;
    if (ato_isnullorempty(shmac)) { return; }
    if (ato_strstartswithi(shmac, "sha1") == 0 || ato_strstartswithi(shmac, "1") == 0) { hmac = 1; }
    else if (ato_strstartswithi(shmac, "sha224") == 0 || ato_strstartswithi(shmac, "224") == 0) { hmac = 224; }
    else if (ato_strstartswithi(shmac, "sha256") == 0 || ato_strstartswithi(shmac, "256") == 0) { hmac = 256; }
    else if (ato_strstartswithi(shmac, "sha384") == 0 || ato_strstartswithi(shmac, "384") == 0) { hmac = 384; }
    else if (ato_strstartswithi(shmac, "sha512") == 0 || ato_strstartswithi(shmac, "512") == 0) { hmac = 512; }
    pbe->hmac = hmac;
}

static void _enc_set(ato_Pbe *pbe, const char *senc) {
    int enc = 0;
    if (ato_isnullorempty(senc)) { return; }
    if (ato_strstartswithi(senc, "3des") == 0 || ato_strstartswithi(senc, "3") == 0) { enc = 3; }
    else if (ato_strstartswithi(senc, "aes128") == 0 || ato_strstartswithi(senc, "128") == 0) { enc = 128; }
    else if (ato_strstartswithi(senc, "aes192") == 0 || ato_strstartswithi(senc, "192") == 0) { enc = 192; }
    else if (ato_strstartswithi(senc, "aes256")== 0 || ato_strstartswithi(senc, "256") == 0) { enc = 256; }
    pbe->enc = enc;
}

static bool _set(ato_Pbe *pbe, const char *pbestr) {
    if (ato_streq(pbe->str, pbestr)) { return FALSE; }
    if (ato_isnullorempty(pbestr)) {
        pbe->str = ato_free(pbe->str);
        pbe->hmac = pbe->enc = 0;
    } else {
        const char *hmac = pbestr, *enc = NULL, *sep = NULL;
        sep = strchr(pbestr, '/');
        if (sep != NULL) { enc = sep+1; }
        _hmac_set(pbe, hmac);
        _enc_set(pbe, enc);
        _calculate_str(pbe);
    }
    pbe->iterations = _ATO_PBE_ITERATIONS_LEGACY;
    if (_isPbe2(pbe)) {
        pbe->iterations = _ATO_PBE_ITERATIONS;
    }
    return TRUE;
}

static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}
/*********************************************************************************/
int ato__pbe_init(void) {
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__pbe_deinit(void) {
    static bool invoked = FALSE;
    if (invoked) return;
    invoked = TRUE;
}

/*********************************************************************************/
ato_Pbe *ato_pbe_create(ato_Ctx *ctx, ato_Pbe **obj, const char *pbestr) {
    ato_Pbe *pbe;
    ATO_IGNORE(ctx);
    ATO_ASSERT_ISNOTALLOCATED(obj);
    *obj = pbe = calloc(1, sizeof(ato_Pbe));
    assert(pbe != NULL);
    pbe->iterations = _ATO_PBE_ITERATIONS_LEGACY; // default
    _set(pbe, pbestr);
    return pbe;
}

void *ato_pbe_free(ato_Pbe *pbe) {
    if (pbe == NULL) return NULL;
    pbe->str = ato_free(pbe->str);
    free(pbe);
    return NULL;
}

/*********************************************************************************/
int ato_pbe_hmac(const ato_Pbe *pbe) { assert(pbe != NULL); return pbe->hmac; }
int ato_pbe_enc(const ato_Pbe *pbe) { assert(pbe != NULL); return pbe->enc; }
int ato_pbe_iter(const ato_Pbe *pbe) { assert(pbe != NULL); return pbe->iterations; }
const char *ato_pbe_str(const ato_Pbe *pbe) { assert(pbe != NULL); return pbe->str; }

bool ato_pbe_set(ato_Pbe *pbe, const char *pbestr) { assert(pbe != NULL); return _set(pbe, pbestr); }
bool ato_pbe_isPbe2(const ato_Pbe *pbe) { return _isPbe2(pbe); }
