/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"

#include "atosi.h"

#include "atostr.h"
#include "atoutil.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_SI;
static unsigned long _moduleid = ATO_BASE_MODULEID_SI;

static int _loglevel = ATO_LOG_WARN;

typedef struct _ato_SoftwareInfo ato_SoftwareInfo;
struct _ato_SoftwareInfo {
    char *organisation;
    char *product;
    char *version;
    char *timestamp;
    char *source;
    char *hash;
    char *info;
};

static ato_SoftwareInfo g_softwareinfo = { NULL, NULL, NULL, NULL, NULL, NULL, NULL };

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level) {
    _loglevel = level;
}

/*********************************************************************************/
static void *_ato_si_free(ato_SoftwareInfo *si) {
    if (si == NULL) { return NULL; }
    ato_free(si->organisation);
    ato_free(si->product);
    ato_free(si->version);
    ato_free(si->timestamp);
    ato_free(si->source);
    ato_free(si->hash);
    //free(si);
    return NULL;
}

/*********************************************************************************/
int ato__si_init(void) {
    static bool invoked = FALSE;
    if (invoked) { return ATO_ERR_OK; }
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}
void ato__si_deinit(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    _ato_si_free(si);
}

/*********************************************************************************/
bool ato__si_isvalid(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return
        !ato_isnullorempty(si->organisation)
        && !ato_isnullorempty(si->product)
        && !ato_isnullorempty(si->version)
        && !ato_isnullorempty(si->timestamp)
        && !ato_isnullorempty(si->source)
        && !ato_isnullorempty(si->hash)
        ;
}

/*********************************************************************************/
int ato_si_set(const char *organisation, const char *product, const char *version, const char *timestamp, const char *source, const char *hash) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    _ato_si_free(si);
    if (organisation == NULL || product == NULL || version == NULL || timestamp == NULL || source == NULL || hash == NULL) {
		return ATO_ERR_INVALID_ARGS;
    }
    si->organisation = ato_strdup(organisation, 0);
    si->product = ato_strdup(product, 0);
    si->version = ato_strdup(version, 0);
    si->timestamp = ato_strdup(timestamp, 0);
    si->source = ato_strdup(source, 0);
    si->hash = ato_strdup(hash, 0);
    si->info = calloc(strlen(si->organisation) + strlen(si->product) + strlen(si->version) + strlen(si->timestamp) + strlen(si->source) + strlen(si->hash) + 100, sizeof(char));
    sprintf(si->info, "%s|%s|%s|%s|%s|%s", si->organisation, si->product, si->version, si->timestamp, si->source, si->hash);
    return ATO_ERR_OK;
}

const char *ato_si_organisation(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return si->organisation;
}

const char *ato_si_product(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return si->product;
}

const char *ato_si_version(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return si->version;
}

const char *ato_si_timestamp(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return si->timestamp;
}

const char *ato_si_source(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return si->source;
}

const char *ato_si_hash(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return si->hash;
}

const char *ato_si_info(void) {
    ato_SoftwareInfo *si = &g_softwareinfo;
    return si->info;
}
