/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atostr.h"

#include "atoutilcore.h"
#include "atoutil.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_UTIL;
static unsigned long _moduleid = ATO_BASE_MODULEID_UTIL;

static int _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
int ato__util_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}
void ato__util_deinit(void)
{
}

/*********************************************************************************/
static long _filesize(FILE *fp);

/*********************************************************************************/
void *ato_alloc(size_t sz) {
    return calloc(sz, sizeof(char)+1);
}

void *ato_free(void *v)
{
    if (v != NULL)
        free(v);
    return NULL;
}

char *ato_freeclr(char *v)
{
    if (v == NULL) return NULL;
    memset(v, 0, strlen(v));
    return ato_free(v);
}

bool ato_fileexists(const char *path)
{
    FILE *fp = fopen(path, "r");
    if (fp != NULL)
        fclose(fp);
    return fp != NULL;
}

int ato_fileload(ato_String **buffer, const char *filepath)
{
    int errcode = ATO_ERR_OK;
    char *buf = NULL;
    FILE *fp = NULL;
    size_t len = 0, rlen = 0;
    assert(filepath != NULL);

    ATO_ASSERT_ISNOTALLOCATED(buffer);

    if ((fp = fopen(filepath, "rb")) == NULL)
        return ATO_ERR_FILEOPEN;

    if ((len = (size_t)_filesize(fp)) == 0) {
        errcode = ATO_ERR_FILESTAT;
        goto finally;
    }

    buf = calloc(len + 1, 1);
    assert(buf != NULL);

    rlen = fread(buf, 1, len, fp);
    if (rlen != len) {
        free(buf);
        errcode = ATO_ERR_FILEREAD;
        goto finally;
    }

    ato_str_create(buffer, buf, len, FALSE);
    errcode = ATO_ERR_OK;
finally:
    if (fp != NULL) fclose(fp);
    return errcode;
}

static bool s_fwrite(const char *buffer, size_t len, FILE *fp) {
    size_t wlen = fwrite(buffer, sizeof(char), len, fp);
    return wlen == len;
}

int ato_fileappend(const char *pre, ato_String *buffer, const char *post, const char *filepath)
{
    int errcode = ATO_ERR_OK;
    FILE* fp;
    assert(filepath != NULL);

    if ((fp = fopen(filepath, "ab")) == NULL)
        return ATO_ERR_FILEOPEN;

    if (pre && !s_fwrite(pre, strlen(pre), fp)) { errcode = ATO_ERR_FILEWRITE; goto finally; }
    if (buffer && !s_fwrite(ato_str_value(buffer), ato_str_len(buffer), fp)) { errcode = ATO_ERR_FILEWRITE; goto finally; }
    if (post && !s_fwrite(post, strlen(post), fp)) { errcode = ATO_ERR_FILEWRITE; goto finally; }

    errcode = ATO_ERR_OK;
finally:
    if (fp != NULL) { fflush(fp); fclose(fp); }
    return errcode;
}


int ato_filewrite(ato_String *buffer,  const char *filepath)
{
    int errcode = ATO_ERR_OK;
    FILE* fp;
    assert(filepath != NULL);

    if ((fp = fopen(filepath, "wb")) == NULL)
        return ATO_ERR_FILEOPEN;

    if (buffer && !s_fwrite(ato_str_value(buffer), ato_str_len(buffer), fp)) { errcode = ATO_ERR_FILEWRITE; goto finally; }

    errcode = ATO_ERR_OK;
finally:
    if (fp != NULL) { fflush(fp); fclose(fp); }
    return errcode;
}

size_t ato_strlen(const char *src)
{
    if (src == NULL)
        return 0;
    return strlen(src);
}

char *ato_strdup(const char *src, size_t padlen)
{
    char *dst = NULL;
    if (src == NULL) return NULL;
    dst = calloc(strlen(src) + 1 + padlen, sizeof(char));
    assert(dst != NULL);
    return strcpy(dst, src);
}
char *ato_strndup(const char *src, size_t padlen, size_t len)
{
    char *dst = NULL;
    if (src == NULL) return NULL;
    dst = calloc(strlen(src) + 1 + padlen, sizeof(char));
    assert(dst != NULL);
    strncpy(dst, src, len)[len] = '\0';
    return dst;
}

char *ato_strcat(const char *src1, const char *src2, const char *src3)
{
    char *v = NULL;
    size_t dstlen = 0;
    dstlen = ato_strlen(src1) + ato_strlen(src2) + ato_strlen(src3);
    v = calloc(dstlen + 1, sizeof(char));
    assert(v != NULL);
    if (src1 != NULL) strcat(v, src1);
    if (src2 != NULL) strcat(v, src2);
    if (src3 != NULL) strcat(v, src3);
    return v;
}

char *ato_strcat4(const char *src1, const char *src2, const char *src3, const char *src4)
{
    char *v = NULL;
    size_t dstlen = 0;
    dstlen = ato_strlen(src1) + ato_strlen(src2) + ato_strlen(src3) + ato_strlen(src4);
    v = calloc(dstlen + 1, sizeof(char));
    assert(v != NULL);
    if (src1 != NULL) strcat(v, src1);
    if (src2 != NULL) strcat(v, src2);
    if (src3 != NULL) strcat(v, src3);
    if (src4 != NULL) strcat(v, src4);
    return v;
}

/*
char *ato_pathcat(const char *path1, const char *path2)
{
    if (path2[0] == '/' || path2[0] == '\\')
        return ato_strcat(path1, "", path2);
    else
        return ato_strcat(path1, "/", path2);
}
*/

static bool _ispathsep(const char sep)
{
    return sep == '/' || sep == '\\';
}

char *ato_pathcat(const char *path1, const char *path2)
{
    if (path1 == NULL) {
        return ato_strdup(path2, 0);
    } else if (path2 == NULL) {
        return ato_strdup(path1, 0);
    } else if (*path1 == '\0') {
        return ato_strdup(path2, 0);
    } else if (*path2 == '\0') {
        return ato_strdup(path1, 0);
    }
    else { // This means that path1 and path2 are non-NULL and at least length 1
        bool isp2absolute = ((_ispathsep(path2[0]) && _ispathsep(path2[1])) || path2[1] == ':');
        size_t i1 = strlen(path1) - 1;
        char *p1 = ato_strdup(path1, 0);
        const char *p2 = path2;
        char *s = NULL;
        assert(!isp2absolute); ATO_IGNORE(isp2absolute);
        // Get rid of any single "/" at the end - leave "//" alone
        if (_ispathsep(p1[i1]) && !_ispathsep(p1[i1-1]))
            p1[i1] = '\0';
        // Skip "/" at the beginning
        while (_ispathsep(*p2))
            p2++;

        s = ato_strcat(p1, "/", p2);
        free(p1);
        return s;
    }
}


void *ato_memdup(const void *src, size_t len, size_t padlen)
{
    char *dst = NULL;
    if (src == NULL) return NULL;
    dst = calloc(len + 1 + padlen, sizeof(char));
    assert(dst != NULL);
    return memcpy(dst, src, len);
}


static size_t _strtok_count(const char *src, const char *delim)
{
    size_t count = 0;
    if (*src != '\0') {
        for (; *src != '\0'; src++) {
            if (strchr(delim, *src) != NULL)
                count++;
        }
        count++;
    }
    return count;
}
static size_t _strtok_len(const char *src, const char *delim)
{
    size_t len = 0;
    for (; *src != '\0'; src++) {
        if (strchr(delim, *src) != NULL)
            break;
        len++;
    }
    return len;
}
static const char *_strtok_dup(char *dst, const char *src, const char *delim)
{
    for (; *src != '\0'; src++, dst++) {
        if (strchr(delim, *src) != NULL)
            break;
        *dst = *src;
    }
    *dst = '\0';
    return *src == '\0' ? NULL : ++src;
}

/*
Bit horrible - consider using regex
*/
size_t ato_strtok2array(char ***dst, const char *src, const char *delim)
{
    size_t tindex = 0, tcount = 0; // token vars
    char **d = NULL;
    const char *s = NULL;
    assert(src != NULL); assert(delim != NULL);
    ATO_ASSERT_ISNOTALLOCATED(dst);

    tcount = _strtok_count(src, delim);
    if (tcount == 0)
        return 0;

    d = calloc(tcount+1, sizeof(void *));
    s = src;
    while (s != NULL) {
        size_t size = _strtok_len(s, delim);
        d[tindex] = calloc(size+1, sizeof(char));
        s = _strtok_dup(d[tindex], s, delim);
        tindex++;
    }
    *dst = d;
    return tcount;
}
void ato_strtokfree(char *dst[])
{
    size_t i = 0;
    if (dst == NULL)
        return;
    for (i = 0; dst[i] != NULL; i++)
        free(dst[i]);
    free(dst);
}

void ato_localtime(time_t *timep, struct tm *result)
{
#if defined(ATO_WINXX)
    errno_t err = localtime_s(result, timep);
    assert(err == 0); ATO_IGNORE(err);
#else
    struct tm *r = localtime_r(timep, result);
    assert(r != NULL);
#endif
}

void ato_gmtime(time_t *timep, struct tm *result)
{
#if defined(ATO_WINXX)
    errno_t err = gmtime_s(result, timep);
    assert(err == 0); ATO_IGNORE(err);
#else
    struct tm *r = gmtime_r(timep, result);
    assert(r != NULL);
#endif
}

char *ato_strtok(char *str, const char *delimiters, char **context)
{
#if defined(ATO_WINXX)
    return strtok_s(str, delimiters, context);
#else
    return strtok_r(str, delimiters, context);
#endif
}

size_t ato_strftime_utc(char *buf, size_t sizeinbytes, time_t *timep, bool isUtc)
{
        struct tm tms;
        if (isUtc) {
            ato_localtime(timep, &tms);
        } else {
            ato_gmtime(timep, &tms);
        }
        return strftime(buf, sizeinbytes, "%Y-%m-%dT%H:%M:%S", &tms);
}

static int _utctimediffsecs(void)
{
    int diff = 0;
    time_t now = 0, local = 0, utc = 0;
    struct tm tm_local;
    struct tm tm_utc;

    time(&now);
    ato_localtime(&now, &tm_local);
    local = mktime(&tm_local);
    ato_gmtime(&now, &tm_utc);
    utc = mktime(&tm_utc);

    {
    //diff = (int)difftime(local, utc); // this will generate a compiler error about casting from double to int, whereas the code below doesn't.
    double d_diff = difftime(local, utc);
    diff = (int)d_diff;
    ATO_IGNORE(d_diff);
    }

    return diff;
}

time_t ato_str2time(const char *src)
{
    /*TODO: be nice to do as a regex validation of one of the 4 formats below e.g. 2010-07-27T16:01:31.254+10:00*/
    struct tm datetime;
    time_t timet;
    size_t len = 0, req_len1 = 0, req_len1a = 0, req_len2 = 0, req_len2a = 0;
    int tz_h = 0, tz_m = 0, tz_s = 0;
    char **timeelts = NULL, *t = NULL;
    size_t count = 0, tndx = 0;
    bool hasmillis = FALSE;
    char tz_sign = '\0';
    bool utc = (strchr(src, 'Z') != NULL);

    assert(src != NULL);
    req_len1 =  strlen("ccyy-mm-ddThh:mm:ss.xxx+hh:mm"); // Main format one
    req_len2 =  strlen("ccyy-mm-ddThh:mm:ss.xxxZ"); // Main format two
    req_len1a = strlen("ccyy-mm-ddThh:mm:ss+hh:mm"); // without millis
    req_len2a = strlen("ccyy-mm-ddThh:mm:ssZ"); // without millis
    len = strlen(src);
    assert(len == req_len1 || len == req_len1a || len == req_len2 || len == req_len2a);
    hasmillis = strchr(src, '.') != NULL;
    // work out if the timezone, if any, is ahead or behind
    if (!utc) {
        if (len == req_len1)
            tz_sign = src[19];
        else if (len == req_len1a)
            tz_sign = src[23];
    }

    count = ato_strtok2array(&timeelts, src, "-T:.+Z");
    memset(&datetime, 0, sizeof(struct tm));
    t = timeelts[tndx++];
    datetime.tm_year = (int)strtol(t, NULL, 10) - 1900;
    t = timeelts[tndx++];
    datetime.tm_mon = (int)strtol(t, NULL, 10) - 1;
    t = timeelts[tndx++];
    datetime.tm_mday = (int)strtol(t, NULL, 10);
    t = timeelts[tndx++];
    datetime.tm_hour = (int)strtol(t, NULL, 10);
    t = timeelts[tndx++];
    datetime.tm_min = (int)strtol(t, NULL, 10);
    t = timeelts[tndx++];
    datetime.tm_sec = (int)strtol(t, NULL, 10);

    if (tndx < count && hasmillis) {
        // process then ignore millis
        t = timeelts[tndx++];
        //int millis = (int)strtol(t, NULL, 10);
    }
    if (tndx < count) {
        t = timeelts[tndx++];
        tz_h = (int)strtol(t, NULL, 10);
    }
    if (tndx < count) {
        t = timeelts[tndx++];
        tz_m = (int)strtol(t, NULL, 10);
    }
    ato_strtokfree(timeelts);

    tz_s = tz_h * 60 * 60 + tz_m * 60;
    if (tz_sign == '-')
        tz_s = -tz_s;
    if (utc) {
        if (tz_sign == '\0')
            tz_s = _utctimediffsecs();
        datetime.tm_sec += tz_s;
    }
    //datetime.tm_isdst = -1;
    timet = mktime(&datetime);
    assert(timet != -1);
    return timet;
}


// e.g. https://thirdparty.authentication.business.gov.au/R3.0/vanguard/S007v1.1/service.svc
char *ato_host(char **buffer, const char *url)
{
    char **urlelts = NULL, *host = NULL;
    size_t count = 0;
    ATO_ASSERT_ISNOTALLOCATED(buffer);
    assert(url != NULL);
    count = ato_strtok2array(&urlelts, url, "/:");
    assert(count > 3);
    host = urlelts[3];
    *buffer = ato_strdup(host, 0);
    ato_strtokfree(urlelts);
    return *buffer;
}

bool ato_isnullorempty(const char *str)
{
    return str == NULL || str[0] == '\0';
}

bool ato_isnulloremptyws(const char *str, size_t len)
{
    size_t i = 0;
    if (len == 0) len = ato_strlen(str);
    if (str == NULL) return TRUE;
    for (i = 0; i < len; i++) {
        if (!isspace((int)str[i]))
            return FALSE;
    }
    return TRUE;
}

char *ato_striff(char *str1, char *str2) {
    return str1 != NULL ? str1 : str2;
}

size_t ato_trim(char *str, size_t len)
{
    if (len == 0) len = strlen(str);
    while (len > 0 && isspace((int)str[len-1])) {
        str[len-1] = '\0';
        len -= 1;
    }
    return len;
}

char *ato_substitute(char **dst, const char *src, const char *value)
{
    char *src_o = NULL;
    char *d = NULL;
    char *dst_o = NULL;

    ATO_ASSERT_ISNOTALLOCATED(dst);
    if (src == NULL)
        src = "";
    //assert(src != NULL);
    if (value == NULL)
        return *dst = ato_strdup(src, 0);

    src_o = strchr(src, (int)'$');
    if (src_o == NULL)
        return *dst = ato_strdup(src, 0);

    d = ato_strdup(src, strlen(value));
    dst_o = strchr(d, (int)'$');
    *dst_o = '\0';
    strcat(d, value);
    strcat(d, ++src_o);

    *dst = d;
    return d;
}

char *ato_substitutevar(char **dst, const char *src, const char *name, const char *value)
{
    char *src_o = NULL;
    char *d = NULL;
    char *dst_o = NULL;

    ATO_ASSERT_ISNOTALLOCATED(dst);
    if (src == NULL)
        src = "";
    //assert(src != NULL);
    if (value == NULL)
        return *dst = ato_strdup(src, 0);

    src_o = strstr(src, name);
    if (src_o == NULL)
        return *dst = ato_strdup(src, 0);

    d = ato_strdup(src, strlen(value));
    dst_o = strstr(d, name);
    *dst_o = '\0';
    strcat(d, value);
    strcat(d, src_o + strlen(name));

    *dst = d;
    return d;
}

bool ato_streq(const char *src, const char *src2)
{
    if (src == NULL && src2 == NULL)
        return TRUE;
    if ((src == NULL && src2 != NULL) || (src != NULL && src2 == NULL))
        return FALSE;
    return strcmp(src, src2) == 0;
}

bool ato_streqi(const char *src, const char *src2)
{
    if (src == NULL && src2 == NULL)
        return TRUE;
    if ((src == NULL && src2 != NULL) || (src != NULL && src2 == NULL))
        return FALSE;
    if (strlen(src) != strlen(src2))
        return FALSE;
    while (*src2 != '\0') {
        if (tolower((int)*src) != tolower((int)*src2))
            return FALSE;
        src++; src2++;
    }
    return TRUE;
}

bool ato_strstartswith(const char *src, const char *src2)
{
    if (src == NULL && src2 == NULL)
        return TRUE;
    if ((src == NULL && src2 != NULL) || (src != NULL && src2 == NULL))
        return FALSE;
    if (strlen(src2) > strlen(src))
        return FALSE;
    while (*src2 != '\0') {
        if (*src != *src2)
            return FALSE;
        src++; src2++;
    }
    return TRUE;
}

bool ato_strstartswithi(const char *src, const char *src2)
{
    if (src == NULL && src2 == NULL)
        return TRUE;
    if ((src == NULL && src2 != NULL) || (src != NULL && src2 == NULL))
        return FALSE;
    if (strlen(src2) > strlen(src))
        return FALSE;
    while (*src2 != '\0') {
        if (tolower((int)*src) != tolower((int)*src2))
            return FALSE;
        src++; src2++;
    }
    return TRUE;
}

char *ato_substr(const char *str, const char *start, const char *end) {
    char *result;
    if (str == NULL || start == NULL || end == NULL) { return NULL; }
    if (end < start) { return NULL; }
    result = calloc(end - start + 5, sizeof(str));
    memcpy(result, start, end - start);
    (result + (end - start))[0] = '\0';
    return result;
}

char *ato_substrmatch(const char *str, const char *matchstart, const char *matchend) {
    char *start, *end;
    if (str == NULL || matchstart == NULL || matchend == NULL) { return NULL; }
    start = strstr(str, matchstart);
    if (start == NULL) { return NULL; }
    start += strlen(matchstart);
    end = strstr(start, matchend);
    if (end == NULL) { return NULL; }
    return ato_substr(str, start, end);
}

char *ato_itoa(char *dst, int value)
{
    assert(dst != NULL);
    assert(sprintf(dst, "%d", value) != -1); ATO_IGNORE(value);
    return dst;
}

void *ato_value_v(void *value, void *defvalue) { return value != NULL ? value : defvalue; }

const char *ato_value(const char *value, const char *defvalue)
{
    const char *v = NULL;

    if (!ato_isnullorempty(value)) {
        v = value;
    } else {
        v = defvalue;
    }
    return v;
}

const char *ato_valuec(const char *value, const char *defvalue)
{
    const char *v = NULL;

    if (!ato_isnullorempty(value)) {
        v = value;
    } else {
        v = defvalue;
    }
    return v;
}

int ato_value_int(const char *value, int defvalue)
{
    int v = FALSE;

    if (!ato_isnullorempty(value)) {
        v = (int)strtol(value, NULL, 10);
    } else {
        v = defvalue;
    }
    return v;
}

bool ato_value_bool(const char *value, bool defvalue)
{
    bool v = FALSE;

    if (!ato_isnullorempty(value)) {
        v = (value[0] == '1' || value[0] == 'y' || value[0] == 'Y' || value[0] == 't' || value[0] == 'T');
    } else {
        v = defvalue;
    }
    return v;
}

// We also strip leading/trailing spaces
char *ato_stripouterxml(char *text)
{
    char *t_new = NULL;
    char *t_start = text;
    char *t_end = NULL;
    char savech = '\0';
    if (text == NULL)
        return text;
    if (*t_start != '<')
        return text;
    while (*t_start && *t_start != '>')
        t_start++;
    if (*t_start) t_start++;
    while (*t_start && isspace((int)*t_start))
        t_start++;
    if (!*t_start)
        return text;
    t_end = (text + (strlen(text)-1));
    if (*t_end != '>')
        return text;
    while (*t_end && *t_end != '<')
        t_end--;
    if (*t_end) t_end--;
    while (*t_end && isspace((int)*t_end))
        t_end--;
    if (!*t_end)
        return text;
    t_end++;
    savech = *t_end;
    *t_end = '\0';
    t_new = ato_strdup(t_start, 0);
    *t_end = savech;
    free(text);
    return t_new;
}
/***********************************************************************/

static long _filesize(FILE *fp)
{
    long start, end;

    if (fseek(fp, 0, SEEK_SET) != 0)
        return 0;
    if ((start = ftell(fp)) == -1L)
        return 0;
    if (fseek(fp, 0, SEEK_END) != 0)
        return 0;
    if ((end = ftell(fp)) == -1L)
        return 0;
    if (fseek(fp, 0, SEEK_SET) != 0)
        return 0;
    return end - start;
}

/***********************************************************************/
