#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>

#include "libxml/tree.h"
#include "libxml/parser.h"
#include "libxml/xpath.h"
#include "libxml/xpathInternals.h"
#include "libxml/xmlsave.h"

#include "atointernal.h"
#include "atotypes.h"
#include "atoexcept.h"
#include "atostr.h"
#include "atoutil.h"
#include "atolog.h"
#include "atoerr.h"
#include "atoctx.h"
#include "atoerrh.h"

#include "atoxml.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_XML;
static unsigned long _moduleid = ATO_BASE_MODULEID_XML;

static int _loglevel = ATO_LOG_WARN;

/*********************************************************************************/

struct _ato_Xml {
    bool isdirty;
    xmlDoc *doc;
    xmlXPathContext *xpathCtx;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
static size_t _size(xmlNodeSet *nodes);

/*********************************************************************************/
int ato__xml_init()
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;

    xmlInitParser();
    LIBXML_TEST_VERSION
    xmlTreeIndentString = "\t";
    ATO_IGNORE(xmlTreeIndentString);

    invoked = TRUE;
    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__xml_deinit()
{
    static bool invoked = FALSE;
    if (invoked) return;
    invoked = TRUE;

    xmlCleanupParser();
}

/*********************************************************************************/

// copy the contents of xc to c
static char *_copy_xc2c(char *buffer, size_t buflen, const xmlChar *xc)
{
    char *v = NULL;

    if (xc != NULL) {
        size_t len = (size_t)xmlStrlen(xc);
        if (buffer == NULL) {
            v = calloc(len+1, sizeof(char));
        } else {
            len = len > buflen ? buflen : len;
            v = buffer;
        }
        if (v != NULL) {
            strncpy(v, (const char*)xc, len);
            v[len] = '\0';
        }
    }
    return v;
}

// copy the contents of xc to c and free xc
static char *_move_xc2c(char *buffer, size_t buflen, xmlChar *xc)
{
    char *v = _copy_xc2c(buffer, buflen, xc);
    if (xc != NULL)
        xmlFree(xc);
    return v;
}

static char *_value(char *buffer, size_t buflen, xmlNode *node)
{
    return node ? _move_xc2c(buffer, buflen, xmlNodeGetContent(node)) : NULL;
}

void ato_xml_setnodes(ato_Ctx *ctx, ato_Xml *xml, const char *nodesxpath, const char *idxpath, ato_xml_setnode setnode, void *userobj)
{
    const char *function = "ato_xml_setnodes";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int errcode = ATO_ERR_OK;
    xmlXPathObject *xpathObj = NULL;
    size_t i = 0;
    char *id = NULL;
    ATO_CTX_FN_START(ctx);

    Try {
        xpathObj = xmlXPathEvalExpression((const xmlChar*)nodesxpath, xml->xpathCtx);
        if (xpathObj != NULL) {
            for (i = 0; i < _size(xpathObj->nodesetval); i++) {
                xmlNode *pnode = NULL;
                xmlNode *idnode = NULL;
                pnode = xpathObj->nodesetval->nodeTab[i];
                idnode = ato_xml_findnode(xml, pnode, idxpath, NULL);
                if (idnode == NULL)
                    Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_XMLNODENOTFOUND, strlen(nodesxpath) + strlen(idxpath), "%s/@%s", nodesxpath, idxpath);
                id = _value(NULL, 0, idnode);
                setnode(ctx, userobj, id, pnode);
                id = ato_free(id);
            }
        }
    } Catch (errcode) {

    }

    if (xpathObj != NULL) xmlXPathFreeObject(xpathObj);
    id = ato_free(id);
    ATO_CTX_FN_END(ctx, errcode);
    if (errcode != ATO_ERR_OK) Throw errcode;
}

static int _dumpnode(xmlDoc *doc, xmlBuffer *nodeBuffer, xmlNode *node)
{
    xmlSaveCtxt *sctx = NULL;
    sctx = xmlSaveToBuffer(nodeBuffer, NULL, 0);
    if (!sctx) return -1;
    xmlSaveTree(sctx, node);
    ATO_IGNORE(doc);
    return xmlSaveClose(sctx);
}

static char *_valueraw(xmlDoc *doc, char *buffer, size_t buflen, xmlNode *node, bool stripouter)
{
    size_t len = 0;
    xmlBuffer *nodeBuffer = NULL;
    char *v = NULL;

    if (node == NULL) return NULL;

    nodeBuffer = xmlBufferCreate();
    //if (xmlNodeDump(nodeBuffer, doc, node, 0, 1) != -1) {
    if (_dumpnode(doc, nodeBuffer, node) != -1) {
        len = (size_t)nodeBuffer->use;
        if (buffer == NULL) {
            v = calloc(len+1, sizeof(char));
        } else {
            len = len > buflen ? buflen : len;
            v = buffer;
        }
        if (v != NULL) {
            strncpy(v, (char*)nodeBuffer->content, len);
            v[len] = '\0';
        }
    }
    xmlBufferFree(nodeBuffer);
    if (stripouter)
        return ato_stripouterxml(v);
    else
        return v;
}

char *ato_xml_value(ato_Xml *xml, void *xnode, char **buffer)
{
    ATO_ASSERT_ISNOTALLOCATED(buffer);
    assert(xml != NULL); ATO_IGNORE(xml);
    *buffer = _value(NULL, 0, xnode);
    return *buffer;
}

char *ato_xml_valueraw(ato_Xml *xml, void *xnode, char **buffer, bool stripouter)
{
    ATO_ASSERT_ISNOTALLOCATED(buffer);
    assert(xml != NULL); ATO_IGNORE(xml);
    *buffer = _valueraw(xml->doc, NULL, 0, xnode, stripouter);
    return *buffer;
}

char *ato_xml_nodevaluevar(ato_Xml *xml, void *pxnode, const char *xpath, const char *var, char **value, ato_eXmlNodeContent raw)
{
    xmlNode *node = NULL;
    assert(xml != NULL); assert(xpath != NULL);
    ATO_ASSERT_ISNOTALLOCATED(value);
    node = ato_xml_findnode(xml, pxnode, xpath, var);
    if (raw == ATO_XML_RAW_FALSE)
        *value = ato_xml_value(xml, node, value);
    else
        *value = ato_xml_valueraw(xml, node, value, raw == ATO_XML_RAW_INNER);
    return *value;
}

char *ato_xml_nodevalue(ato_Xml *xml, void *pxnode, const char *xpath, char **value, ato_eXmlNodeContent raw)
{
    return ato_xml_nodevaluevar(xml, pxnode, xpath, NULL, value, raw);
}

void ato_xml_setvalue(ato_Xml *xml, void *xnode, const char *value)
{
    char *oldvalue = NULL;
    assert(xnode != NULL); //assert(value != NULL);
    xml->isdirty = TRUE;
    ato_xml_value(xml, xnode, &oldvalue);
    if (oldvalue == NULL || !ato_streq(value, oldvalue))
        xmlNodeSetContent(xnode, (const xmlChar*)value);
    oldvalue = ato_free(oldvalue);
}

void ato_xml_setvaluestr(ato_Xml *xml, void *xnode, ato_String *value)
{
    assert(xnode != NULL); //assert(value != NULL);
    xml->isdirty = TRUE;
    xmlNodeSetContentLen(xnode, (const xmlChar*)ato_str_value(value), ato_str_len(value));
}

static void _xml_setfragment(ato_Ctx *ctx, ato_Xml *xml, xmlNode *pxnode, const char *value, size_t vlen)
{
    const char *function = "_xml_setfragment";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    xmlDoc *doc = NULL;
    xmlNode *xnode = NULL;
    xmlNode *newxnode = NULL;
    int errcode = ATO_ERR_OK;
    char *copyvalue = NULL;
    size_t copylen = vlen;
    ATO_CTX_FN_START(ctx);

    if (copylen > 1000)
        copylen = vlen;
    copyvalue = ato_memdup(value, copylen, 1);
    copyvalue[copylen] = '\0';
    if ((doc = xmlParseMemory(value, (int)vlen)) == NULL) {
        errcode =  ATO_CTX_NEWERR(ctx, ATO_ERR_XMLPARSE, copyvalue);
    } else if ((xnode = xmlDocGetRootElement(doc)) == NULL) {
        errcode = ATO_CTX_NEWERR(ctx, ATO_ERR_XMLROOT, copyvalue);
    } else {
        newxnode = xmlDocCopyNode(xnode, xml->doc, 1);
        if (newxnode)
            newxnode = xmlAddChild(pxnode, newxnode);
        if (!newxnode) {
            const char *s = "Failed to load xml fragment into document: [%s]";
            errcode = ATO_CTX_VNEWERR(ctx, ATO_ERR_XML, strlen(s), s, copyvalue);
        }
    }

    if (doc) xmlFreeDoc(doc);
    if (copyvalue) free(copyvalue);

    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

static bool _startswith(const char *str, const char *cstr)
{
    assert(str != NULL); assert(cstr != NULL);
    while (*str && isspace((int)*str)) ++str;
    while (*cstr && *str == *cstr) { str++; cstr++; }
    return *cstr == '\0';
}

static void _xml_setvalueraw(ato_Ctx *ctx, ato_Xml *xml, xmlNode *xnode, const char *value)
{
    const char *function = "_xml_setvalueraw";
    size_t vlen = ato_strlen(value);
    char *oldvalue = NULL;
    int errcode = ATO_ERR_OK;
    ato_Log *log = ato_ctx_log(ctx);
    ATO_CTX_FN_START(ctx);
    assert(xnode != NULL); assert(value != NULL);
    xml->isdirty = TRUE;
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "New value length = %d", vlen);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "value first char = '%c'", value[0]);
    ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Checking current value against new value");
    ato_xml_valueraw(xml, xnode, &oldvalue, TRUE);
    if (oldvalue == NULL || !ato_streq(value, oldvalue)) {
        if (_startswith(value, "<")) {
            ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Clearing current content");
            xmlNodeSetContent(xnode, (const xmlChar*)"");
            ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Loading a XML fragment");
            _xml_setfragment(ctx, xml, xnode, value, vlen);
        } else {
            xmlChar *vx = NULL;
            ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Encoding content (non-xml)");
            vx = xmlEncodeEntitiesReentrant(xml->doc, (const xmlChar*)value);
            ATO_LOG_MSG(log, ATO_LOG_DEBUG, "Setting content (non-xml)");
            xmlNodeSetContent(xnode, (const xmlChar*)vx);
            xmlFree(vx);
        }
    } else {
        ATO_LOG_MSG(log, ATO_LOG_DEBUG, "No change");
    }
    oldvalue = ato_free(oldvalue);
    ATO_CTX_FN_END(ctx, errcode);
}

void *ato_xml_encodedvalue_create(ato_Xml *xml, const char *value)
{
    return xmlEncodeEntitiesReentrant(xml->doc, (const xmlChar*)value);
}

void *ato_xml_encodedvalue_free(ato_Xml *xml, void *encodedvalue)
{
    ATO_IGNORE(xml);
    if (encodedvalue)   xmlFree((xmlChar*)encodedvalue);
    return NULL;
}

bool ato_xml_setnodevaluevar(ato_Xml *xml, void *pxnode, const char *xpath, const char *var, const char *value)
{
    xmlNode *node = NULL;
    assert(xml != NULL); assert(xpath != NULL); assert(value != NULL);
    node = ato_xml_findnode(xml, pxnode, xpath, var);
    if (node != NULL)
        ato_xml_setvalue(xml, node, value);
    return node != NULL;
}
bool ato_xml_setnodevalue(ato_Xml *xml, void *pxnode, const char *xpath, const char *value)
{
    return ato_xml_setnodevaluevar(xml, pxnode, xpath, NULL, value);
}

static void _ato_xml_setvalue(ato_Xml *xml, void *xnode, const char *value, size_t len)
{
    char *oldvalue = NULL;
    assert(xnode != NULL); //assert(value != NULL);
    xml->isdirty = TRUE;
    ato_xml_value(xml, xnode, &oldvalue);
    if (oldvalue == NULL || !ato_streq(value, oldvalue)) {
        if (len == 0) {
            xmlChar *encodedvalue = xmlEncodeEntitiesReentrant(xml->doc, (const xmlChar*)value);
            xmlNodeSetContent(xnode, encodedvalue);
            xmlFree(encodedvalue);
        } else {
            xmlNodeSetContentLen(xnode, (const xmlChar*)value, len);
        }
    }
    oldvalue = ato_free(oldvalue);
    ATO_IGNORE(len);
}

static bool _ato_xml_setnodevaluevarlen(ato_Xml *xml, void *pxnode, const char *xpath, const char *var, const char *value, size_t len)
{
    xmlNode *node = NULL;
    assert(xml != NULL); assert(xpath != NULL); assert(value != NULL);
    node = ato_xml_findnode(xml, pxnode, xpath, var);
    if (node != NULL)
        _ato_xml_setvalue(xml, node, value, len);
    return node != NULL;
}
bool ato_xml_setnodevaluelen(ato_Xml *xml, void *pxnode, const char *xpath, const char *value, size_t len)
{
    return _ato_xml_setnodevaluevarlen(xml, pxnode, xpath, NULL, value, len);
}

bool ato_xml_createnodevaluevar(ato_Xml *xml, void *pxnode, const char *xpath, const char *name, const char *var, const char *value)
{
    xmlNode *pnode = pxnode;
    xmlNode *node = pxnode;
    assert(xml != NULL);
    if (xpath)
        pnode = node = ato_xml_findnode(xml, pnode, xpath, var);
    if (node != NULL) {
        node = ato_xml_findnode(xml, node, name, NULL);
        if (!value) {
            if (node) {
                ato_xml_deletenode(xml, node);
                return TRUE;
            }
        } else {
            if (!node)
                node = ato_xml_addelt(xml, pnode, name);
            ato_xml_setvalue(xml, node, value);
            return TRUE;
        }
    }
    return FALSE;
}
bool ato_xml_createnodevalue(ato_Xml *xml, void *pxnode, const char *xpath, const char *name, const char *value)
{
    return ato_xml_createnodevaluevar(xml, pxnode, xpath, name, NULL, value);
}

static bool _ato_xml_createnodevaluevar_len(ato_Xml *xml, void *pxnode, const char *xpath, const char *name, const char *var, const char *value, size_t len)
{
    xmlNode *node = pxnode;
    assert(xml != NULL);
    if (xpath)
        node = ato_xml_findnode(xml, pxnode, xpath, var);
    if (node != NULL) {
        node = ato_xml_findnode(xml, node, name, NULL);
        if (!value) {
            if (node) {
                ato_xml_deletenode(xml, node);
                return TRUE;
            }
        } else {
            if (!node)
                node = ato_xml_addelt(xml, pxnode, name);
            _ato_xml_setvalue(xml, node, value, len);
            return TRUE;
        }
    }
    return FALSE;
}
bool ato_xml_createnodevaluelen(ato_Xml *xml, void *pxnode, const char *xpath, const char *name, const char *value, size_t len)
{
    return _ato_xml_createnodevaluevar_len(xml, pxnode, xpath, name, NULL, value, len);
}

static bool _xml_createnodevaluevarraw(ato_Ctx *ctx, ato_Xml *xml, void *pxnode, const char *xpath, const char *name, const char *var, const char *value)
{
    const char *function = "_xml_createnodevaluevarraw";
    xmlNode *node = pxnode;
    bool ok = FALSE;
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    assert(xml != NULL);
    if (xpath)
        node = ato_xml_findnode(xml, pxnode, xpath, var);
    if (node != NULL) {
        if (name)
            node = ato_xml_findnode(xml, node, name, NULL);
        if (!value) {
            if (node) {
                ato_xml_deletenode(xml, node);
                ok = TRUE;
            }
        } else {
            if (!node)
                node = ato_xml_addelt(xml, pxnode, name);
            _xml_setvalueraw(ctx, xml, node, value);
            ok = TRUE;
        }
    }
    ATO_CTX_FN_END(ctx, errcode);
    return ok;
}

bool ato_xml_createnodevalueraw(ato_Ctx *ctx, ato_Xml *xml, void *pxnode, const char *xpath, const char *name, const char *value)
{
    const char *function = "ato_xml_createnodevalueraw";
    bool ok = FALSE;
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    ok = _xml_createnodevaluevarraw(ctx, xml, pxnode, xpath, name, NULL, value);
    ATO_CTX_FN_END(ctx, errcode);
    return ok;
}

/*
int ato_xml_createnodevalueraw2(ato_Ctx *ctx, ato_Xml *xml, void *pxnode, const char *value, const char *prefix, const char *href)
{
    const char *function = "ato_xml_createnodevalueraw2";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    xmlNode *parent = pxnode;
    xmlDoc *doc = NULL;
    xmlNode *xnode = NULL;
    xmlNode *newxnode = NULL;
    xmlNs *ns = NULL;
    if ((doc = xmlParseMemory(value, (int)strlen(value))) == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_XMLPARSE, value);
    if ((xnode = xmlDocGetRootElement(doc)) == NULL) {
        xmlFreeDoc(doc);
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_XMLROOT, value);
    }
    ns = xmlNewNs(xnode, (const xmlChar *)href, (const xmlChar *)prefix);
    xmlSetNs(xnode, ns);
    newxnode = xmlDocCopyNode(xnode, xml->doc, 1);
    if (newxnode)
        newxnode = xmlAddChild(parent, newxnode);
    xmlFreeDoc(doc);
    if (!newxnode) {
        const char *s = "Failed to load xml fragment into document: [%s]";
        Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_XML, strlen(s), s, value);
    }
    return 0;
}*/

static void _load(ato_Ctx *ctx, ato_Xml *xml, ato_String *buffer)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_load";
    if ((xml->doc = xmlParseMemory(ato_str_value(buffer), ato_str_len(buffer))) == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_XMLPARSE, "");
    if (xmlDocGetRootElement(xml->doc) == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_XMLROOT, "");
    if ((xml->xpathCtx = xmlXPathNewContext(xml->doc)) == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_XMLXPCONTEXT, "");
}

void ato_xml_namespace_register(ato_Ctx *ctx, ato_Xml *xml, const char *prefix, const char *uri)
{
    static const char *function = "ato_xml_addnamespace";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    bool ok = FALSE;
    assert(xml != NULL); assert(prefix != NULL); assert(uri != NULL);
    ok = xmlXPathRegisterNs(xml->xpathCtx, (const xmlChar *)prefix, (const xmlChar *)uri) == 0;
    if (!ok)
        Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_UNKNOWN, strlen(prefix) + strlen(uri), "%s:%s", prefix, uri);
}

// deprecated
void ato_xml_addnamespace(ato_Ctx *ctx, ato_Xml *xml, const char *prefix, const char *uri) {
    ato_xml_namespace_register(ctx, xml, prefix, uri);
}

void ato_xml_create(ato_Ctx *ctx, ato_Xml **obj, ato_String *buffer)
{
    ato_Xml *xml = NULL;

    ATO_ASSERT_ISNOTALLOCATED(obj);
    assert(buffer != NULL);

    *obj = xml = calloc(1, sizeof(ato_Xml));
    assert(xml != NULL);

    _load(ctx, xml, buffer);

}

void *ato_xml_free(ato_Xml *xml)
{
    if (xml == NULL) return NULL;

    if (xml->xpathCtx != NULL) { xmlXPathFreeContext(xml->xpathCtx); xml->xpathCtx = NULL; }
    if (xml->doc != NULL) { xmlFreeDoc(xml->doc); xml->doc = NULL; }
    free(xml);
    return NULL;
}

/*********************************************************************************/
static size_t _size(xmlNodeSet *nodes)
{
    return (nodes != NULL) ? (size_t)nodes->nodeNr : 0;
}

void ato_xml_listfree(ato_Xml *xml, void *list)
{
    xmlXPathObject *xpathObj = list;
    assert(xml != NULL); ATO_IGNORE(xml);
    if (xpathObj != NULL) xmlXPathFreeObject(xpathObj);
}
size_t ato_xml_listcount(ato_Xml *xml, void *list)
{
    xmlXPathObject *xpathObj = list;
    assert(xml != NULL); ATO_IGNORE(xml);
    if (xpathObj == NULL) return 0;
    return _size(xpathObj->nodesetval);
}
void *ato_xml_listitem(ato_Xml *xml, void *list, size_t index)
{
    xmlNode *node = NULL;
    xmlXPathObject *xpathObj = list;
    size_t count = 0;
    assert(xml != NULL); ATO_IGNORE(xml);
    if (list == NULL) return NULL;
    count = _size(xpathObj->nodesetval);
    if (index >= count) return NULL;
    node = xpathObj->nodesetval->nodeTab[index];
    assert(node != NULL);
    if (node->type != XML_ELEMENT_NODE && node->type != XML_ATTRIBUTE_NODE)
        node = NULL;
    return node;

}
void *ato_xml_listload(ato_Xml *xml, void *pxnode, const char *xpath, const char *var)
{
    xmlXPathObject *xpathObj = NULL;
    size_t size = 0;
    char *newxpath = NULL;
    const char *xpathexpr = NULL;

    assert(xml != NULL); assert(xpath != NULL);

    xpathexpr = xpath;
    if (var != NULL)
        xpathexpr = ato_substitute(&newxpath, xpath, var);
    if (xpathexpr == NULL)
        goto finally;

    xml->xpathCtx->node = pxnode;
    xpathObj = xmlXPathEvalExpression((const xmlChar*)xpathexpr, xml->xpathCtx);
    if(xpathObj == NULL) goto finally;
    size = _size(xpathObj->nodesetval);
    if (size == 0)
        goto finally;

finally:
    if (newxpath != NULL) free(newxpath);
    if (size == 0) {
        if (xpathObj != NULL) xmlXPathFreeObject(xpathObj);
        xpathObj = NULL;
    }
    xml->xpathCtx->node = NULL;
    return xpathObj;
}

void *ato_xml_findnode(ato_Xml *xml, void *pxnode, const char *xpath, const char *var)
{
    xmlXPathObject *xpathObj = NULL;
    xmlNodeSet *nodes = NULL;
    xmlNode *node = NULL;
    size_t size = 0;
    char *newxpath = NULL;
    const char *xpathexpr = NULL;

    if (xpath == NULL) {
        node = xmlDocGetRootElement(xml->doc);
        goto finally;
    }
    xpathexpr = xpath;
    if (var != NULL)
        xpathexpr = ato_substitute(&newxpath, xpath, var);
    if (xpathexpr == NULL)
        goto finally;

    xml->xpathCtx->node = pxnode;
    xpathObj = xmlXPathEvalExpression((const xmlChar*)xpathexpr, xml->xpathCtx);
    if(xpathObj == NULL) goto finally;
    nodes = xpathObj->nodesetval;
    size = _size(nodes);
    if (size == 0)
        goto finally;
    assert(size == 1);
    node = nodes->nodeTab[0];
    assert(node != NULL);
    if (node->type != XML_ELEMENT_NODE && node->type != XML_ATTRIBUTE_NODE)
        node = NULL;

finally:
    if (newxpath != NULL) free(newxpath);
    if (xpathObj != NULL) xmlXPathFreeObject(xpathObj);
    xml->xpathCtx->node = NULL;
    return node;
}

static void _xml_save(ato_Ctx *ctx, ato_Xml *xml, ato_String **buffer, bool trim)
{
    const char *function = "_xml_save";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    xmlChar *buf = NULL;
    char *chbuf = NULL;
    int size = 0;

    assert(xml != NULL);
    ATO_ASSERT_ISNOTALLOCATED(buffer);

    if (trim) {
        xmlKeepBlanksDefault(0);
        xmlDocDumpFormatMemory(xml->doc, &buf, &size, 1);
    } else {
        xmlDocDumpMemory(xml->doc, &buf, &size);
    }

    if (size == 0) {
        if (buf != NULL) { xmlFree(buf); buf = NULL; }
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_UNKNOWN, "Failed to save buffer");
    }

    chbuf = calloc(size + 1, sizeof(xmlChar));
    assert(chbuf != NULL);
    memcpy(chbuf, buf, size * sizeof(xmlChar));
    if (buf != NULL) { xmlFree(buf); buf = NULL; }
    //chbuf = (char *)buf;

    chbuf[size] = '\0';
    xml->isdirty = FALSE;
    ato_str_create(buffer, chbuf, size, TRUE);
}

void ato_xml_savetrim(ato_Ctx *ctx, ato_Xml *xml, ato_String **buffer)
{
    _xml_save(ctx, xml, buffer, TRUE);
}
void ato_xml_save(ato_Ctx *ctx, ato_Xml *xml, ato_String **buffer)
{
    _xml_save(ctx, xml, buffer, FALSE);
}

void ato_xml_savef(ato_Ctx *ctx, ato_Xml *xml, const char *filepath)
{
    const char *function = "ato_xml_savef";
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    int len = 0;
    int errcode = ATO_ERR_OK;

    assert(xml != NULL);
    len = xmlSaveFile(filepath, xml->doc);
//  {
//      ato_String *buffer = NULL;
//      ato_xml_save(ctx, xml, &buffer);
//      errcode = ato_filewrite(buffer, filepath);
//      ato_str_free(buffer);
//  }
    if (errcode != ATO_ERR_OK || len <= 0)
        Throw ATO_CTX_NEWERR(ctx, ATO_ERR_UNKNOWN, "Failed to save to file");
}

void ato_xml_deletenode(ato_Xml *xml, void *xnode)
{
    assert(xml != NULL);
    if (xnode != NULL) {
        xml->isdirty = TRUE;
        xmlUnlinkNode(xnode);
        xmlFreeNode(xnode);
    }
}
void ato_xml_dropdtd(ato_Xml *xml)
{
    xmlDtd *dtd = NULL;

    assert(xml != NULL);

    dtd = xmlGetIntSubset(xml->doc);
    if (dtd != NULL) {
        xml->isdirty = TRUE;
        xmlUnlinkNode((xmlNodePtr)dtd);
        xmlFreeDtd(dtd);
    }
}

static void _finalizenode(ato_Xml *xml, xmlNode *node, bool clearNs)
{
        if (clearNs && xml != NULL && node->ns != NULL) {
            //int count = 0;
            //xmlFreeNsList(node->ns);
          //if (node->nsDef == node->ns) node->nsDef = NULL;
            //node->ns = NULL;
            //count = xmlReconciliateNs(xml->doc, node);//xmlDocGetRootElement(xml->doc));
            //assert(count != -1);
        }
}

void *ato_xml_addnodecopy2(ato_Xml *xml, void *pxnode, const char *xpathExprForParent, void *node, bool clearNs)
{
    xmlNode *newnode = ato_xml_clonenode(xml, node, clearNs);
    if (newnode == NULL) return NULL;
    return ato_xml_addnode2(xml, pxnode, xpathExprForParent, newnode);
}

void *ato_xml_addnodecopy(ato_Xml *xml, const char *xpathExprForParent, void *node, bool clearNs)
{
    xmlNode *n = node;
    xmlNode *nodeParent = ato_xml_findnode(xml, NULL, xpathExprForParent, NULL);
    xmlNode *newnode = NULL;
    if (nodeParent == NULL) return NULL;
    if ((newnode = ato_xml_clonenode(xml, n, clearNs)) == NULL) return NULL;
    newnode = xmlAddChild(nodeParent, newnode);
    if (newnode != NULL) {
        xml->isdirty = TRUE;
        _finalizenode(xml, newnode, clearNs);
    }
    return newnode;
}
void *ato_xml_clonenode(ato_Xml *xml, void *node, bool clearNs)
{
    xmlNode *newnode = NULL;
    if (node != NULL) {
        newnode = xmlDocCopyNode(node, xml->doc, 1);
        //_finalizenode(xml, NULL, clearNs);
        ATO_IGNORE(clearNs);
    }
    return newnode;
}

void *ato_xml_addnode2(ato_Xml *xml, void *pxnode, const char *xpathExprForParent, void *node)
{
    xmlNode *newnode = NULL;
    xmlNode *nodeParent = NULL;
    if (xpathExprForParent == NULL)
        nodeParent = pxnode;
    else
        nodeParent = ato_xml_findnode(xml, pxnode, xpathExprForParent, NULL);
    if (nodeParent == NULL) return NULL;
    newnode = xmlAddChild(nodeParent, node);
    if (newnode != NULL) {
        xml->isdirty = TRUE;
    }
    return newnode;
}

void *ato_xml_addnode(ato_Xml *xml, const char *xpathExprForParent, void *node)
{
    xmlNode *newnode = NULL;
    xmlNode *nodeParent = ato_xml_findnode(xml, NULL, xpathExprForParent, NULL);
    if (nodeParent == NULL) return NULL;
    newnode = xmlAddChild(nodeParent, node);
    if (newnode != NULL)
        xml->isdirty = TRUE;
    return newnode;
}

bool ato_xml_addattr(ato_Xml *xml, void *node, const char *name, const char *value)
{
    xmlAttr *attr = NULL;
    const char *s = name;
    assert(name != NULL); assert(value != NULL);

    if (s[0] == '@') s++;
    attr = xmlNewProp(node, (const xmlChar*)s, (const xmlChar*)value);
    if (attr != NULL)
        xml->isdirty = TRUE;

    return attr != NULL;
}

void *ato_xml_addelt(ato_Xml *xml, void *pnode, const char *name)
{
    const char *s = strchr(name, ':');
    xmlNode *newnode = NULL;
    assert(pnode != NULL);

    if (s != NULL) s++;
    else s = name;
    newnode = xmlNewChild(pnode, NULL, (const xmlChar *)s, NULL);
    xml->isdirty = TRUE;
    return newnode;
}

void *ato_xml_createelt(ato_Xml *xml, void *pnode, const char *name)
{
    xmlNode *newnode = NULL;
    assert(pnode != NULL);

    newnode = ato_xml_findnode(xml, pnode, name, NULL);
    if (newnode == NULL) {
        newnode = ato_xml_addelt(xml, pnode, name);
    }
    return newnode;
}

void *ato_xml_createattr(ato_Xml *xml, void *pnode, const char *name, const char *prefix, const char *href)
{
    xmlNode *newnode = NULL;
    assert(pnode != NULL);

    assert(*name == '@');
    newnode = ato_xml_findnode(xml, pnode, name, NULL);
    if (newnode == NULL) {
        xmlNs *ns = NULL;
        xmlAttr *attr = NULL;
        const char *s = strchr(name, ':');
        if (s != NULL) s++;
        else {
            s = strchr(name, '@');
            if (s != NULL) s++;
            else s = name;
        }

        if (href) {
            ns = xmlNewNs(pnode, (const xmlChar *)href, (const xmlChar *)prefix);
            attr = xmlNewNsProp(pnode, ns, (const xmlChar *)s, (const xmlChar *)"");
        } else {
            attr = xmlNewProp(pnode, (const xmlChar *)s, (const xmlChar *)"");
        }
        assert(attr != NULL);
        xml->isdirty = TRUE;
        return attr;
    }
    return newnode;
}

bool ato_xml_isdirty(ato_Xml *xml)
{
    assert(xml != NULL);
    return xml->isdirty;
}

/*
cur = node->ns;
if (cur != NULL) {
    if (cur->prefix == NULL && nameSpace == NULL && cur->href != NULL)
        return cur;
    if (cur->prefix != NULL && nameSpace != NULL && cur->href != NULL && xmlStrEqual(cur->prefix, nameSpace))
        return cur;
}
*/

static xmlNs *_xmlSearchNsUp(xmlDoc *doc, xmlNode *node, const xmlChar *nsprefix)
{
    if (!node) return NULL;
    if (node->ns && node->ns->prefix && xmlStrEqual(node->ns->prefix, nsprefix))
        return node->ns;
    if (node->nsDef && node->nsDef->prefix && xmlStrEqual(node->nsDef->prefix, nsprefix))
        return node->nsDef;
    {
    xmlNs *ns = NULL;
    if (node->parent && (ns = _xmlSearchNsUp(doc, node->parent, nsprefix)) != NULL)
        return ns;
    }
    return NULL;
}

char *ato_xml_findnamespace(ato_Xml *xml, void *node, const char *nsprefix, char **value)
{
    xmlNs *ns = NULL;
    assert(xml != NULL);
    ATO_IGNORE(xml);
    ATO_ASSERT_ISNOTALLOCATED(value);

    if (ato_isnullorempty(nsprefix))
        return *value = NULL;

    if (node == NULL)
        node = xmlDocGetRootElement(xml->doc);
    ns = _xmlSearchNsUp(xml->doc, node, (const xmlChar *)nsprefix);
    if (ns)
        *value = _copy_xc2c(NULL, 0, ns->href);
    return *value;
}

void *ato_xml_nodeparent(ato_Xml *xml, void *xnode)
{
    xmlNode *node = xnode;
    ATO_IGNORE(xml);
    if (!node) return NULL;
    return node->parent;
}

char *ato_xml_root_namespace(ato_Xml *xml, char **value)
{
    xmlNode *root = xmlDocGetRootElement(xml->doc);
    xmlNs *ns = NULL;
    if(root == NULL) { return FALSE; }
    ns = root->ns;
    *value = _copy_xc2c(NULL, 0, ns->href);
    return *value;
}


/*********************************************************************************/
/*********************************************************************************/
/*
bool ato_xml_root_namespace_set(ato_Xml *xml, const char *value)
{
    xmlNs *ns = NULL;
    xmlNode *root = xmlDocGetRootElement(xml->doc);
    if(root == NULL) { return FALSE; }
    ns = root->ns;
    if (ns != NULL) { xmlFreeNsList(ns); ns = NULL; }
    xmlSetNs(root, ns);
    ns = xmlNewNs(root, (const xmlChar *)value, NULL);
    xmlSetNs(root, ns);
    xmlReconciliateNs(xml->doc, root);
    return TRUE;
}

bool matches(const xmlChar *pname, const xmlChar *nname, const char *pname2, const char *nname2)
{
    return xmlStrcmp(pname, (const xmlChar*) pname2) == 0
    || xmlStrcmp(nname, (const xmlChar*) nname2) == 0;
}
bool matches_value(const xmlChar *nname)
{
    return
        xmlStrcmp(nname, (const xmlChar*) "MessageID") == 0
        || xmlStrcmp(nname, (const xmlChar*) "Created") == 0
        || xmlStrcmp(nname, (const xmlChar*) "Expires") == 0
        || xmlStrcmp(nname, (const xmlChar*) "DigestValue") == 0
        || xmlStrcmp(nname, (const xmlChar*) "SignatureValue") == 0
        ;
}
bool matches_attr(const xmlChar *pname, const xmlChar *nname, const xmlChar *aname)
{
    return
        (xmlStrcmp(nname, (const xmlChar*) "BinarySecurityToken") == 0 && xmlStrcmp(aname, (const xmlChar*) "Id") == 0)
        || (xmlStrcmp(pname, (const xmlChar*) "SecurityTokenReference") == 0 && xmlStrcmp(nname, (const xmlChar*) "Reference") == 0 && xmlStrcmp(aname, (const xmlChar*) "URI") == 0)
        ;
}

void print_value(xmlDoc *doc, xmlNode *node)
{
    xmlChar *value = xmlNodeListGetString(doc, node->children, 1);
    printf("%s.%s", node->parent->name, node->name);
    if (value != NULL) {
        printf(" = %s\n", value);
        xmlFree(value);
    } else {
        printf("\n");
    }
}

void print_element(xmlDoc *doc, xmlNode *node)
{
    const xmlChar *parentname = node->parent->name;
    if (matches_value(node->name)) {
        print_value(doc, node);
        if (matches(parentname, node->name, "Header", "MessageID")) {
            char guid[100];
            strcpy(guid, "urn:uuid:");
            pseudo_guid(guid + strlen(guid));
            xmlNodeSetContent(node, (const xmlChar*) guid);
            print_value(doc, node);
        }
    } else {
        xmlAttr *attr = node->properties;
        for (attr = node->properties; attr != NULL; attr = attr->next) {
            if (matches_attr(parentname, node->name, attr->name)) {
                printf("%s.%s:%s = %s\n", parentname, node->name, attr->name, xmlGetProp(node, attr->name));
            }
        }
    }
}

void print_element_names(xmlDoc *doc, xmlNode *node)
{
    xmlNode *cur_node = NULL;
    for (cur_node = node; cur_node != NULL; cur_node = cur_node->next) {
        if (cur_node->type != XML_ELEMENT_NODE)
            continue;
        print_element(doc, cur_node);
        print_element_names(doc, cur_node->children);
    }
}

void ato__xml_dump(ato_Xml *xml)
{
    char guid[50];
    xmlNode *root_element = NULL;
    xmlImpl *impl = NULL;
    impl = xml;
    root_element = xmlDocGetRootElement(xml->doc);
    print_element_names(xml->doc, root_element);
    printf("\nguid = %s\n", pseudo_guid(guid));
}

bool ato__xml_stripids(ato_Xml *xml)
{
    size_t i = 0;
    xmlXPathObjectPtr xpathObj = NULL;
    int code = 0;

    xpathObj = _getnodes("// *[@xml:id]", "", xml->xpathCtx);
    if(xpathObj == NULL || _size(xpathObj->nodesetval) == 0)
        return FALSE;

    for (i = 0; i < _size(xpathObj->nodesetval); i++) {
        xmlNodePtr node = xpathObj->nodesetval->nodeTab[i];
        xmlNsPtr ns = xmlSearchNs(xml->doc, node, (const xmlChar*) "xml");
        code = xmlUnsetNsProp(node, ns, (const xmlChar*) "id");
    }

    return TRUE;
}

*/
