#ifndef __ATO_STM_H__
#define __ATO_STM_H__

/** @addtogroup atostm_api_stm

@{
*/

/**
The STM object used to interact with an STS service to retrieve Security Tokens (ato_StmTkn).
*/
typedef struct _ato_Stm ato_Stm;

#ifdef __cplusplus
extern "C" {
#endif

/**
Create an STM object from the XML stored in stmtemplate.

@param ctx the current thread local Context.
@param stm the address of the object to create.
@param stsurl the URL of the STS service to use. If NULL, the value is read from the current configuration.
@param stmtemplate the XML content.

@return ATO_ERR_OK or an errcode.
*/
ATO_STM_EXPORT int ato_stm_create(ato_Ctx *ctx, ato_Stm **stm, const char *stsurl, const char *stmtemplate);

/**
Free the STM object.

@param ctx the current thread local Context.
@param stm the STM object. If NULL do nothing.
*/
ATO_STM_EXPORT void ato_stm_free(ato_Ctx *ctx, ato_Stm *stm);

/**
If the template loaded by ato_stm_create() contains "variables", update them with runtime values.

Variable names in the template are defined by @@{varname}. The varname (without the @@{}) is used in the vars list.

The template itself as loaded by ato_stm_create() is not affected.

If this is called multiple times, it accumulates names/values in an internal list.
If a name already exists, it's value is replaced.
To clear the list, use NULL.
Any NULL names/values are ignored. Otherwise it is expected to be a list of pairs of variable name1, variable value1, ...
Variable names that do not exist or where the value is NULL, are ignored.

The name must match exactly (case-sensitive) what is in the template and must be <= 50 chars.

The last call determines the values used for subsequent calls to ato_stm_stsissue().

@param ctx the current thread local Context.
@param stm the STM object.
@param vars an array of name1, value1, name2, value2, ... If NULL clear the existing list.

@return ATO_ERR_OK or an errcode.
*/
ATO_STM_EXPORT int ato_stm_set_template_vars(ato_Ctx *ctx, ato_Stm *stm, const char * const *vars);

/**
Same as ato_stm_set_template_vars() except use a comma delimited string.

@param ctx the current thread local Context.
@param stm the STM object.
@param varstr a string of comma delimited chars e.g. name1,value1,name2,value2, ... If NULL clear the existing list.

@return ATO_ERR_OK or an errcode.
*/
ATO_STM_EXPORT int ato_stm_set_template_varstr(ato_Ctx *ctx, ato_Stm *stm, const char *varstr);

/**
Purge the STM token cache for the current context/thread.

@param ctx the current thread local Context.
@param purgeall if TRUE remove all tokens, otherwise remove only stale (expired) tokens.
*/
ATO_STM_EXPORT void ato_stm_purgecache(ato_Ctx *ctx, bool purgeall);

/**
Set the expiry interval (mins) for requested SecurityTokens at time of request - see also ato_stmtkm_isexpired() and ato_stmtkn_expirytime().

This is applied to all subsequent requests. if not explicitly set, the default is 5 minutes.
The allowed ranged is 5 to 30 mins. If less than 5 mins is specified, then 5 mins is set; if greater than
30 mins is specified, then 30 mins is set.

@param stm the STM object.
@param expirymins the number of minutes from the request time that the SecurityToken should remain valid.
*/
ATO_STM_EXPORT void ato_stm_setexpiryinterval(ato_Stm *stm, size_t expirymins);

/**
The current expiry time interval in minutes - see also ato_stm_setexpiryinterval()

This is applied to requests (ato_stm_stsissue()).
The SecurityToken is set to expired at the time of the request + specified number of minutes.

@param stm the STM object.

@return the number of minutes from the request time that the SecurityToken should remain valid.
*/
ATO_STM_EXPORT size_t ato_stm_expiryinterval(ato_Stm *stm);

/**
Submit an STS issue request to the STS service and return a SecurityToken response object.

The Security Token expiry time is set to a default value unless set using ato_stm_setexpirytime().
If the SecurityToken already exists int the cache and is not expired, then this is used instead,
and no call is made to the STS service.

@param ctx the current thread local Context.
@param stm the STM object.
@param st the SecurityToken object address (do not free) or NULL if none or an error occurred.
@param certificate X509 certificate.
@param privatekey P8 private key.
@param relyingpartyurl the relying party for which the token is issued.

@return ATO_ERR_OK or an errcode.
*/
ATO_STM_EXPORT int ato_stm_stsissue(ato_Ctx *ctx, ato_Stm *stm, ato_StmTkn **st, ato_String *certificate, ato_String *privatekey, const char *relyingpartyurl);

/**
Deserialise a SecurityToken (ST) based on the contents of a previously serialised ST
- see ato_stmtkn_xml().

This creates a corresponding ST object in the ST cache.
An error results if the ST already exists in the cache.
However, expired STs are accepted, so check ato_stmtkn_isexpired() before using the ST.

@param ctx the current thread local Context.
@param stm the STM object.
@param st the SecurityToken object address (do not free) or NULL if none or an error occurred.
@param atorstr Serialised ST content which includes the RSTR and stkey.

@return ATO_ERR_OK or an errcode.
*/
ATO_STM_EXPORT int ato_stm_loadst(ato_Ctx *ctx, ato_Stm *stm, ato_StmTkn **st, ato_String *atorstr);

/**
Get the network total timeout value, in seconds, used for requests to Vanguard STS.

@param ctx the current thread local Context.
@param stm the STM object.

@return the current timeout value.
*/
ATO_STM_EXPORT size_t ato_stm_timeout(ato_Ctx *ctx, ato_Stm *stm);

/**
Set the network total timeout value, in seconds, used for requests to Vanguard STS.

@param ctx the current thread local Context.
@param stm the STM object.
@param timeout the timeout value to set.
*/
ATO_STM_EXPORT void ato_stm_settimeout(ato_Ctx *ctx, ato_Stm *stm, size_t timeout);

/**
Get the network connect timeout value, in seconds, used for requests to Vanguard STS.

@param ctx the current thread local Context.
@param stm the STM object.

@return the current timeout value.
*/
ATO_STM_EXPORT size_t ato_stm_timeoutconnect(ato_Ctx *ctx, ato_Stm *stm);

/**
Set the network connect timeout value, in seconds, used for requests to Vanguard STS.

@param ctx the current thread local Context.
@param stm the STM object.
@param timeout the timeout value to set.
*/
ATO_STM_EXPORT void ato_stm_settimeoutconnect(ato_Ctx *ctx, ato_Stm *stm, size_t timeout);

#ifdef __cplusplus
}
#endif

/*! @} */

#endif /*__ATO_STM_H__*/
