#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"
#include "atobnet/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intcsr.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_CSR;
static unsigned long _moduleid = SBR_CSR_MODULEID_CSR;
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
struct _sbr_Csr {
    size_t timeout;
    size_t timeoutconnect;
    char *url;
    char *servicename;
    char *servicenamespace;
    char *request_servicename;
    char *response_servicename;
    ato_String *xtemplate;
};

/*********************************************************************************/
static void _getcfgvalue(ato_Ctx *ctx, char **value, const char *servicetype, const char *name, const char *defvalue)
{
    char cfgname[1000];
    const char *cfgvalue = NULL;
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_getcfgvalue";

    strcat(strcpy(cfgname, name), servicetype);
    cfgvalue = ato_arg_value(cfgname);
    if (cfgvalue == NULL) { cfgvalue = defvalue; }

    if (cfgvalue != NULL) {
        *value = ato_strdup(cfgvalue, 0);
    } else {
        Throw ATO_CTX_VNEWERR(ctx, ATO_ERR_CFGNOTFOUND, strlen(cfgname),
            "Failed to find value for context item '%s'.", cfgname);
    }
}

// ...part1...<RequestXXX>....part2....</RequestXXX>...part3
static char *_patchtemplate_servicename(ato_Ctx *ctx, const char *srctemplate, const char *servicename, const char *servicenamespace)
{
    static const char *RequestXXXstart = "<RequestXXX>";
    static const char *RequestXXXend = "</RequestXXX>";
    char *dsttemplate = NULL, *dst = NULL;
    size_t dstlen = 0;
    const char *reqstarttag = NULL, *reqendtag = NULL;
    const char *part1 = NULL, *part2 = NULL, *part3 = NULL;
    size_t part1_len = 0, part2_len = 0, part3_len = 0;

    ATO_IGNORE(ctx);
    assert(srctemplate != NULL);

    part1 = srctemplate;
    reqstarttag = strstr(part1, RequestXXXstart);
    assert(reqstarttag != NULL);
    part2 = strchr(reqstarttag, '>');
    assert(part2 != NULL);
    part2++;
    reqendtag = strstr(part2, RequestXXXend);
    assert(reqendtag != NULL);
    part3 = strchr(reqendtag, '>');
    assert(part3 != NULL);
    part3++;

    assert(reqstarttag - part1 > 0);
    assert(reqendtag - part2 > 0);

    part1_len = (size_t) (reqstarttag - part1);
    part2_len = (size_t) (reqendtag - part2);
    part3_len = strlen(part3);


    dstlen = part1_len +
        strlen(servicename) + strlen(servicenamespace) + 20 +
        part2_len +
        strlen(servicename) + 20 +
        part3_len;
    dst = dsttemplate = calloc(dstlen+1, sizeof(char));
    assert(dst != NULL);
    assert(dsttemplate != NULL);

    strncpy(dst, part1, part1_len);

    dst = dst + part1_len;
    strcpy(dst, "<Request");
    strcat(dst, servicename);
    strcat(dst, " xmlns=\"");
    strcat(dst, servicenamespace);
    strcat(dst, "\">");

    dst = dst + strlen(dst);
    strncpy(dst, part2, part2_len);

    dst = dst + strlen(dst);
    strcpy(dst, "</Request");
    strcat(dst, servicename);
    strcat(dst, ">");

    strcat(dst, part3);
    assert(strlen(dsttemplate) <= dstlen);
    return dsttemplate;
}

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
int sbr__csr_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void sbr__csr_deinit(void)
{
}

/*********************************************************************************/
void sbr__csr_create(ato_Ctx *ctx, sbr_Csr **obj, const char *servicetype, const char *csrurl, const char *csrtemplate)
{
    const char *function = "sbr__csr_create";
    int errcode = ATO_ERR_OK;
    sbr_Csr *csr = NULL;
    char *patchedtemplate = NULL;
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);
    assert(ato_isnullorempty(servicetype) == FALSE);
    assert(ato_isnullorempty(csrtemplate) == FALSE);

    *obj = csr = calloc(1, sizeof(sbr_Csr));
    assert(csr != NULL);

    csr->timeout = ato_arg_value_int("sbr.timeout");
    if (csr->timeout == 0)
        csr->timeout = ato_bnet_timeout_default(ctx);
    csr->timeoutconnect = ato_arg_value_int("sbr.timeout.connect");
    if (csr->timeoutconnect == 0)
        csr->timeoutconnect = ato_bnet_timeoutconnect_default(ctx);

    _getcfgvalue(ctx, &(csr->url), servicetype, "sbr.service.url.", csrurl);
    _getcfgvalue(ctx, &(csr->servicename), servicetype, "sbr.service.name.", NULL);
    _getcfgvalue(ctx, &(csr->servicenamespace), servicetype, "sbr.service.namespace.", NULL);
    patchedtemplate = _patchtemplate_servicename(ctx, csrtemplate, csr->servicename, csr->servicenamespace);
    ato_str_create(&(csr->xtemplate), patchedtemplate, strlen(patchedtemplate), TRUE);

    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
int sbr_csr_create(ato_Ctx *ctx, sbr_Csr **csr, const char *servicetype, const char *csrurl, const char *csrtemplate)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_csr_create";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        sbr__csr_create(ctx, csr, servicetype, csrurl, csrtemplate);
    } Catch (errcode) {
        errcode = sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

void sbr_csr_free(sbr_Csr *csr)
{
    if (csr == NULL) return;

    if (csr->url != NULL) free(csr->url);
    if (csr->servicenamespace != NULL) free(csr->servicenamespace);
    if (csr->servicename != NULL) free(csr->servicename);
    if (csr->request_servicename != NULL) free(csr->request_servicename);
    if (csr->response_servicename != NULL) free(csr->response_servicename);

    ato_str_free(csr->xtemplate);
    free(csr);
}

size_t sbr_csr_timeout(ato_Ctx *ctx, sbr_Csr *csr)
{
    ATO_IGNORE(ctx);
    return csr->timeout;
}

void sbr_csr_settimeout(ato_Ctx *ctx, sbr_Csr *csr, size_t timeout)
{
    ATO_IGNORE(ctx);
    csr->timeout = timeout;
}

size_t sbr_csr_timeoutconnect(ato_Ctx *ctx, sbr_Csr *csr)
{
    ATO_IGNORE(ctx);
    return csr->timeoutconnect;
}

void sbr_csr_settimeoutconnect(ato_Ctx *ctx, sbr_Csr *csr, size_t timeout)
{
    ATO_IGNORE(ctx);
    csr->timeoutconnect = timeout;
}

const char *sbr_csr_url(ato_Ctx *ctx, sbr_Csr *csr)
{
    ATO_IGNORE(ctx);
    return csr->url;
}

ato_String *sbr_csr_template(ato_Ctx *ctx, sbr_Csr *csr)
{
    ATO_IGNORE(ctx);
    return csr->xtemplate;
}

const char *sbr_csr_servicenamespace(ato_Ctx *ctx, sbr_Csr *csr)
{
    ATO_IGNORE(ctx);
    return csr->servicenamespace;
}
const char *sbr_csr_servicename(ato_Ctx *ctx, sbr_Csr *csr)
{
    ATO_IGNORE(ctx);
    return csr->servicename;
}
