#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intmsgparty.h"
#include "intdoca.h"
#include "intdoc.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_DOC;
static unsigned long _moduleid = SBR_CSR_MODULEID_DOC;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;

static const char *xBusinessDocuments = ".//data:BusinessDocuments";
static const char *xBusinessDocument = "./data:BusinessDocument";
static const char *xSequenceNumber = "data:BusinessDocument.Sequence.Number";
static const char *xCreationDatetime = "data:BusinessDocument.Creation.Datetime";
static const char *xValidationUniformResourceIdentifier = "data:BusinessDocument.ValidationUniformResourceIdentifier.Text";
static const char *xBusinessGeneratedIdentifier = "data:BusinessDocument.BusinessGeneratedIdentifier.Text";
static const char *xGovernmentGeneratedIdentifier = "data:BusinessDocument.GovernmentGeneratedIdentifier.Text";

static const char *xBusinessDocumentInstances = ".//data:BusinessDocumentInstances";
static const char *xBusinessDocumentInstance = "./data:BusinessDocumentInstance";
static const char *xBusinessDocumentInstance_seqnr = "./data:BusinessDocumentInstance[data:BusinessDocument.Sequence.Number=\"$\"]";
static const char *xText = "data:BusinessDocument.Instance.Text";

/*********************************************************************************/
struct _sbr_Doc {
    sbr_eMsgSourceType msgsrctype;
    char guid[50];
    char *sequencenr;
    char *contentvalue;
    ato_String *content;
    char *validationuri;
    char *governmentid;
    char *businessid;
    char *createtime;
    sbr_DocAs *attachments;
    ato_Xml *xml;
    void *xnode;
    void *xnodebody;
    size_t *attachmentcount;
};

struct _sbr_Docs {
    sbr_eMsgSourceType msgsrctype;
    ato_List *itemlist;
    ato_Iterator *iter;
    size_t *attachmentcount;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void _freecontent(sbr_Doc *doc)
{
    doc->content = ato_str_free(doc->content);
}

static void _freeitem(sbr_Doc *doc)
{
    if (doc == NULL) return;

    doc->contentvalue = ato_free(doc->contentvalue);
    doc->validationuri = ato_free(doc->validationuri);
    doc->governmentid = ato_free(doc->governmentid);
    doc->businessid = ato_free(doc->businessid);
    doc->createtime = ato_free(doc->createtime);
    doc->sequencenr = ato_free(doc->sequencenr);
    _freecontent(doc);

    sbr__doca_free(doc->attachments);
    free(doc);
}

static void _process_doc_body(ato_Ctx *ctx, ato_Xml *xml, void *pnode, ato_String **content)
{
    char *text = NULL;
    ato_xml_nodevalue(xml, pnode, xText, &text, ATO_XML_RAW_INNER);
    ato_str_create(content, text, strlen(text), TRUE);
    ATO_IGNORE(ctx);
}

static void _createitem(ato_Ctx *ctx, sbr_Doc **obj, sbr_eMsgSourceType msgsrctype, ato_Xml *xml, void *xnode, void *xnodebody)
{
    sbr_Doc *doc = NULL;

    assert(ctx != NULL);
    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = doc = calloc(1, sizeof(sbr_Doc));
    assert(doc != NULL);
    ato_new_pseudo_guid(ctx, doc->guid);
    doc->msgsrctype = msgsrctype;
    doc->xml = xml;
    doc->xnode = xnode;

    if (xnode) {
        ato_xml_nodevalue(xml, xnode, xSequenceNumber, &(doc->sequencenr), FALSE);
        doc->xnodebody = ato_xml_findnode(xml, xnodebody, xBusinessDocumentInstance_seqnr, doc->sequencenr);
        assert(doc->xnodebody != NULL);
        ato_xml_nodevalue(xml, xnode, xCreationDatetime, &(doc->createtime), FALSE);
        ato_xml_nodevalue(xml, xnode, xValidationUniformResourceIdentifier, &(doc->validationuri), FALSE);
        ato_xml_nodevalue(xml, xnode, xBusinessGeneratedIdentifier, &(doc->businessid), FALSE);
        ato_xml_nodevalue(xml, xnode, xGovernmentGeneratedIdentifier, &(doc->governmentid), FALSE);
        _process_doc_body(ctx, xml, doc->xnodebody, &(doc->content));
    }
}

/*********************************************************************************/
int sbr__doc_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void sbr__doc_deinit(void)
{
}

void sbr__doc_free(sbr_Docs *docs)
{
    if (docs == NULL) return;

    {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(docs->itemlist, node)) != NULL)
            _freeitem(ato_lst_value(node));
    }
    ato_lst_free(docs->itemlist);
    ato_iter_free(docs->iter);
    free(docs);
}

void sbr__doc_create(ato_Ctx *ctx, sbr_Docs **obj, sbr_eMsgSourceType msgsrctype,
                                        ato_Xml *xml, void *pxnode, size_t *attachmentcount)
{
    const char *function = "sbr__doc_create";
    int errcode = ATO_ERR_OK;
    sbr_Docs *docs = NULL;
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);

    if (!xml || !pxnode) {
        *obj = docs = calloc(1, sizeof(sbr_Docs));
        assert(docs != NULL);
        docs->msgsrctype = msgsrctype;
        ato_lst_create(&(docs->itemlist));
        docs->iter = ato_iter_create_lst(docs->itemlist);
        docs->attachmentcount = attachmentcount;
    } else {
        void *xnode = NULL, *xnodebody = NULL;
        xnode = ato_xml_findnode(xml, pxnode, xBusinessDocuments, NULL);
        xnodebody = ato_xml_findnode(xml, pxnode, xBusinessDocumentInstances, NULL);
        if (xnode && xnodebody) {
            void *xlist = NULL;

            *obj = docs = calloc(1, sizeof(sbr_Docs));
            assert(docs != NULL);
            docs->msgsrctype = msgsrctype;
            ato_lst_create(&(docs->itemlist));
            docs->iter = ato_iter_create_lst(docs->itemlist);
            docs->attachmentcount = attachmentcount;

            xlist = ato_xml_listload(xml, xnode, xBusinessDocument, NULL);
            if (xlist != NULL) {
                size_t i = 0, count = 0;

                count = ato_xml_listcount(xml, xlist);
                for (i = 0; i < count; i++) {
                    void *psubnode = NULL;
                    sbr_Doc *item = NULL;
                    psubnode = ato_xml_listitem(xml, xlist, i);
                    assert(psubnode != NULL);
                    _createitem(ctx, &item, msgsrctype, xml, psubnode, xnodebody);
                    ato_lst_add(docs->itemlist, item->guid, item);
                    sbr__doca_create(ctx, &(item->attachments), msgsrctype, xml, psubnode, pxnode, attachmentcount);
                }
                ato_xml_listfree(xml, xlist);
            }
        }
    }

    ATO_CTX_FN_END(ctx, errcode);
}

size_t sbr__doc_count(sbr_Docs *docs)
{
    if (docs == NULL) return 0;
    return ato_lst_count(docs->itemlist);
}
void sbr__doc_iterator(sbr_Docs *docs, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (docs == NULL) return;
    *iter = ato_iter_create_lst(docs->itemlist);
}
sbr_Doc *sbr__doc_firstv(sbr_Docs *docs)
{
    if (docs == NULL) return NULL;
    return ato_iter_firstv(docs->iter);
}
sbr_Doc *sbr__doc_nextv(sbr_Docs *docs)
{
    if (docs == NULL) return NULL;
    return ato_iter_nextv(docs->iter);
}

sbr_Doc *sbr__doc_add(ato_Ctx *ctx, sbr_Docs *docs)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "sbr__doc_add";
    ato_ListNode *node = NULL;
    sbr_Doc *doc = NULL;
    char buf[50];

    assert(ctx != NULL); assert(docs != NULL);
    _createitem(ctx, &doc, docs->msgsrctype, NULL, NULL, NULL);
    node = ato_lst_add(docs->itemlist, doc->guid, doc);
    if (node == NULL)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Error in adding Document into SBDM.");
    doc->sequencenr = ato_strdup(ato_itoa(buf, (int)sbr__doc_count(docs)), 0);
    doc->attachmentcount = docs->attachmentcount;
    return doc;
}

void sbr__doc_remove(ato_Ctx *ctx, sbr_Docs *docs, sbr_Doc *doc)
{
    ato_ListNode *node = ato_lst_find(docs->itemlist, sbr_doc_iid(doc));
    if (node != NULL) {
        _freeitem(ato_lst_value(node));
        ato_lst_delete(node);
    }
    ATO_IGNORE(ctx);
}

static void _render_new(ato_Ctx *ctx, sbr_Doc *doc, ato_Xml *xml, void *pxnode)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_render_new";
    const char *emsg = "Failed to render new document - can't find %s.";
    void *docsnode = NULL;
    assert(xml != NULL); assert(pxnode != NULL);
    docsnode = ato_xml_findnode(xml, pxnode, xBusinessDocuments, NULL);
    if (!docsnode)
        Throw ATO_CTX_VNEWERR(ctx, SBR_CSR_ERR_GENERAL, strlen(emsg), emsg, xBusinessDocuments);
    doc->xnode = ato_xml_addelt(xml, docsnode, xBusinessDocument);
    if (!doc->xnode)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render new document - can't add document.");

    docsnode = ato_xml_findnode(xml, pxnode, xBusinessDocumentInstances, NULL);
    if (!docsnode)
        Throw ATO_CTX_VNEWERR(ctx, SBR_CSR_ERR_GENERAL, strlen(emsg), emsg, xBusinessDocumentInstances);
    doc->xnodebody = ato_xml_addelt(xml, docsnode, xBusinessDocumentInstance);
    if (!doc->xnodebody)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render new document - can't add document body.");
}

void sbr__doc_render(ato_Ctx *ctx, sbr_Docs *docs, ato_Xml *xml, void *pxnode)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "sbr__doc_render";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    assert(ctx != NULL);
    if (docs) {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(docs->itemlist, node)) != NULL) {
            sbr_Doc *doc = ato_lst_value(node);
            if (!doc->xnode) {
                _render_new(ctx, doc, xml, pxnode);
            }
            if (!doc->xnode)
                Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render document.");
            if (!doc->xnodebody)
                Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render document body.");
            doc->xml = xml;
            ato_xml_createnodevalue(doc->xml, doc->xnode, NULL, xSequenceNumber, doc->sequencenr);
            ato_xml_createnodevalue(doc->xml, doc->xnode, NULL, xCreationDatetime, doc->createtime);
            ato_xml_createnodevalue(doc->xml, doc->xnode, NULL, xValidationUniformResourceIdentifier, doc->validationuri);
            ato_xml_createnodevalue(doc->xml, doc->xnode, NULL, xBusinessGeneratedIdentifier, doc->businessid);
            ato_xml_createnodevalue(doc->xml, doc->xnode, NULL, xGovernmentGeneratedIdentifier, doc->governmentid);
            ato_xml_createnodevalue(doc->xml, doc->xnodebody, NULL, xSequenceNumber, doc->sequencenr);
            ato_xml_createnodevalueraw(ctx, doc->xml, doc->xnodebody, NULL, xText, ato_str_value(doc->content));
            sbr__doca_render(ctx, doc->attachments, xml, doc->xnode, pxnode);
        }
    }

    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
const char *sbr_doc_iid(sbr_Doc *doc)
{
    assert(doc != NULL);
    return doc->guid;
}

const char *sbr_doc_content(sbr_Doc *doc)
{
    assert(doc != NULL);
    return ato_str_value(doc->content);
}

const char *sbr_doc_governmentid(sbr_Doc *doc)
{
    assert(doc != NULL);
    return doc->governmentid;
}

const char *sbr_doc_businessid(sbr_Doc *doc)
{
    assert(doc != NULL);
    return doc->businessid;
}

const char *sbr_doc_createtime(sbr_Doc *doc)
{
    assert(doc != NULL);
    return doc->createtime;
}

const char *sbr_doc_validationuri(sbr_Doc *doc)
{
    assert(doc != NULL);
    return doc->validationuri;
}

size_t sbr_doc_sequencenr(sbr_Doc *doc)
{
    assert(doc != NULL);
    return (size_t)strtol(doc->sequencenr, NULL, 10);
}

sbr_eMsgSourceType sbr_doc_msgsourcetype(sbr_Doc *doc)
{
    assert(doc != NULL);
    return doc->msgsrctype;
}

int sbr_doc_setcontent(ato_Ctx *ctx, sbr_Doc *doc, const char *content, ato_eParamAction contentaction)
{
    assert(ctx != NULL); assert(doc != NULL);

    _freecontent(doc);
    if (content) {
        switch (contentaction) {
            case ATO_PARAMACTION_CONST:
                ato_str_createconst(&(doc->content), content, strlen(content), TRUE);
                break;
            case ATO_PARAMACTION_FREE:
                assert(contentaction == ATO_PARAMACTION_FREE);
                break;
            default:
            //case ATO_PARAMACTION_DEFAULT:
            //case ATO_PARAMACTION_COPY:
                ato_str_create(&(doc->content), ato_strdup(content, 0), strlen(content), TRUE);
                break;
        }
    }

    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doc_setcontentref(ato_Ctx *ctx, sbr_Doc *doc, char *content, ato_eParamAction contentaction)
{
    assert(ctx != NULL); assert(doc != NULL);

    _freecontent(doc);
    if (content) {
        switch (contentaction) {
            case ATO_PARAMACTION_CONST:
                ato_str_createconst(&(doc->content), content, strlen(content), TRUE);
                break;
            case ATO_PARAMACTION_FREE:
                ato_str_create(&(doc->content), content, strlen(content), TRUE);
                assert(contentaction == ATO_PARAMACTION_FREE);
                break;
            case ATO_PARAMACTION_DEFAULT:
            case ATO_PARAMACTION_COPY:
                ato_str_create(&(doc->content), ato_strdup(content, 0), strlen(content), TRUE);
                break;
            default:
                assert(contentaction);
        }
    }

    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doc_setvalidationuri(ato_Ctx *ctx, sbr_Doc *doc, const char *value)
{
    assert(ctx != NULL); assert(doc != NULL);
    ato_free(doc->validationuri);
    doc->validationuri = ato_strdup(value, 0);
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doc_setgovernmentid(ato_Ctx *ctx, sbr_Doc *doc, const char *value)
{
    assert(ctx != NULL); assert(doc != NULL);
    ato_free(doc->governmentid);
    doc->governmentid = ato_strdup(value, 0);
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doc_setbusinessid(ato_Ctx *ctx, sbr_Doc *doc, const char *value)
{
    assert(ctx != NULL); assert(doc != NULL);
    ato_free(doc->businessid);
    doc->businessid = ato_strdup(value, 0);
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doc_setcreatetime(ato_Ctx *ctx, sbr_Doc *doc, time_t value)
{
    assert(ctx != NULL); assert(doc != NULL);
    doc->createtime = ato_free(doc->createtime);
    if (value) {
        char createtimestr[101];
        struct tm t;
        ato_gmtime(&value, &t);
        strftime(createtimestr, 100, "%Y-%m-%dT%H:%M:%S.000000Z", &t);
        doc->createtime = ato_strdup(createtimestr, 0);
    }
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

size_t sbr_doc_attachment_count(sbr_Doc *doc)
{
    assert(doc != NULL);
    return sbr__doca_count(doc->attachments);
}
void sbr_doc_attachment_iterator(sbr_Doc *doc, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (doc == NULL) return;
    sbr__doca_iterator(doc->attachments, iter);
}
sbr_DocA *sbr_doc_attachment_firstv(sbr_Doc *doc)
{
    if (doc == NULL) return NULL;
    return sbr__doca_firstv(doc->attachments);
}
sbr_DocA *sbr_doc_attachment_nextv(sbr_Doc *doc)
{
    if (doc == NULL) return NULL;
    return sbr__doca_nextv(doc->attachments);
}

int sbr_doc_attachment_add(ato_Ctx *ctx, sbr_Doc *doc, sbr_DocA **attachment,
                                               const char *description,
                                               const char *filename,
                                               const char *contenttype
                                               )
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_doc_attachment_add";
    int errcode = ATO_ERR_OK;
    assert(ctx != NULL); assert(doc != NULL);
    if (doc->msgsrctype == SBR_RESPONSE)
        return -1;
    ATO_CTX_FN_START(ctx);
    Try {
        if (!doc->attachments)
            sbr__doca_create(ctx, &(doc->attachments), doc->msgsrctype, NULL, NULL, NULL, doc->attachmentcount);
        *attachment = sbr__doca_add(ctx, doc->attachments);
        //sbr_doca_setcontent(ctx, *attachment, content);
        sbr_doca_setdescription(ctx, *attachment, description);
        sbr_doca_setfilename(ctx, *attachment, filename);
        sbr_doca_setcontenttype(ctx, *attachment, contenttype);
    } Catch (errcode) {
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

/*
int sbr_doc_attachment_remove(ato_Ctx *ctx, sbr_Doc *doc, sbr_DocA *attachment)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_doc_attachment_remove";
    int errcode = ATO_ERR_OK;
    assert(ctx != NULL); assert(doc != NULL);
    if (doc->msgsrctype == SBR_RESPONSE)
        return -1;
    ATO_CTX_FN_START(ctx);
    Try {
        sbr__doca_remove(ctx, doc->attachments, attachment);
    } Catch (errcode) {
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}*/

