#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intmsgparty.h"
#include "intdoca.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_DOCA;
static unsigned long _moduleid = SBR_CSR_MODULEID_DOCA;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;

static const char *xAttachments = "./data:Attachments";
static const char *xAttachment = "./data:Attachment";
static const char *xSequenceNumber = "data:Message.Attachment.Sequence.Number";
static const char *xFileName = "data:Message.Attachment.FileName.Text";
static const char *xDescription = "data:Message.Attachment.Description.Text";

static const char *xStandardBusinessDocumentBody = ".//data:StandardBusinessDocumentBody";
static const char *xAttachmentInstances = "./data:AttachmentInstances";
static const char *xAttachmentInstance = "./data:AttachmentInstance";
static const char *xAttachmentInstance_seqnr = "./data:AttachmentInstance[data:Message.Attachment.Sequence.Number=\"$\"]";
static const char *xText = "data:Message.Attachment.Instance.BinaryObject";
//static const char *xContentType = "data:Message.Attachment.Instance.BinaryObject/@mimetype:contentType";
static const char *xContentType = "@nsmime:contentType";
/*********************************************************************************/

struct _sbr_DocA {
    sbr_eMsgSourceType msgsrctype;
    char guid[50];
    char *sequencenr;
    bool contentfree;
    ato_String *content;
    char *description;
    char *filename;
    char *contenttype;
    ato_Xml *xml;
    void *xnode;
    void *xnodebody;
};

struct _sbr_DocAs {
    sbr_eMsgSourceType msgsrctype;
    ato_List *itemlist;
    ato_Iterator *iter;
    size_t *attachmentcount;
};
/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void _freecontent(sbr_DocA *attachment)
{
    if (attachment->contentfree)
        ato_str_free(attachment->content);
    attachment->contentfree = FALSE;
    attachment->content = NULL;
}

static void _freeitem(sbr_DocA *attachment)
{
    if (attachment == NULL) return;

    attachment->description = ato_free(attachment->description);
    attachment->filename = ato_free(attachment->filename);
    attachment->contenttype = ato_free(attachment->contenttype);
    attachment->sequencenr = ato_free(attachment->sequencenr);
    _freecontent(attachment);

    free(attachment);
}

static void _process_doca_body(ato_Ctx *ctx, ato_Xml *xml, void *pnode, ato_String **content, char **contenttype)
{
    char *text = NULL;
    void *xnode = ato_xml_findnode(xml, pnode, xText, NULL);
    ato_xml_nodevalue(xml, xnode, xContentType, contenttype, ATO_XML_RAW_FALSE);
    ato_xml_nodevalue(xml, pnode, xText, &text, ATO_XML_RAW_FALSE);
    ato_str_create(content, text, strlen(text), TRUE);
    ATO_IGNORE(ctx);
}

static void _createitem(ato_Ctx *ctx, sbr_DocA **obj, sbr_eMsgSourceType msgsrctype, ato_Xml *xml, void *xnode, void *xnodebody, size_t *attachmentcount)
{
    sbr_DocA *attachment = NULL;

    assert(ctx != NULL);
    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = attachment = calloc(1, sizeof(sbr_DocA));
    assert(attachment != NULL);
    ato_new_pseudo_guid(ctx, attachment->guid);
    attachment->msgsrctype = msgsrctype;
    attachment->xml = xml;
    attachment->xnode = xnode;
    (*attachmentcount)++;

    if (xnode) { // loading an existing node
        ato_xml_nodevalue(xml, xnode, xSequenceNumber, &(attachment->sequencenr), FALSE);
        attachment->xnodebody = ato_xml_findnode(xml, xnodebody, xAttachmentInstance_seqnr, attachment->sequencenr);
        assert(attachment->xnodebody != NULL);
        ato_xml_nodevalue(xml, xnode, xFileName, &(attachment->filename), FALSE);
        ato_xml_nodevalue(xml, xnode, xDescription, &(attachment->description), FALSE);
        _process_doca_body(ctx, xml, attachment->xnodebody, &(attachment->content), &(attachment->contenttype));
        attachment->contentfree = TRUE;
    } else { // creating a new node
        char buf[50];
        attachment->sequencenr = ato_strdup(ato_itoa(buf, (int)*attachmentcount), 0);
    }
}

/*********************************************************************************/
int sbr__doca_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void sbr__doca_deinit(void)
{
}

void sbr__doca_free(sbr_DocAs *attachments)
{
    if (attachments == NULL) return;

    {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(attachments->itemlist, node)) != NULL)
            _freeitem(ato_lst_value(node));
    }
    ato_lst_free(attachments->itemlist);
    ato_iter_free(attachments->iter);
    free(attachments);
}

void sbr__doca_create(ato_Ctx *ctx, sbr_DocAs **obj, sbr_eMsgSourceType msgsrctype,
                                ato_Xml *xml, void *pxnode, void *pxnodebody, size_t *attachmentcount)
{
    const char *function = "sbr__doca_create";
    int errcode = ATO_ERR_OK;
    sbr_DocAs *attachments = NULL;
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);

    if (!xml || !pxnode) {
        *obj = attachments = calloc(1, sizeof(sbr_DocAs));
        assert(attachments != NULL);
        attachments->msgsrctype = msgsrctype;
        ato_lst_create(&(attachments->itemlist));
        attachments->iter = ato_iter_create_lst(attachments->itemlist);
        attachments->attachmentcount = attachmentcount;
    } else {
        void *xnode = NULL, *xnodebody = NULL;
        xnode = ato_xml_findnode(xml, pxnode, xAttachments, NULL);
        if (xnode)
            xnodebody = ato_xml_findnode(xml, pxnodebody, xStandardBusinessDocumentBody, NULL);
        if (xnodebody)
            xnodebody = ato_xml_findnode(xml, xnodebody, xAttachmentInstances, NULL);
        if (xnode && xnodebody) {
            void *xlist = NULL;

            *obj = attachments = calloc(1, sizeof(sbr_DocAs));
            assert(attachments != NULL);
            attachments->msgsrctype = msgsrctype;
            ato_lst_create(&(attachments->itemlist));
            attachments->iter = ato_iter_create_lst(attachments->itemlist);
            attachments->attachmentcount = attachmentcount;

            xlist = ato_xml_listload(xml, xnode, xAttachment, NULL);
            if (xlist != NULL) {
                size_t i = 0, count = 0;

                count = ato_xml_listcount(xml, xlist);
                for (i = 0; i < count; i++) {
                    void *psubnode = NULL;
                    sbr_DocA *item = NULL;
                    psubnode = ato_xml_listitem(xml, xlist, i);
                    assert(psubnode != NULL);
                    _createitem(ctx, &item, msgsrctype, xml, psubnode, xnodebody, attachments->attachmentcount);
                    ato_lst_add(attachments->itemlist, item->guid, item);
                }
                ato_xml_listfree(xml, xlist);
            }
        }
    }

    ATO_CTX_FN_END(ctx, errcode);
}


size_t sbr__doca_count(sbr_DocAs *attachments)
{
    if (attachments == NULL) return 0;
    return ato_lst_count(attachments->itemlist);
}
void sbr__doca_iterator(sbr_DocAs *attachments, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (attachments == NULL) return;
    *iter = ato_iter_create_lst(attachments->itemlist);
}
sbr_DocA *sbr__doca_firstv(sbr_DocAs *attachments)
{
    if (attachments == NULL) return NULL;
    return ato_iter_firstv(attachments->iter);
}
sbr_DocA *sbr__doca_nextv(sbr_DocAs *attachments)
{
    if (attachments == NULL) return NULL;
    return ato_iter_nextv(attachments->iter);
}

sbr_DocA *sbr__doca_add(ato_Ctx *ctx, sbr_DocAs *attachments)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "sbr__doca_add";
    ato_ListNode *node = NULL;
    sbr_DocA *attachment = NULL;

    assert(attachments != NULL);
    _createitem(ctx, &attachment, attachments->msgsrctype, NULL, NULL, NULL, attachments->attachmentcount);
    node = ato_lst_add(attachments->itemlist, attachment->guid, attachment);
    if (node == NULL)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Error in adding Attachment into Document.");
    return attachment;
}

void sbr__doca_remove(ato_Ctx *ctx, sbr_DocAs *attachments, sbr_DocA *attachment)
{
    ato_ListNode *node = ato_lst_find(attachments->itemlist, sbr_doca_iid(attachment));
    if (node != NULL) {
        _freeitem(ato_lst_value(node));
        ato_lst_delete(node);
    }
    ATO_IGNORE(ctx);
}

static void _render_new(ato_Ctx *ctx, sbr_DocA *attachment, ato_Xml *xml, void *pxnode, void *pxnodebody)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_render_new";
    const char *emsg = "Failed to render new attachment - can't find %s.";
    void *sbdmbodynode = NULL;
    void *attachmentsnode = NULL;
    assert(xml != NULL); assert(pxnode != NULL); assert(pxnodebody != NULL);
    attachmentsnode = ato_xml_findnode(xml, pxnode, xAttachments, NULL);
    if (!attachmentsnode)
        attachmentsnode = ato_xml_addelt(xml, pxnode, xAttachments);
    if (!attachmentsnode)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render new attachment - can't add attachment container.");
    attachment->xnode = ato_xml_addelt(xml, attachmentsnode, xAttachment);
    if (!attachment->xnode)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render new attachment - can't add attachment.");

    sbdmbodynode = ato_xml_findnode(xml, pxnodebody, xStandardBusinessDocumentBody, NULL);
    if (!sbdmbodynode)
        Throw ATO_CTX_VNEWERR(ctx, SBR_CSR_ERR_GENERAL, strlen(emsg), emsg, xStandardBusinessDocumentBody);
    attachmentsnode = ato_xml_findnode(xml, sbdmbodynode, xAttachmentInstances, NULL);
    if (!attachmentsnode)
        attachmentsnode = ato_xml_addelt(xml, sbdmbodynode, xAttachmentInstances);
    if (!attachmentsnode)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render new attachment - can't add attachment body container.");
    attachment->xnodebody = ato_xml_addelt(xml, attachmentsnode, xAttachmentInstance);
    if (!attachment->xnodebody)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render new attachment - can't add attachment body.");
}

static void _render_contenttype(sbr_DocA *attachment, const char *value)
{
    void *pxnode = ato_xml_findnode(attachment->xml, attachment->xnodebody, xText, NULL);
    void *xnode = NULL;

    xnode = ato_xml_createattr(attachment->xml, pxnode, xContentType, "nsmime", "http://www.w3.org/2005/05/xmlmime");
    ato_xml_setvalue(attachment->xml, xnode, value);
}
void sbr__doca_render(ato_Ctx *ctx, sbr_DocAs *attachments, ato_Xml *xml, void *pxnode, void *pxnodebody)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "sbr__doca_render";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);

    assert(ctx != NULL);
    if (attachments) {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(attachments->itemlist, node)) != NULL) {
            sbr_DocA *attachment = ato_lst_value(node);
            if (!attachment->xnode) {
                _render_new(ctx, attachment, xml, pxnode, pxnodebody);
            }
            if (!attachment->xnode)
                Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render attachment.");
            if (!attachment->xnodebody)
                Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Failed to render attachment body.");
            attachment->xml = xml;
            ato_xml_createnodevalue(attachment->xml, attachment->xnode, NULL, xSequenceNumber, attachment->sequencenr);
            ato_xml_createnodevalue(attachment->xml, attachment->xnode, NULL, xFileName, attachment->filename);
            ato_xml_createnodevalue(attachment->xml, attachment->xnode, NULL, xDescription, attachment->description);
            ato_xml_createnodevalue(attachment->xml, attachment->xnodebody, NULL, xSequenceNumber, attachment->sequencenr);
            ato_xml_createnodevaluelen(attachment->xml, attachment->xnodebody, NULL, xText, ato_str_value(attachment->content), ato_str_len(attachment->content));
            _render_contenttype(attachment, attachment->contenttype);
        }
    }

    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
const char *sbr_doca_iid(sbr_DocA *attachment)
{
    assert(attachment != NULL);
    return attachment->guid;
}

ato_String *sbr_doca_content(sbr_DocA *attachment)
{
    assert(attachment != NULL);
    return attachment->content;
}

const char *sbr_doca_contenttype(sbr_DocA *attachment)
{
    assert(attachment != NULL);
    return attachment->contenttype;
}

const char *sbr_doca_description(sbr_DocA *attachment)
{
    assert(attachment != NULL);
    return attachment->description;
}

const char *sbr_doca_filename(sbr_DocA *attachment)
{
    assert(attachment != NULL);
    return attachment->filename;
}

size_t sbr_doca_sequencenr(sbr_DocA *attachment)
{
    assert(attachment != NULL);
    return (size_t)strtol(attachment->sequencenr, NULL, 10);
}

sbr_eMsgSourceType sbr_doca_msgsourcetype(sbr_DocA *attachment)
{
    assert(attachment != NULL);
    return attachment->msgsrctype;
}

int sbr_doca_setcontent(ato_Ctx *ctx, sbr_DocA *attachment, ato_String *content, ato_eParamAction contentaction)
{
    assert(ctx != NULL); assert(attachment != NULL);

    _freecontent(attachment);
    if (content) {
        switch (contentaction) {
            case ATO_PARAMACTION_CONST:
                attachment->contentfree = FALSE;
                attachment->content = content;
                break;
            case ATO_PARAMACTION_FREE:
                attachment->contentfree = TRUE;
                attachment->content = content;
                break;
            default:
            //case ATO_PARAMACTION_DEFAULT:
            //case ATO_PARAMACTION_COPY:
                attachment->contentfree = TRUE;
                ato_str_dup(&(attachment->content), content);
                break;
        }
    }
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doca_setdescription(ato_Ctx *ctx, sbr_DocA *attachment, const char *value)
{
    assert(ctx != NULL); assert(attachment != NULL);
    ato_free(attachment->description);
    attachment->description = ato_strdup(value, 0);
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doca_setfilename(ato_Ctx *ctx, sbr_DocA *attachment, const char *value)
{
    assert(ctx != NULL); assert(attachment != NULL);
    ato_free(attachment->filename);
    attachment->filename = ato_strdup(value, 0);
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

int sbr_doca_setcontenttype(ato_Ctx *ctx, sbr_DocA *attachment, const char *value)
{
    assert(ctx != NULL); assert(attachment != NULL);
    ato_free(attachment->contenttype);
    attachment->contenttype = ato_strdup(value, 0);
    ATO_IGNORE(ctx);
    return ATO_ERR_OK;
}

