#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "libxml/tree.h"
#include "libxml/parser.h"
#include "libxml/xpath.h"
#include "libxml/xpathInternals.h"
#include "libxml/xmlmemory.h"

#include "xmlsec/xmlsec.h"
#include "xmlsec/xmltree.h"
#include "xmlsec/xmldsig.h"
#include "xmlsec/crypto.h"
#include "xmlsec/parser.h"

#include "akversion.h"

#include "atobase/private/pall.h"
#include "atobnet/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intcsr.h"
#include "intmsgevent.h"
#include "intmsgts.h"
#include "intmsgparty.h"
#include "intmsgrec.h"
#include "intdoca.h"
#include "intdoc.h"
#include "intsbdm.h"
#include "intresponse.h"
#include "intrequest.h"
#include "interrfn.h"
#include "intlib.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_LIB;
static unsigned long _moduleid = SBR_CSR_MODULEID_LIB;

static int _loglevel = ATO_LOG_WARN;

static const char _defcsrtemplatefile[] = "sbrcsr_template.xml";

/*********************************************************************************/
static void _load_errdefs(void);

/*********************************************************************************/
const char *sbr_csr_version(void) { return SBRCSR_VERSION; }

static void _setloglevel(ato_eLoglevel level)
{
        _loglevel = level;
}

int sbr__csrlib_init(void)
{
        static bool invoked = FALSE;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        return ATO_ERR_OK;
}
void sbr__csrlib_deinit(void)
{
        static bool invoked = FALSE;
        if (invoked) return;
        invoked = TRUE;
}

int sbr__csrlib_process_errcode(ato_Ctx *ctx, int errcode, const char *module, const char *function, int line)
{
        int liberrcode = SBR_CSR_ERR_GENERAL;
        if (errcode == ATO_ERR_OK)
                return errcode;
        if (errcode >= SBR_CSR_ERR_MIN && errcode <= SBR_CSR_ERR_MAX)
                return errcode;

        if (errcode == ATO_ERR_NET_TIMEOUT)
                liberrcode = SBR_CSR_ERR_NETTIMEOUT;
        else if (errcode == ATO_ERR_NET_SOAP)
                liberrcode = SBR_CSR_ERR_NETRECEIVER;
        else if (errcode == ATO_ERR_NET)
                liberrcode = SBR_CSR_ERR_NETCOMMS;
        return ato_ctx_err_new(ctx, _library, module, function, line, liberrcode, ATO_ESEVERITY_WARN, NULL, "");
}

/*********************************************************************************/
#define LIBXMLINITED "94C96C36-70EC-48e1-A652-475067FFEB77"
static int _ato_tmpl_init(void)
{
        static bool invoked = FALSE;
        int errcode = ATO_ERR_OK;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        if (!ato_base_var(LIBXMLINITED, NULL)) {
                //printf("CSR initialising xmlsec\n");
                ato_base_var_set(LIBXMLINITED, "TRUE");

                xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
                xmlSubstituteEntitiesDefault(1);

                if(xmlSecInit() < 0)
                        errcode = 100;
                if(xmlSecCheckVersion() != 1)
                        errcode = ATO_ERR_VERSION_DEP;
                if(xmlSecCryptoAppInit(NULL) < 0)
                        errcode = 101;
                if(xmlSecCryptoInit() < 0)
                        errcode = ATO_ERR_INIT;
        }

        return errcode;
}

static void _ato_tmpl_deinit(void)
{
        const char *value = NULL;
        static bool invoked = FALSE;
        if (invoked) return;
        invoked = TRUE;

        if (ato_base_var(LIBXMLINITED, &value)) {
                if (value) {
                        //printf("CSR deinitialising xmlsec\n");
                        ato_base_var_set(LIBXMLINITED, NULL);
                        xmlSecCryptoShutdown();
                        xmlSecCryptoAppShutdown();
                        xmlSecShutdown();
                }
        }
}
/*********************************************************************************/
int sbr_csr_init(ato_Ctx *ctx, unsigned short flag)
{
        const char *function = "sbr_csr_init";
        static bool invoked = FALSE;
        int errcode = ATO_ERR_OK;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        assert(flag != 0); ATO_IGNORE(flag);
        assert(ctx != NULL);

    assert(ato_base_isinited() == TRUE);
    assert(ato_bnet_isinited() == TRUE);

        _load_errdefs();

        if (errcode == ATO_ERR_OK)
                errcode = sbr__csrlib_init();

        if (errcode == ATO_ERR_OK)
                errcode = _ato_tmpl_init();

        if (errcode == ATO_ERR_OK)
                errcode = sbr__msgp_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__msgts_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__msgrec_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__msgevent_init();

        if (errcode == ATO_ERR_OK)
                errcode = sbr__doca_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__doc_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__sbdm_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__res_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__req_init();
        if (errcode == ATO_ERR_OK)
                errcode = sbr__csr_init();

        //errcode = ato_errh_sethandler(sbr__csr_errfn, SBR_CSR_LIBRARY);

        if (errcode != ATO_ERR_OK)
                ATO_CTX_NEWERR(ctx, errcode, "Failed to initialise CSR library");
        return errcode;
}
void sbr_csr_deinit(void)
{
        static bool invoked = FALSE;
        if (invoked) return;
        invoked = TRUE;

        sbr__csr_deinit();
        sbr__res_deinit();
        sbr__req_deinit();
        sbr__sbdm_deinit();
        sbr__doc_deinit();
        sbr__doca_deinit();

        sbr__msgp_deinit();
        sbr__msgts_deinit();
        sbr__msgrec_deinit();
        sbr__msgevent_deinit();

        _ato_tmpl_deinit();
        sbr__csrlib_deinit();
}

/*********************************************************************************/
#define _ATO_ADD_ERRDEF(code, name, msg) ato_errdef_add(code, _library, name, msg)

static void _load_errdefs(void)
{
        _ATO_ADD_ERRDEF(SBR_CSR_ERR_GENERAL, "SBR_CSR_ERR_GENERAL", "");
        _ATO_ADD_ERRDEF(SBR_CSR_ERR_NETSENDER, "SBR_CSR_ERR_NETSENDER", "");
        _ATO_ADD_ERRDEF(SBR_CSR_ERR_NETRECEIVER, "SBR_CSR_ERR_NETRECEIVER", "");
        _ATO_ADD_ERRDEF(SBR_CSR_ERR_NETUNAVAILABLE, "SBR_CSR_ERR_NETUNAVAILABLE", "");
        _ATO_ADD_ERRDEF(SBR_CSR_ERR_NETCOMMS, "SBR_CSR_ERR_NETCOMMS", "");
        _ATO_ADD_ERRDEF(SBR_CSR_ERR_NETTIMEOUT, "SBR_CSR_ERR_NETTIMEOUT", "");
}
/*********************************************************************************/

const char *sbr_csr_default_templatefile(void)
{
        return _defcsrtemplatefile;
}

const char *sbr_msgsourcetypestr(sbr_eMsgSourceType msgsrctype)
{
        switch (msgsrctype) {
                case SBR_REQUEST: return "SBR_REQUEST";
                case SBR_RESPONSE: return "SBR_RESPONSE";
                default: return "";
        }
}

int sbr_csr_loadtemplate(ato_Ctx *ctx, char **buffer, const char *dirname, const char *filename) {
    const char *function = "sbr_csr_loadtemplate";
    int errcode = ATO_ERR_OK;
    char *buf = NULL;
    ato_String *str = NULL;
    char *path = NULL;
    ATO_ASSERT_ISNOTALLOCATED(buffer);

    assert(dirname);
    if (filename == NULL) { filename = ""; }
    path = ato_pathcat(dirname, filename);

    errcode = ato_fileload(&str, path);
    if (errcode != ATO_ERR_OK) {
        ATO_CTX_VNEWERR(ctx, errcode, strlen(path), "Failed to load csr template file '%s'", path);
    } else {
        buf = ato_str_valueasowner(str);
    }

    if (path) free(path);
    ato_str_free(str);
    *buffer = buf;
    return sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
}
