#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intmsgevent.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_MSGEVENT;
static unsigned long _moduleid = SBR_CSR_MODULEID_MSGEVENT;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;

static const char *xMessageEventItems = ".//data:MessageEventItems";
static const char *xMessageEventItem = "./data:MessageEventItem";
static const char *xMessageEventItem_Code = "data:Message.Event.Item.Error.Code";
static const char *xMessageEventItem_Severity = "data:Message.Event.Item.Severity.Code";
static const char *xMessageEventItem_ShortD = "data:Message.Event.Item.Short.Description";
static const char *xMessageEventItem_LongD = "data:Message.Event.Item.Detailed.Description";

/*********************************************************************************/
struct _sbr_MsgEventItemLocation {
    size_t sequencenr;
    char *path;
};

struct _sbr_MsgEventItemParam {
    char *name;
    char *value;
};

struct _sbr_MsgEventItem {
    char *code;
    char *severity;
    char *shortdesc;
    char *longdesc;
    ato_List *locations;
    ato_List *params;
    ato_Iterator *iter_L;
    ato_Iterator *iter_P;
};

struct _sbr_MsgEventItems {
    ato_List *itemlist;
    ato_Iterator *iter;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

static void _freeitem(sbr_MsgEventItem *eventitem)
{
    ato_ListNode *node = NULL;
    if (eventitem == NULL) return;

    if (eventitem->code != NULL) free(eventitem->code);
    if (eventitem->severity != NULL) free(eventitem->severity);
    if (eventitem->shortdesc != NULL) free(eventitem->shortdesc);
    if (eventitem->longdesc != NULL) free(eventitem->longdesc);

    if (eventitem->locations != NULL) {
        while ((node = ato_lst_next(eventitem->locations, node)) != NULL) {
            sbr_MsgEventItemLocation *location = ato_lst_value(node);
            location->path = ato_free(location->path);
            free(location);
        }
        ato_lst_free(eventitem->locations);
    }
    if (eventitem->params != NULL) {
        while ((node = ato_lst_next(eventitem->params, node)) != NULL)  {
            sbr_MsgEventItemParam *param = ato_lst_value(node);
            param->name = ato_free(param->name);
            param->value = ato_free(param->value);
            free(param);
        }
        ato_lst_free(eventitem->params);
    }
    ato_iter_free(eventitem->iter_L);
    ato_iter_free(eventitem->iter_P);
    free(eventitem);
}

static void _location_add(ato_Ctx *ctx, sbr_MsgEventItem *eventitem, size_t sequencenr, const char *path)
{
    sbr_MsgEventItemLocation *location = NULL;
    location = calloc(1, sizeof(sbr_MsgEventItemLocation));
    assert(location != NULL);
    location->sequencenr = sequencenr;
    location->path = ato_strdup(path, 0);
    ato_lst_add(eventitem->locations, NULL, location);
    ATO_IGNORE(ctx);
}

static void _param_add(ato_Ctx *ctx, sbr_MsgEventItem *eventitem, const char *name, const char *value)
{
    sbr_MsgEventItemParam *param = NULL;
    param = calloc(1, sizeof(sbr_MsgEventItemParam));
    assert(param != NULL);
    param->name = ato_strdup(name, 0);
    param->value = ato_strdup(value, 0);
    ato_lst_add(eventitem->params, NULL, param);
    ATO_IGNORE(ctx);
}

static void _process_msgevent_param(ato_Ctx *ctx, sbr_MsgEventItem *eventitem, ato_Xml *xml, void *pnode)
{
    static const char *xLocation = ".//data:Parameter";
    static const char *xId = "data:Message.Event.Item.Parameter.Identifier";
    static const char *xText = "data:Message.Event.Item.Parameter.Text";
    size_t i = 0, count = 0;
    void *xlist = NULL;
    xlist = ato_xml_listload(xml, pnode, xLocation, NULL);

    if (xlist == NULL)
        return;

    count = ato_xml_listcount(xml, xlist);
    for (i = 0; i < count; i++) {
        void *psubnode = NULL;
        char *id = NULL, *text = NULL;
        psubnode = ato_xml_listitem(xml, xlist, i);
        assert(psubnode != NULL);
        ato_xml_nodevalue(xml, psubnode, xId, &id, FALSE);
        ato_xml_nodevalue(xml, psubnode, xText, &text, FALSE);
        _param_add(ctx, eventitem, id, text);
    }
    ato_xml_listfree(xml, xlist);
}

static void _process_msgevent_location(ato_Ctx *ctx, sbr_MsgEventItem *eventitem, ato_Xml *xml, void *pnode)
{
    static const char *xLocation = ".//data:Location";
    static const char *xSequenceNumber = "data:BusinessDocument.Sequence.Number";
    static const char *xLocationPath = "data:Message.Event.Item.Location.Path.Text";
    size_t i = 0, count = 0;
    void *xlist = NULL;
    xlist = ato_xml_listload(xml, pnode, xLocation, NULL);

    if (xlist == NULL)
        return;

    count = ato_xml_listcount(xml, xlist);
    for (i = 0; i < count; i++) {
        void *psubnode = NULL;
        char *nr = NULL, *path = NULL;
        psubnode = ato_xml_listitem(xml, xlist, i);
        assert(psubnode != NULL);
        ato_xml_nodevalue(xml, psubnode, xSequenceNumber, &nr, FALSE);
        ato_xml_nodevalue(xml, psubnode, xLocationPath, &path, FALSE);
        _location_add(ctx, eventitem, (size_t)strtol(nr, NULL, 10), path);
    }
    ato_xml_listfree(xml, xlist);
}

static void _createitem(ato_Ctx *ctx, sbr_MsgEventItem **obj, ato_Xml *xml, void *xnode)
{
    sbr_MsgEventItem *eventitem = NULL;

    assert(ctx != NULL);
    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = eventitem = calloc(1, sizeof(sbr_MsgEventItem));
    assert(eventitem != NULL);

    if (xnode) {
        ato_xml_nodevalue(xml, xnode, xMessageEventItem_Code, &(eventitem->code), FALSE);
        ato_xml_nodevalue(xml, xnode, xMessageEventItem_Severity, &(eventitem->severity), FALSE);
        ato_xml_nodevalue(xml, xnode, xMessageEventItem_ShortD, &(eventitem->shortdesc), FALSE);
        ato_xml_nodevalue(xml, xnode, xMessageEventItem_LongD, &(eventitem->longdesc), FALSE);
        ato_lst_create(&(eventitem->locations));
        ato_lst_create(&(eventitem->params));
        eventitem->iter_L = ato_iter_create_lst(eventitem->locations);
        eventitem->iter_P = ato_iter_create_lst(eventitem->params);
        _process_msgevent_location(ctx, eventitem, xml, xnode);
        _process_msgevent_param(ctx, eventitem, xml, xnode);
    }
}
/*********************************************************************************/
int sbr__msgevent_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void sbr__msgevent_deinit(void)
{
}

void sbr__msgevent_free(sbr_MsgEventItems *eventitems)
{
    if (eventitems == NULL) return;

    {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(eventitems->itemlist, node)) != NULL)
            _freeitem(ato_lst_value(node));
    }
    ato_lst_free(eventitems->itemlist);
    ato_iter_free(eventitems->iter);
    free(eventitems);
}

void sbr__msgevent_create(ato_Ctx *ctx, sbr_MsgEventItems **obj,
    ato_Xml *xml, void *pxnode)
{
    const char *function = "sbr__msgevent_create";
    int errcode = ATO_ERR_OK;
    sbr_MsgEventItems *eventitems = NULL;
    void *xnode = NULL;
    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);

    xnode = ato_xml_findnode(xml, pxnode, xMessageEventItems, NULL);
    if (xnode) {
        void *xlist = NULL;

        *obj = eventitems = calloc(1, sizeof(sbr_MsgEventItems));
        assert(eventitems != NULL);

        xlist = ato_xml_listload(xml, xnode, xMessageEventItem, NULL);
        if (xlist != NULL) {
            size_t i = 0, count = 0;
            ato_lst_create(&(eventitems->itemlist));

            count = ato_xml_listcount(xml, xlist);
            for (i = 0; i < count; i++) {
                void *psubnode = NULL;
                sbr_MsgEventItem *item = NULL;
                psubnode = ato_xml_listitem(xml, xlist, i);
                assert(psubnode != NULL);
                _createitem(ctx, &item, xml, psubnode);
                ato_lst_add(eventitems->itemlist, NULL, item);
            }
            ato_xml_listfree(xml, xlist);
        }
        eventitems->iter = ato_iter_create_lst(eventitems->itemlist);
    }

    ATO_CTX_FN_END(ctx, errcode);
}

size_t sbr__msgevent_count(sbr_MsgEventItems *eventitems)
{
    if (eventitems == NULL) return 0;
    return ato_lst_count(eventitems->itemlist);
}

void sbr__msgevent_iterator(sbr_MsgEventItems *eventitems, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (eventitems == NULL) return;
    *iter = ato_iter_create_lst(eventitems->itemlist);
}

sbr_MsgEventItem *sbr__msgevent_firstv(sbr_MsgEventItems *eventitems)
{
    if (eventitems == NULL) return NULL;
    return ato_iter_firstv(eventitems->iter);
}
sbr_MsgEventItem *sbr__msgevent_nextv(sbr_MsgEventItems *eventitems)
{
    if (eventitems == NULL) return NULL;
    return ato_iter_nextv(eventitems->iter);
}

/*********************************************************************************/
const char *sbr_msgevent_code(sbr_MsgEventItem *eventitem) { assert(eventitem != NULL); return eventitem->code; }
const char *sbr_msgevent_severity(sbr_MsgEventItem *eventitem) { assert(eventitem != NULL); return eventitem->severity; }
const char *sbr_msgevent_shortdesc(sbr_MsgEventItem *eventitem) { assert(eventitem != NULL); return eventitem->shortdesc; }
const char *sbr_msgevent_longdesc(sbr_MsgEventItem *eventitem) { assert(eventitem != NULL); return eventitem->longdesc; }
size_t sbr_msgevent_location_sequencenr(sbr_MsgEventItemLocation *location) { assert(location != NULL); return location->sequencenr; }
const char *sbr_msgevent_location_path(sbr_MsgEventItemLocation *location) { assert(location != NULL); return location->path; }
const char *sbr_msgevent_param_name(sbr_MsgEventItemParam *param) { assert(param != NULL); return param->name; }
const char *sbr_msgevent_param_value(sbr_MsgEventItemParam *param) { assert(param != NULL); return param->value; }

size_t sbr_msgevent_location_count(sbr_MsgEventItem *eventitem)
{
    assert(eventitem != NULL);
    return ato_lst_count(eventitem->locations);
}
void sbr_msgevent_location_iterator(sbr_MsgEventItem *eventitem, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (eventitem == NULL) return;
    *iter = ato_iter_create_lst(eventitem->locations);
}
sbr_MsgEventItemLocation *sbr_msgevent_location_firstv(sbr_MsgEventItem *eventitem)
{
    if (eventitem == NULL) return NULL;
    return ato_iter_firstv(eventitem->iter_L);
}
sbr_MsgEventItemLocation *sbr_msgevent_location_nextv(sbr_MsgEventItem *eventitem)
{
    if (eventitem == NULL) return NULL;
    return ato_iter_nextv(eventitem->iter_L);
}

size_t sbr_msgevent_param_count(sbr_MsgEventItem *eventitem)
{
    assert(eventitem != NULL);
    return ato_lst_count(eventitem->params);
}
void sbr_msgevent_param_iterator(sbr_MsgEventItem *eventitem, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (eventitem == NULL) return;
    *iter = ato_iter_create_lst(eventitem->params);
}
sbr_MsgEventItemParam *sbr_msgevent_param_firstv(sbr_MsgEventItem *eventitem)
{
    if (eventitem == NULL) return NULL;
    return ato_iter_firstv(eventitem->iter_P);
}
sbr_MsgEventItemParam *sbr_msgevent_param_nextv(sbr_MsgEventItem *eventitem)
{
    if (eventitem == NULL) return NULL;
    return ato_iter_nextv(eventitem->iter_P);
}
