#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>

#include "atobase/all.h"
#include "atoakm/all.h"
#include "atostm/all.h"
#include "sbrcsr/all.h"

#include "dump.h"

static bool toconsole = FALSE;

//static const char *_blankifnull(const char *value) { return value ? value : ""; }

static void _echofp(FILE *fp, size_t *indent, const char *format, va_list args)
{
    size_t i = 0;
    size_t count = (*indent) * 2;
    if (fp == NULL) return;
    for (; i < count; i++) {
        fprintf(fp, " ");
    }
    vfprintf(fp, format, args);
    fprintf(fp, "\n");
}

static void _echo(bool console, FILE *fp, size_t *indent, const char *format, ...)
{
    va_list args;
    if (console && fp != stdout) {
        va_start(args, format);
        _echofp(stdout, indent, format, args);
        va_end(args);
    }
    if (fp) {
        va_start(args, format);
        _echofp(fp, indent, format, args);
        va_end(args);
    }
}

static void _dump_party(FILE *fp, size_t *indent, sbr_Sbdm *sbdm, sbr_eMsgPartyType msgptype)
{
    sbr_MsgParty *party = sbr_sbdm_party(sbdm, msgptype);
    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    if (party) {
        _echo(TRUE, fp, indent, "PARTY type=%s", sbr_msgp_typestr(sbr_msgp_type(party)));
        _echo(toconsole, fp, indent, "designation=%s", sbr_msgp_iddesignation(party));
        _echo(toconsole, fp, indent, "scheme=%s", sbr_msgp_idname(party));
    } else {
        _echo(TRUE, fp, indent, "NO PARTY of type=%s", sbr_msgp_typestr(msgptype));
    }

    (*indent)--;
}

static void _dump_receipt(FILE *fp, size_t *indent, sbr_Sbdm *sbdm)
{
    sbr_MsgReceipt *rec = sbr_sbdm_receipt(sbdm);
    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    if (rec) {
        _echo(TRUE, fp, indent, "RECEIPT id=%s", sbr_msgrec_id(rec));
        _echo(toconsole, fp, indent, "datetime=%s", sbr_msgrec_datetime(rec));
    } else {
        _echo(TRUE, fp, indent, "NO RECEIPT");
    }

    (*indent)--;
}

static void _dump_timestamp(FILE *fp, size_t *indent, sbr_MsgTimestamp *ts, size_t index, size_t count)
{
    if (!ts) return;

    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    _echo(TRUE, fp, indent, "TIMESTAMP index=%d of %d", index, count);
    _echo(toconsole, fp, indent, "datetime=%s", sbr_msgts_datetime(ts));
    _echo(toconsole, fp, indent, "code=%s", sbr_msgts_source(ts));

    (*indent)--;
}

static void _dump_timestamps(FILE *fp, size_t *indent, sbr_Sbdm *sbdm)
{
    size_t index = 1, count = sbr_sbdm_timestamp_count(sbdm);
    sbr_MsgTimestamp *ts = NULL;
    if (count) _echo(TRUE, fp, indent, "Timestamps (start) ===============================");
    for (ts = sbr_sbdm_timestamp_firstv(sbdm); ts; ts = sbr_sbdm_timestamp_nextv(sbdm)) {
        _dump_timestamp(fp, indent, ts, index++, count);
    }
    if (count) _echo(TRUE, fp, indent, "Timestamps (end) =================================");
}

static void _dump_ei_location(FILE *fp, size_t *indent, sbr_MsgEventItemLocation *location, size_t index, size_t count)
{
    if (!location) return;

    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    _echo(TRUE, fp, indent, "LOCATION index=%d of %d", index, count);
    _echo(toconsole, fp, indent, "sequencenr=%d", sbr_msgevent_location_sequencenr(location));
    _echo(toconsole, fp, indent, "path=%s", sbr_msgevent_location_path(location));

    (*indent)--;
}

static void _dump_ei_param(FILE *fp, size_t *indent, sbr_MsgEventItemParam *param, size_t index, size_t count)
{
    if (!param) return;

    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    _echo(TRUE, fp, indent, "PARAM index=%d of %d", index, count);
    _echo(toconsole, fp, indent, "name=%s", sbr_msgevent_param_name(param));
    _echo(toconsole, fp, indent, "value=%s", sbr_msgevent_param_value(param));

    (*indent)--;
}

static void _dump_ei_locations(FILE *fp, size_t *indent, sbr_MsgEventItem *eventi)
{
    size_t index = 1, count = sbr_msgevent_location_count(eventi);
    sbr_MsgEventItemLocation *location = NULL;
    if (count) _echo(TRUE, fp, indent, "Locations (start) ================================");
    for (location = sbr_msgevent_location_firstv(eventi); location; location = sbr_msgevent_location_nextv(eventi)) {
        _dump_ei_location(fp, indent, location, index++, count);
    }
    if (count) _echo(TRUE, fp, indent, "Locations (end) ==================================");
}

static void _dump_ei_params(FILE *fp, size_t *indent, sbr_MsgEventItem *eventi)
{
    size_t index = 1, count = sbr_msgevent_param_count(eventi);
    sbr_MsgEventItemParam *param = NULL;
    if (count) _echo(TRUE, fp, indent, "Params (start) ===================================");
    for (param = sbr_msgevent_param_firstv(eventi); param; param = sbr_msgevent_param_nextv(eventi)) {
        _dump_ei_param(fp, indent, param, index++, count);
    }
    if (count) _echo(TRUE, fp, indent, "Params (end) =====================================");
}

void dump_eventitem(FILE *fp, size_t *indent, sbr_MsgEventItem *eventi, size_t index, size_t count, bool dumpinner)
{
    if (!eventi) return;

    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    _echo(TRUE, fp, indent, "EVENTITEM index=%d of %d", index, count);
    _echo(toconsole, fp, indent, "code=%s", sbr_msgevent_code(eventi));
    _echo(toconsole, fp, indent, "severity=%s", sbr_msgevent_severity(eventi));
    _echo(toconsole, fp, indent, "shortdesc=%s", sbr_msgevent_shortdesc(eventi));
    _echo(toconsole, fp, indent, "longdesc=%s", sbr_msgevent_longdesc(eventi));

    if (dumpinner) {
        _dump_ei_locations(fp, indent, eventi);
        _dump_ei_params(fp, indent, eventi);
    }

    (*indent)--;
}

static void _dump_eventitems(FILE *fp, size_t *indent, sbr_Sbdm *sbdm)
{
    size_t index = 1, count = sbr_sbdm_eventitem_count(sbdm);
    sbr_MsgEventItem *ei = NULL;
    if (count) _echo(TRUE, fp, indent, "Eventitems (start) ===============================");
    for (ei = sbr_sbdm_eventitem_firstv(sbdm); ei; ei = sbr_sbdm_eventitem_nextv(sbdm)) {
        dump_eventitem(fp, indent, ei, index++, count, TRUE);
    }
    if (count) _echo(TRUE, fp, indent, "Eventitems (end) =================================");
}


static void _dump_attachment(ato_Ctx *ctx, FILE *fp, size_t *indent, sbr_DocA *attachment, size_t index, size_t tcount, size_t docindex)
{
    ato_String *content = NULL;
    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    _echo(TRUE, fp, indent, "ATTACHMENT index=%d of %d (iid=%s)", index, tcount, sbr_doca_iid(attachment));
    _echo(toconsole, fp, indent, "contenttype=%s", sbr_doca_contenttype(attachment));
    _echo(toconsole, fp, indent, "description=%s", sbr_doca_description(attachment));
    _echo(toconsole, fp, indent, "filename=%s", sbr_doca_filename(attachment));
    _echo(toconsole, fp, indent, "sequencenr=%d", sbr_doca_sequencenr(attachment));
    _echo(toconsole, fp, indent, "msgsourcetypestr=%s", sbr_msgsourcetypestr(sbr_doca_msgsourcetype(attachment)));
    content = sbr_doca_content(attachment);
    if (content) {
        char fname[50];
        FILE *fout = NULL;
        ato_String *rawcontent = NULL;
        sprintf(fname, "doc-%" PRINTF_SIZET "u-attachment-%" PRINTF_SIZET "u-out.pdf", docindex, index);
        fout = fopen(fname, "wb");
        assert(fout != NULL);
        _echo(toconsole, fp, indent, "b64 content=%.50s, binary dumped to file %s", ato_str_value(content), fname);
        ato_base64decode_array(ctx, &rawcontent, ato_str_value(content), ato_str_len(content));
        fwrite(ato_str_value(rawcontent), sizeof(char), ato_str_len(rawcontent), fout);
        ato_str_free(rawcontent);
        fclose(fout);
    }

    (*indent)--;
}

static void _dump_attachments(ato_Ctx *ctx, FILE *fp, size_t *indent, sbr_Doc *doc, size_t docindex)
{
    size_t index = 1, count = sbr_doc_attachment_count(doc);
    sbr_DocA *doca = NULL;

    if (count) _echo(TRUE, fp, indent, "Attachments (start) ==============================");
    for (doca = sbr_doc_attachment_firstv(doc); doca; doca = sbr_doc_attachment_nextv(doc)) {
        _dump_attachment(ctx, fp, indent, doca, index++, count, docindex);
    }
    if (count) _echo(TRUE, fp, indent, "Attachments (end) ================================");
}

static void _dump_doc(ato_Ctx *ctx, FILE *fp, size_t *indent, sbr_Doc *doc, size_t index, size_t tcount)
{
    if (!doc) return;

    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    _echo(TRUE, fp, indent, "DOC index=%d of %d (iid=%s)", index, tcount, sbr_doc_iid(doc));
    _echo(toconsole, fp, indent, "sequencenr=%d", sbr_doc_sequencenr(doc));
    _echo(toconsole, fp, indent, "createtime=%s", sbr_doc_createtime(doc));
    _echo(toconsole, fp, indent, "validationuri=%s", sbr_doc_validationuri(doc));
    _echo(toconsole, fp, indent, "governmentid=%s", sbr_doc_governmentid(doc));
    _echo(toconsole, fp, indent, "businessid=%s", sbr_doc_businessid(doc));
    _echo(toconsole, fp, indent, "content=%.50s", sbr_doc_content(doc));
    _echo(toconsole, fp, indent, "msgsourcetype=%s", sbr_msgsourcetypestr(sbr_doc_msgsourcetype(doc)));

    _dump_attachments(ctx, fp, indent, doc, index);

    (*indent)--;
}

static void _dump_docs(ato_Ctx *ctx, FILE *fp, size_t *indent, sbr_Sbdm *sbdm)
{
    size_t index = 1, count = sbr_sbdm_doc_count(sbdm);
    sbr_Doc *doc = NULL;
    if (count) _echo(TRUE, fp, indent, "Documents (start) ================================");
    for (doc = sbr_sbdm_doc_firstv(sbdm); doc; doc = sbr_sbdm_doc_nextv(sbdm)) {
        _dump_doc(ctx, fp, indent, doc, index++, count);
    }
    if (count) _echo(TRUE, fp, indent, "Documents (end) ==================================");
}

static void _dump_sbdm(ato_Ctx *ctx, FILE *fp, size_t *indent, sbr_Sbdm *sbdm, size_t index, size_t count)
{
    if (!sbdm)
        return;

    (*indent)++;

    _echo(toconsole, fp, indent, "--------------------------------------------------");
    _echo(TRUE, fp, indent, "SBDM index=%d of %d (iid=%s)", index, count, sbr_sbdm_iid(sbdm));
    _echo(toconsole, fp, indent, "msgsourcetype=%s", sbr_msgsourcetypestr(sbr_sbdm_msgsourcetype(sbdm)));
    _echo(toconsole, fp, indent, "messagetype=%s", sbr_sbdm_messagetype(sbdm));
    _echo(toconsole, fp, indent, "maxseveritycode=%s", sbr_sbdm_maxseveritycode(sbdm));

    _dump_party(fp, indent, sbdm, SBR_MSG_SENDER);
    _dump_party(fp, indent, sbdm, SBR_MSG_RECEIVER);
    _dump_receipt(fp, indent, sbdm);

    _dump_timestamps(fp, indent, sbdm);
    _dump_eventitems(fp, indent, sbdm);

    _dump_docs(ctx, fp, indent, sbdm);

    (*indent)--;
}

static void _dump_sbdms_res(ato_Ctx *ctx, FILE *fp, size_t *indent, sbr_Response *response)
{
    size_t index = 1, count = sbr_res_sbdm_count(response);
    sbr_Sbdm *sbdm = NULL;
    if (count) _echo(TRUE, fp, indent, "SBDM (start) =====================================");
    for (sbdm = sbr_res_sbdm_firstv(response); sbdm; sbdm = sbr_res_sbdm_nextv(response)) {
        _dump_sbdm(ctx, fp, indent, sbdm, index++, count);
    }
    if (count) _echo(TRUE, fp, indent, "SBDM (end) =======================================");
}

static void _dump_sbdms_req(ato_Ctx *ctx, FILE *fp, size_t *indent, sbr_Request *request)
{
    size_t index = 1, count = sbr_req_sbdm_count(request);
    sbr_Sbdm *sbdm = NULL;
    if (count) _echo(TRUE, fp, indent, "SBDM (start) =====================================");
    for (sbdm = sbr_req_sbdm_firstv(request); sbdm; sbdm = sbr_req_sbdm_nextv(request)) {
        _dump_sbdm(ctx, fp, indent, sbdm, index++, count);
    }
    if (count) _echo(TRUE, fp, indent, "SBDM (end) =======================================");
}

void dump_response(ato_Ctx *ctx, sbr_Response *r, const char *filename)
{
    size_t indent = 0;
    FILE *fp = NULL;
    if (!r) return;

    if (filename) {
        fp = fopen(filename, "w");
        assert(fp != NULL);
    }

    _echo(TRUE, fp, &indent, "\nCSR Response");

    indent++;
    _dump_sbdms_res(ctx, fp, &indent, r);
    indent--;

    if (fp)
        _echo(TRUE, NULL, &indent, "CSR Response content dumped to %s\n", filename);

    if (fp) fclose(fp);
}

void dump_request(ato_Ctx *ctx, sbr_Request *r, const char *filename)
{
    size_t indent = 0;
    FILE *fp = NULL;
    if (!r) return;

    if (filename) {
        fp = fopen(filename, "w");
        assert(fp != NULL);
    }

    _echo(TRUE, fp, &indent, "\nCSR Request");

    indent++;
    _dump_sbdms_req(ctx, fp, &indent, r);
    indent--;

    if (fp)
        _echo(TRUE, NULL, &indent, "CSR Request content dumped to %s\n", filename);

    if (fp) fclose(fp);
}

