#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"

#include "atoksprop.h"

static const char *_library = ATO_AKM_LIBRARY;
static const char *_module = ATO_AKM_MODULE_KSPROP;
static unsigned long _moduleid = ATO_AKM_MODULEID_KSPROP;

/*********************************************************************************/
struct _ato_ksProperties {
    bool isLegacy;
    ato_Xml *xml;
    const char *ver;
    char *salt;
    ato_Pbe *pbe;
    ato_PwdKdf *pk;

    int expiry_min;
    bool RenewalIncludeSoftwareInfo;
    char *uri;

    ato_List *nv;
};

/*********************************************************************************/
static ato_eLoglevel _loglevel = ATO_LOG_WARN;

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
int ato__ksprop_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}

void ato__ksprop_deinit(void)
{
}

/*********************************************************************************/
static void _clearnv(ato_ksProperties *ksprop) {
    if (ksprop->nv != NULL) {
        ato_ListNode *lnode = NULL;
        while ((lnode = ato_lst_next(ksprop->nv, lnode)) != NULL) {
            ato_free(ato_lst_value(lnode));
        }
    ato_lst_clear(ksprop->nv);
    }
}
static void _dupnv(ato_ksProperties *ksprop, const ato_ksProperties *properties) {
    if (properties != NULL && properties->nv != NULL) {
        ato_ListNode *lnode = NULL;
        while ((lnode = ato_lst_next(properties->nv, lnode)) != NULL) {
            ato_lst_add(ksprop->nv, ato_lst_key(lnode), ato_strdup(ato_lst_value(lnode), 0));
        }
    }
}
/*********************************************************************************/
ato_ksProperties *ato_ksprop_create(void) {
    ato_ksProperties *properties = calloc(1, sizeof(ato_ksProperties));
    assert(properties != NULL);
    ato_lst_create(&(properties->nv));
    return properties;
}

int ato__ksprop_createX(ato_Ctx *ctx, ato_ksProperties **obj, const ato_ksProperties *properties, ato_Xml *xml, bool isLegacy, const char *salt, const char *pbemode) {
    ato_ksProperties *ksprop = NULL;
    ATO_ASSERT_ISNOTALLOCATED(obj);
    ATO_IGNORE(ctx);

    if ((ksprop = *obj = ato_ksprop_create()) == NULL) { return ATO_ERR_MEMORY; }

    if (properties != NULL) {
        ksprop->expiry_min = properties->expiry_min;
        ksprop->uri = ato_strdup(properties->uri, 0);
        if (properties->pbe != NULL) {
            pbemode = ato_pbe_str(properties->pbe);
        }
    }

    ksprop->isLegacy = isLegacy;
    ksprop->xml = xml;
    ksprop->ver = isLegacy ? "01.00" : "02.00";
    assert(!ato_isnullorempty(salt));
    ksprop->salt = ato_strdup(salt, 0);
    ato_pbe_create(ctx, &(ksprop->pbe), pbemode);
    _dupnv(ksprop, properties);
    return ATO_ERR_OK;
}

void *ato_ksprop_free(ato_ksProperties *ksprop) {
    if (ksprop == NULL) return NULL;
    ksprop->salt = ato_free(ksprop->salt);
    ksprop->pbe = ato_pbe_free(ksprop->pbe);
    ksprop->xml = ato_xml_free(ksprop->xml);
    ksprop->pk = ato_pk_free(ksprop->pk);
    ksprop->uri = ato_free(ksprop->uri);
    _clearnv(ksprop);
    ksprop->nv = ato_lst_free(ksprop->nv);
    ksprop = ato_free(ksprop);
    return NULL;
}
/*********************************************************************************/

bool ato__ksprop_isLegacy(const ato_ksProperties *ksprop) { assert(ksprop != NULL); return ksprop->isLegacy; }
bool ato__ksprop_isPbe2(const ato_ksProperties *ksprop) { assert(ksprop != NULL); return ato_pbe_isPbe2(ksprop->pbe); }
ato_Xml *ato__ksprop_xml(const ato_ksProperties *ksprop) { assert(ksprop != NULL); return ksprop->xml; }
ato_Pbe *ato__ksprop_pbe(const ato_ksProperties *ksprop) { assert(ksprop != NULL); return ksprop->pbe; }

ato_PwdKdf *ato__ksprop_pk(ato_Ctx *ctx, ato_ksProperties *ksprop, const char *pwd) {
    assert(ksprop != NULL);
    if (pwd == NULL) {
        ksprop->pk = ato_pk_free(ksprop->pk);
    } else if (ksprop->pk == NULL) {
        ato_pk_create(ctx, &(ksprop->pk), pwd, ksprop->salt, ksprop->pbe);
    } else {
        ato_pk_rpwd_set(ksprop->pk, pwd);
        ato_pk_pbe_set(ctx, ksprop->pk, ksprop->pbe);
    }
    return ksprop->pk;
}
void ato__ksprop_pk_clr(ato_Ctx *ctx, ato_ksProperties *ksprop) {
    assert(ksprop != NULL);
    if (ksprop->pk != NULL) {
        ato_pk_rpwd_set(ksprop->pk, NULL);
    }
    ATO_IGNORE(ctx);
}

/*********************************************************************************/
const char *ato_ksprop_ver(const ato_ksProperties *ksprop) { assert(ksprop != NULL); return ksprop->ver; }
const char *ato_ksprop_salt(const ato_ksProperties *ksprop) { assert(ksprop != NULL); return ksprop->salt; }

/*********************************************************************************/
const char *ato_ksprop_PbeMode(const ato_ksProperties *ksprop) { assert(ksprop != NULL); return ksprop->pbe ? ato_pbe_str(ksprop->pbe) : NULL; }
void ato_ksprop_PbeMode_Set(ato_Ctx *ctx, ato_ksProperties *ksprop, const char *pbemode) {
    assert(ksprop != NULL);
    if (pbemode == NULL) { return; }
    ksprop->pbe = ato_pbe_free(ksprop->pbe);
    ato_pbe_create(ctx, &(ksprop->pbe), pbemode);
}

const char *ato_ksprop_var(ato_ksProperties *ksprop, const char *name) {
    ato_ListNode *lnode = ato_lst_find(ksprop->nv, name);
    return lnode == NULL ? NULL : (const char *)ato_lst_value(lnode);
}

void ato_ksprop_var_set(ato_ksProperties *ksprop, const char *name, const char *value) {
    ato_lst_add(ksprop->nv, name, ato_strdup(value, 0));
}

int ato_ksprop_var_int(ato_ksProperties *ksprop, const char *name, int defvalue) {
    const char *value = ato_ksprop_var(ksprop, name);
    return value ? (int) strtol(value, NULL, 10) : defvalue;
}


const char *ato_ksprop_varn_checkIntegrityValue(void) { return "checkIntegrityValue"; }
