/** @file keystore.c
* @brief Contains methods to perform keystore operations and retrieve credentials.
*
* This is used the primary module to access all functionality with the atokeystore library.
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#ifdef __APPLE__
#include "TargetConditionals.h"
#if TARGET_OS_IPHONE && TARGET_IPHONE_SIMULATOR
#define TARGET_OS_IOS 1
#elif TARGET_OS_IPHONE
#define TARGET_OS_IOS 1
#else
#define TARGET_OS_OSX 1
#endif
#endif

#if TARGET_OS_IOS
#include "akversion_ios.h"
#else
#include "akversion.h"
#endif

#include "atobase/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"
#include "atoksprop.h"
#include "atocredential.h"
#include "atokeystore.h"
#include "atorenew.h"
#include "atoerrfn.h"
#include "atolib.h"

#if defined(WINAPI_FAMILY)
static char *getenv(const char *name) { ATO_IGNORE(name); return NULL; }
#endif

static const char *_library = ATO_AKM_LIBRARY;
static const char *_module = ATO_AKM_MODULE_LIB;
static unsigned long _moduleid = ATO_AKM_MODULEID_LIB;

static int _loglevel = ATO_LOG_WARN;

static char _deflocation[FILENAME_MAX +1];
static const char _defksfile[] = "keystore.xml";
static char *_defkspath = NULL;

/*********************************************************************************/
static int _init(const char *default_ks_dir);
static void _load_errdefs(void);

/*********************************************************************************/
const char *ato_akm_version(void) { return ATOAKM_VERSION; }

static void _setloglevel(ato_eLoglevel level)
{
        _loglevel = level;
}

int ato__akmlib_init(void)
{
        static bool invoked = FALSE;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        return ATO_ERR_OK;
}
void ato__akmlib_deinit(void)
{
        static bool invoked = FALSE;
        if (invoked) return;
        invoked = TRUE;
}

int ato__akmlib_process_errcode(ato_Ctx *ctx, int errcode, const char *module, const char *function, int line)
{
        int liberrcode = ATO_AKM_ERR_GENERAL;
        if (errcode == ATO_ERR_OK)
                return errcode;
        if (errcode >= ATO_AKM_ERR_MIN && errcode <= ATO_AKM_ERR_MAX)
                return errcode;

        if (errcode == ATO_ERR_NET_TIMEOUT)
                liberrcode = ATO_AKM_ERR_NETTIMEOUT;
        else if (errcode == ATO_ERR_NET_SOAP)
                liberrcode = ATO_AKM_ERR_NETRECEIVER;
        else if (errcode == ATO_ERR_NET)
                liberrcode = ATO_AKM_ERR_NETCOMMS;
        return ato_ctx_err_new(ctx, _library, module, function, line, liberrcode, ATO_ESEVERITY_WARN, NULL, "");
}

/*********************************************************************************/
int ato_akm_init_ex(ato_Ctx *ctx, unsigned short flag, const char *default_ks_dir) {
        const char *function = "ato_akm_init";
        static bool invoked = FALSE;
        int errcode = ATO_ERR_OK;
        if (invoked) return ATO_ERR_OK;
        invoked = TRUE;

        assert(flag != 0); ATO_IGNORE(flag);
        assert(ctx != NULL);

        assert(ato_base_isinited() == TRUE);

        _load_errdefs();

        errcode = _init(default_ks_dir);
        if (errcode == ATO_ERR_OK)
                _defkspath = ato_pathcat(_deflocation, _defksfile);

        if (errcode == ATO_ERR_OK)
                errcode = ato__akmlib_init();

        if (errcode == ATO_ERR_OK)
                errcode = ato__ks_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__cr_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__crr_init();
        if (errcode == ATO_ERR_OK)
                errcode = ato__ksprop_init();

        //ato_errh_sethandler(ato__akm_errfn, ATO_AKM_LIBRARY);

        if (errcode != ATO_ERR_OK)
                ATO_CTX_NEWERR(ctx, errcode, "Failed to initialise AKM library");
        return errcode;
}
int ato_akm_init(ato_Ctx *ctx, unsigned short flag)
{
    return ato_akm_init_ex(ctx, flag, NULL);
}

void ato_akm_deinit(void)
{
        static bool invoked = FALSE;
        if (invoked) return;
        invoked = TRUE;

        ato__ksprop_deinit();
        ato__crr_deinit();
        ato__cr_deinit();
        ato__ks_deinit();
        ato__akmlib_deinit();

        if (_defkspath != NULL) free(_defkspath);
}

/*********************************************************************************/
#define _ATO_ADD_ERRDEF(code, name, msg) ato_errdef_add(code, _library, name, msg)
static void _load_errdefs(void)
{
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_GENERAL, "ATO_AKM_ERR_GENERAL", "For errors not otherwise specified");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_NETSENDER, "ATO_AKM_ERR_NETSENDER", "A SOAP fault generated by the sender, not including service unavailable");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_NETRECEIVER, "ATO_AKM_ERR_NETRECEIVER", "A SOAP fault generated by the receiver, not including service unavailable");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_NETUNAVAILABLE, "ATO_AKM_ERR_NETUNAVAILABLE", "A SOAP fault where the remote service is unavailable");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_NETCOMMS, "ATO_AKM_ERR_NETCOMMS", "A general network error has occurred, not including timeouts");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_NETTIMEOUT, "ATO_AKM_ERR_CRYPTO", "A network time has occured trying to connect to a remote service");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_CRYPTO, "ATO_AKM_ERR_CRYPTO", "General cryptographic error");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_NOTRENEWABLE, "ATO_AKM_ERR_NOTRENEWABLE", "Credential is not renewable due to being created too recently");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_RENEW, "ATO_AKM_ERR_RENEW", "Error renewing credential");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_CREDNOTFOUND, "ATO_AKM_ERR_CREDNOTFOUND", "Credential not found in keystore");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_ALIASEXISTS, "ATO_AKM_ERR_ALIASEXISTS", "A credential with the given alias exists and is not replaceable");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_KEYSTORE_CORRUPTED, "ATO_AKM_ERR_KEYSTORE_CORRUPTED", "Keystore is corrupted");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_BADPWD, "ATO_AKM_ERR_BADPWD", "Bad password");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_PWDDEST, "ATO_AKM_ERR_PWDDEST", "The password for the target keystore is wrong");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_SAVE, "ATO_AKM_ERR_SAVE", "Error saving keystore");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_INTEGRITYVALUE, "ATO_AKM_ERR_INTEGRITYVALUE", "Error validating integrity value");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_RENEWNOTCALLED, "ATO_AKM_ERR_RENEWNOTCALLED", "Private key of a credential accessed without first calling renewall method");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_PWDPOLICY, "ATO_AKM_ERR_PWDPOLICY", "The supplied password does not meet the strength requirements of the password policy");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_COPYCREDENTIAL, "ATO_AKM_ERR_COPYCREDENTIAL", "Error during copycredential operation");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_DUPLICATECR, "ATO_AKM_ERR_DUPLICATECR", "Possible duplicate credential");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_LOAD, "ATO_AKM_ERR_LOAD", "Error loading keystore");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_RENEW_NOTAVAILABLE, "ATO_AKM_ERR_RENEW_NOTAVAILABLE", "Renewal not available for the given Certficate Authority (e.g. in a test environment)");
        _ATO_ADD_ERRDEF(ATO_AKM_ERR_PWD, "ATO_AKM_ERR_PWD", "Password processing failed with possible corrupted P8 - see also ATO_AKM_ERR_BADPWD");
}
/*********************************************************************************/

const char *ato_akm_default_dir(void)
{
        return _deflocation;
}
const char *ato_akm_default_kspath(void)
{
        return _defkspath;
}
const char *ato_akm_default_ksfile(void)
{
        return _defksfile;
}
/*********************************************************************************/
static char *_getpath(ato_Ctx *ctx, const char *dirname, const char *filename)
{
        if (ato_isnullorempty(dirname))
                dirname = ato_akm_default_dir();
        if (ato_isnullorempty(filename))
                filename = ato_akm_default_ksfile();

    ATO_IGNORE(ctx);
        return ato_pathcat(dirname, filename);
}

int ato_akm_loadks(ato_Ctx *ctx, char **buffer, const char *dirname, const char *filename)
{
        const char *function = "ato_akm_loadks";
        int errcode = ATO_ERR_OK;
        char *buf = NULL;
        ato_String *str = NULL;
        char *path = NULL;
        ATO_ASSERT_ISNOTALLOCATED(buffer);

        path = _getpath(ctx, dirname, filename);

        errcode = ato_fileload(&str, path);
        if (errcode != ATO_ERR_OK) {
                ATO_CTX_VNEWERR(ctx, errcode, strlen(path), "Failed to load keystore file '%s'", path);
        } else {
                buf = ato_str_valueasowner(str);
        }

        if (path) free(path);
        ato_str_free(str);
        *buffer = buf;
        return errcode;
}

int ato_akm_saveks(ato_Ctx *ctx, const char *buffer, const char *dirname, const char *filename)
{
        const char *function = "ato_akm_saveks";
        int errcode = ATO_ERR_OK;
        ato_String *str = NULL;
        char *path = NULL;

        assert(buffer != NULL);

        path = _getpath(ctx, dirname, filename);

        ato_str_createconst(&str, buffer, strlen(buffer), TRUE);
        errcode = ato_filewrite(str, path);
        if (errcode != ATO_ERR_OK) {
                ATO_CTX_VNEWERR(ctx, errcode, strlen(path), "Failed to save keystore file '%s'", path);
        }

        if (path) free(path);
        ato_str_free(str);
        return errcode;
}

#define KSSUBDIR "ATOMAS"
#if defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64)
static int _init(const char *default_ks_dir)
{
        size_t len = 0;
    const char *datadir = default_ks_dir;
    if (datadir == NULL) { datadir = getenv("APPDATA"); }
    if (datadir == NULL) { datadir = getenv("USERPROFILE"); }
        assert(datadir != NULL);
        len = strlen(datadir);
        assert(len < FILENAME_MAX - 20);

        strcat(strcat(strcpy(_deflocation, datadir), "/"), KSSUBDIR);
    {
    char *s;
    while ((s = strchr(_deflocation, '\\')) != NULL) { *s = '/'; }
    }
    return 0;
}
#elif defined(TARGET_OS_MAC) || defined(__APPLE__)
static int _init(const char *default_ks_dir)
{
    const char *datadir = default_ks_dir;
    if (datadir == NULL) { datadir = getenv("HOME"); }
    if (datadir == NULL) { datadir = getenv("~"); }
    if (datadir == NULL) { datadir = "~"; }
    strcat(strcat(strcpy(_deflocation, datadir), "/Library/Application Support/"), KSSUBDIR);
    return 0;
}
#else
static int _init(const char *default_ks_dir)
{
    const char *datadir = default_ks_dir;
    if (datadir == NULL) { datadir = getenv("HOME"); }
    if (datadir == NULL) { datadir = getenv("~"); }
    if (datadir == NULL) { datadir = "~"; }

    strcat(strcat(strcpy(_deflocation, datadir), "/."), KSSUBDIR);
    return 0;
}
#endif
