#ifndef __ATO_ERRSOAP_H__
#define __ATO_ERRSOAP_H__

/** @addtogroup atobase_api_public_errsoap

@{
*/

/**
This object represents a SOAP fault.

This object will be created and attached to the top level generic
@ref ato_Err object returned by the method in which the error occurred.
Use ato_err_customobj() to retrieve this object.
*/
typedef struct _ato_ErrSoap ato_ErrSoap;

/**
This object represents a code or subcode node in a SOAP fault.
*/
typedef struct _ato_ErrSoapCode ato_ErrSoapCode;

#ifdef __cplusplus
extern "C" {
#endif

/**
Confirms whether or not the error object is of the expected type.

The other methods will assert if this is false.

@param err the soap error object.

@return TRUE if err is of type ato_ErrSoap, otherwise false.
*/
ATO_EXPORT bool ato_errsoap_issoaperr(ato_ErrSoap *err);

/**
Decribes the service node where the error occured.

@param err the soap error object.

@return service node
*/
ATO_EXPORT const char *ato_errsoap_node(ato_ErrSoap *err);

/**
Describes the specifics of the particular error condition, and reflects the finest granularity
of subcode provided in the fault.

@param err the soap error object.

@return the reason details.
*/
ATO_EXPORT const char *ato_errsoap_reason(ato_ErrSoap *err);

/**
This contains low-level details of the fault and is only provided in non-production environments.

@param err the soap error object.

@return fault details.
*/
ATO_EXPORT const char *ato_errsoap_detail(ato_ErrSoap *err);

/**
Returns the number of subcodes including the toplevel code.

@param err the soap error object.

@return the number or codes/subcodes.
*/
ATO_EXPORT size_t ato_errsoap_code_count(ato_ErrSoap *err);

/**
Create an iterator for the collection of errors - caller must free.

This is useful if a collection is being shared between threads and each thread has its own iterator.
It is also useful in the unlikely case that higher level code and nested code are bother iterating the
underlying collection.

If this is not the case then the ato_errsoap_code_firstv() and ato_errsoap_code_nextv() methods are simpler and this method can be ignored.

@param err the soap error object.
@param iter the address of the iterator to create - *iter must be NULL.
This will be NULL if there is no collection.
*/
ATO_EXPORT void ato_errsoap_code_iterator(ato_ErrSoap *err, ato_Iterator **iter);

/**
Get the first code of the collection.

This also initialises the iterator associated with the current sbdm timestamps.

@param err the soap error object.

@return a sub/code or NULL if none.
*/
ATO_EXPORT ato_ErrSoapCode *ato_errsoap_code_firstv(ato_ErrSoap *err);

/**
Get the next code of the collection.

@param err the soap error object.

@return a sub/code or NULL if none.
*/
ATO_EXPORT ato_ErrSoapCode *ato_errsoap_code_nextv(ato_ErrSoap *err);

/**
Returns the value of the code/subcode.

@param code the soap error code.

@return the value of the code/subcode or NULL if none.
*/
ATO_EXPORT const char *ato_errsoap_code_value(ato_ErrSoapCode *code);

/**
Returns the namespace of the code/subcode.

@param code the soap error code.

@return the namespace of the code/subcode or NULL if none.
*/
ATO_EXPORT const char *ato_errsoap_code_ns(ato_ErrSoapCode *code);

#ifdef __cplusplus
}
#endif

/*! @} */

#endif /*__ATO_ERRSOAP_H__*/
